package eu.xenit.json.log4j2;

import eu.xenit.RuntimeContainer;
import eu.xenit.json.DynamicMdcMessageField;
import eu.xenit.json.LogMessageField;
import eu.xenit.json.MdcJsonMessageAssembler;
import eu.xenit.json.MdcMessageField;
import eu.xenit.json.StaticMessageField;
import eu.xenit.json.intern.Closer;
import eu.xenit.json.intern.ConfigurationSupport;
import eu.xenit.json.intern.ErrorReporter;
import eu.xenit.json.intern.JsonMessage;
import eu.xenit.json.intern.JsonSender;
import eu.xenit.json.intern.JsonSenderFactory;
import eu.xenit.json.intern.MessagePostprocessingErrorReporter;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.core.Filter;
import org.apache.logging.log4j.core.LogEvent;
import org.apache.logging.log4j.core.appender.AbstractAppender;
import org.apache.logging.log4j.core.appender.AppenderLoggingException;
import org.apache.logging.log4j.core.config.Configuration;
import org.apache.logging.log4j.core.config.Property;
import org.apache.logging.log4j.core.config.plugins.Plugin;
import org.apache.logging.log4j.core.config.plugins.PluginAttribute;
import org.apache.logging.log4j.core.config.plugins.PluginConfiguration;
import org.apache.logging.log4j.core.config.plugins.PluginElement;
import org.apache.logging.log4j.core.config.plugins.PluginFactory;
import org.apache.logging.log4j.core.impl.Log4jLogEvent;
import org.apache.logging.log4j.core.layout.PatternLayout;
import org.apache.logging.log4j.status.StatusLogger;
import org.apache.logging.log4j.util.Strings;

import java.util.Collections;
import java.util.concurrent.TimeUnit;

import static org.apache.logging.log4j.core.layout.PatternLayout.newBuilder;

/**
 * Logging-Handler for Json . This Java-Util-Logging Handler creates Json Messages and posts
 * them using UDP (default) or TCP. Following parameters are supported/needed:
 * <ul>
 * <li>host (Mandatory): Hostname/IP-Address of the Logstash Host
 * <ul>
 * <li>(the host) for UDP, e.g. 127.0.0.1 or some.host.com</li>
 * <li>See docs for more details</li>
 * </ul>
 * </li>
 * <li>port (Optional): Port, default 12201</li>
 * <li>version (Optional): Json Version 1.0 or 1.1, default 1.0</li>
 * <li>originHost (Optional): Originating Hostname, default FQDN Hostname</li>
 * <li>extractStackTrace (Optional): Post Stack-Trace to StackTrace field (true/false/throwable reference [0 = throwable, 1 =
 * throwable.cause, -1 = root cause]), default false</li>
 * <li>filterStackTrace (Optional): Perform Stack-Trace filtering (true/false), default false</li>
 * <li>mdcProfiling (Optional): Perform Profiling (Call-Duration) based on MDC Data. See <a href="#mdcProfiling">MDC
 * Profiling</a>, default false</li>
 * <li>facility (Optional): Name of the Facility, default json-java</li>
 * <li>additionalFieldTypes (Optional): Type specification for additional and MDC fields. Supported types: String, long, Long,
 * double, Double and discover (default if not specified, discover field type on parseability). Eg. field=String,field2=double</li>
 * <li>ignoreExceptions (Optional): The default is <code>true</code>, causing exceptions encountered while appending events to
 * be internally logged and then ignored. When set to <code>false</code> exceptions will be propagated to the caller, instead.
 * You must set this to false when wrapping this Appender in a <code>FailoverAppender</code>.</li>
 * </ul>
 *
 * <h2>Fields</h2>
 *
 * <p>
 * Log4j v2 supports an extensive and flexible configuration in contrast to other log frameworks (JUL, log4j v1). This allows
 * you to specify your needed fields you want to use in the JSON message. An empty field configuration results in a message
 * containing only
 * </p>
 *
 * <ul>
 * <li>timestamp</li>
 * <li>level (syslog level)</li>
 * <li>host</li>
 * <li>facility</li>
 * <li>message</li>
 * <li>shortMessage</li>
 * </ul>
 *
 * <p>
 * You can add different fields:
 * </p>
 *
 * <ul>
 * <li>Static Literals</li>
 * <li>MDC Fields</li>
 * <li>Log-Event fields (using <a href="http://logging.apache.org/log4j/2.x/manual/layouts.html#PatternLayout">Pattern
 * Layout</a>)</li>
 * </ul>
 * <p>
 * In order to do so, use nested Field elements below the Appender element.
 *
 * <h3>Static Literals</h3> <code>
 * &lt;Field name="fieldName1" literal="your literal value" /&gt;
 * </code>
 *
 * <h3>MDC Fields</h3> <code>
 * &lt;Field name="fieldName1" mdc="name of the MDC entry" /&gt;
 * </code>
 *
 * <h3>Dynamic MDC Fields</h3> <code>
 * &lt;DynamicMdcFields regex="mdc.*"  /&gt;
 * </code>
 *
 *
 * <h3>Log-Event fields</h3>
 * <p>
 * See also: <a href="http://logging.apache.org/log4j/2.x/manual/layouts.html#PatternLayout">Pattern Layout</a>
 * </p>
 *
 * <p>
 * You can use all built-in Pattern Fields:
 * </p>
 * <code>
 * &lt;Field name="simpleClassName" pattern="%C{1}" /&gt;
 * &lt;Field name="timestamp" pattern="%d{dd MMM yyyy HH:mm:ss,SSS}" /&gt;
 * &lt;Field name="level" pattern="%level" /&gt;
 * </code>
 *
 * <p>
 * Additionally, you can add the <strong>host</strong>-Field, which can supply you either the FQDN hostname, the simple hostname
 * or the local address.
 * </p>
 * <table class="overviewSummary" border="0" cellpadding="3" cellspacing="0" style="border-bottom:1px solid #9eadc0;" summary="Details for the %host formatter">
 * <tr>
 * <th class="colFirst">Option</th>
 * <th class="colLast">Description</th>
 * </tr>
 * <tr class="altColor">
 * <td class="colFirst" align="center"><b>host</b> &nbsp;&nbsp;{["fqdn"<br>
 * &nbsp;&nbsp;|"simple"<br>
 * &nbsp;&nbsp;|"address"]}</td>
 * <td class="colLast">
 * <p>
 * Outputs either the FQDN hostname, the simple hostname or the local address.
 * </p>
 * <p>
 * You can follow the throwable conversion word with an option in the form <b>%host{option}</b>.
 * </p>
 * <p>
 * <b>%host{fqdn}</b> default setting, outputs the FQDN hostname, e.g. www.you.host.name.com.
 * </p>
 * <p>
 * <b>%host{simple}</b> outputs simple hostname, e.g. www.
 * </p>
 * <p>
 * <b>%host{address}</b> outputs the local IP address of the found hostname, e.g. 1.2.3.4 or affe:affe:affe::1.
 * </p>
 * </td>
 * </tr>
 * </table>
 *
 * <h2>MDC Field Typing</h2> In some cases, it's required to use a fixed type for fields transported using JSON. MDC is a
 * dynamic value source and since types can vary, so also data types in the JSON JSON vary. You can define
 * {@code DynamicMdcFieldType} rules to declare types with Regex {@link java.util.regex.Pattern}-based rules.
 * <p>
 * <code>
 * &gt;DynamicMdcFieldType regex="business\..*\.field" type="double" /&lt;
 * </code>
 *
 * <a name="mdcProfiling"></a>
 * <h2>MDC Profiling</h2>
 * <p>
 * MDC Profiling allows to calculate the runtime from request start up to the time until the log message was generated. You must
 * set one value in the MDC:
 * <ul>
 * <li>profiling.requestStart.millis: Time Millis of the Request-Start (Long or String)</li>
 * </ul>
 * <p>
 * Two values are set by the Log Appender:
 * </p>
 * <ul>
 * <li>profiling.requestEnd: End-Time of the Request-End in Date.toString-representation</li>
 * <li>profiling.requestDuration: Duration of the request (e.g. 205ms, 16sec)</li>
 * </ul>
 * <p>
 * The {@link #append(LogEvent)} method is thread-safe and may be called by different threads at any time.
 */
@Plugin(name = "Json", category = "Core", elementType = "appender", printObject = true)
public class JsonLogAppender extends AbstractAppender {

    private static final Logger LOGGER = StatusLogger.getLogger();
    private static final ErrorReporter ERROR_REPORTER = (message, e) -> LOGGER.error(message, e, 0);

    private static final ErrorReporter PROPAGATING_ERROR_REPORTER = (message, e) -> {

        if (e != null) {
            throw new AppenderLoggingException(e);
        }

        LOGGER.error(message, null, 0);
    };
    private final MdcJsonMessageAssembler jsonMessageAssembler;
    private final ErrorReporter errorReporter;
    protected JsonSender jsonSender;

    public JsonLogAppender(String name, Filter filter, MdcJsonMessageAssembler jsonMessageAssembler, boolean ignoreExceptions) {

        super(name, filter, null, ignoreExceptions, Property.EMPTY_ARRAY);
        this.jsonMessageAssembler = jsonMessageAssembler;

        ErrorReporter rErrorReporter = getErrorReporter(ignoreExceptions);

        this.errorReporter = new MessagePostprocessingErrorReporter(rErrorReporter);
    }

    @PluginFactory
    public static JsonLogAppender createAppender(@PluginConfiguration final Configuration config,
                                                 @PluginAttribute("name") String name, @PluginElement("Filter") Filter filter,
                                                 @PluginElement("Field") final JsonLogField[] fields,
                                                 @PluginElement("DynamicMdcFields") final JsonDynamicMdcLogFields[] dynamicFieldArray,
                                                 @PluginElement("DynamicMdcFieldTypes") final JsonDynamicMdcFieldType[] dynamicFieldTypeArray,
                                                 @PluginAttribute("host") String host, @PluginAttribute("port") String port,
                                                 @PluginAttribute("version") String version, @PluginAttribute("extractStackTrace") String extractStackTrace,
                                                 @PluginAttribute("originHost") String originHost, @PluginAttribute("includeFullMdc") String includeFullMdc,
                                                 @PluginAttribute("facility") String facility, @PluginAttribute("filterStackTrace") String filterStackTrace,
                                                 @PluginAttribute("mdcProfiling") String mdcProfiling,
                                                 @PluginAttribute("maximumMessageSize") String maximumMessageSize,
                                                 @PluginAttribute("additionalFieldTypes") String additionalFieldTypes,
                                                 @PluginAttribute(value = "ignoreExceptions", defaultBoolean = true) boolean ignoreExceptions) {

        RuntimeContainer.initialize(ERROR_REPORTER);

        MdcJsonMessageAssembler mdcJsonMessageAssembler = new MdcJsonMessageAssembler();

        if (name == null) {
            LOGGER.error("No name provided for {}", JsonLogAppender.class.getSimpleName());
            return null;
        }

        if (Strings.isEmpty(host)) {
            LOGGER.error("No host provided for {}", JsonLogAppender.class.getSimpleName());
            return null;
        }
        mdcJsonMessageAssembler.setHost(host);

        if (Strings.isNotEmpty(port)) {
            mdcJsonMessageAssembler.setPort(Integer.parseInt(port));
        }

        if (Strings.isNotEmpty(version)) {
            mdcJsonMessageAssembler.setVersion(version);
        }

        if (Strings.isNotEmpty(originHost)) {
            PatternLayout patternLayout = newBuilder().withPattern(originHost).withConfiguration(config)
                    .withNoConsoleNoAnsi(false).withAlwaysWriteExceptions(false).build();

            mdcJsonMessageAssembler.setOriginHost(patternLayout.toSerializable(new Log4jLogEvent()));
        }

        if (facility != null) {
            mdcJsonMessageAssembler.setFacility(facility);
        }

        if (extractStackTrace != null) {
            mdcJsonMessageAssembler.setExtractStackTrace(extractStackTrace);
        }

        if (filterStackTrace != null) {
            mdcJsonMessageAssembler.setFilterStackTrace("true".equals(filterStackTrace));
        }

        if (mdcProfiling != null) {
            mdcJsonMessageAssembler.setMdcProfiling("true".equals(mdcProfiling));
        }

        if (includeFullMdc != null) {
            mdcJsonMessageAssembler.setIncludeFullMdc("true".equals(includeFullMdc));
        }

        if (maximumMessageSize != null) {
            mdcJsonMessageAssembler.setMaximumMessageSize(Integer.parseInt(maximumMessageSize));
        }

        if (additionalFieldTypes != null) {
            ConfigurationSupport.setAdditionalFieldTypes(additionalFieldTypes, mdcJsonMessageAssembler);
        }

        if (dynamicFieldTypeArray != null) {
            for (JsonDynamicMdcFieldType jsonDynamicMdcFieldType : dynamicFieldTypeArray) {
                mdcJsonMessageAssembler.setDynamicMdcFieldType(jsonDynamicMdcFieldType.getPattern(),
                        jsonDynamicMdcFieldType.getType());
            }
        }

        configureFields(mdcJsonMessageAssembler, fields, dynamicFieldArray);

        return new JsonLogAppender(name, filter, mdcJsonMessageAssembler, ignoreExceptions);
    }

    /**
     * Configure fields (literals, MDC, layout).
     *
     * @param mdcJsonMessageAssembler the assembler
     * @param fields                  static field array
     * @param dynamicFieldArray       dynamic field array
     */
    private static void configureFields(MdcJsonMessageAssembler mdcJsonMessageAssembler, JsonLogField[] fields,
                                        JsonDynamicMdcLogFields[] dynamicFieldArray) {

        if (fields == null || fields.length == 0) {
            mdcJsonMessageAssembler.addFields(LogMessageField.getDefaultMapping(LogMessageField.NamedLogField.Time, LogMessageField.NamedLogField.Severity, LogMessageField.NamedLogField.ThreadName, LogMessageField.NamedLogField.SourceClassName,
                    LogMessageField.NamedLogField.SourceMethodName, LogMessageField.NamedLogField.SourceLineNumber, LogMessageField.NamedLogField.SourceSimpleClassName, LogMessageField.NamedLogField.LoggerName, LogMessageField.NamedLogField.Marker));
            return;
        }

        for (JsonLogField field : fields) {

            if (Strings.isNotEmpty(field.getMdc())) {
                mdcJsonMessageAssembler.addField(new MdcMessageField(field.getName(), field.getMdc()));
            }

            if (Strings.isNotEmpty(field.getLiteral())) {
                mdcJsonMessageAssembler.addField(new StaticMessageField(field.getName(), field.getLiteral()));
            }

            if (field.getPatternLayout() != null) {
                mdcJsonMessageAssembler.addField(new PatternLogMessageField(field.getName(), null, field.getPatternLayout()));
            }
        }

        if (dynamicFieldArray != null) {
            for (JsonDynamicMdcLogFields jsonDynamicMdcLogFields : dynamicFieldArray) {
                mdcJsonMessageAssembler.addField(new DynamicMdcMessageField(jsonDynamicMdcLogFields.getRegex()));
            }
        }
    }

    private ErrorReporter getErrorReporter(boolean ignoreExceptions) {
        return ignoreExceptions ? ERROR_REPORTER : PROPAGATING_ERROR_REPORTER;
    }

    @Override
    public void append(LogEvent event) {

        if (event == null) {
            return;
        }

        try {
            JsonMessage message = createJsonMessage(event);
            if (!message.isValid()) {
                reportError("JSON Message is invalid: " + message.toJson(), null);
                return;
            }

            if (null == jsonSender || !jsonSender.sendMessage(message)) {
                reportError("Could not send JSON message", null);
            }
        } catch (AppenderLoggingException e) {
            throw e;
        } catch (Exception e) {
            reportError("Could not send JSON message: " + e.getMessage(), e);
        }
    }

    protected JsonMessage createJsonMessage(final LogEvent logEvent) {
        return jsonMessageAssembler.createJsonMessage(new Log4j2LogEvent(logEvent));
    }

    public void reportError(String message, Exception exception) {
        errorReporter.reportError(message, exception);
    }

    @Override
    protected boolean stop(long timeout, TimeUnit timeUnit, boolean changeLifeCycleState) {

        if (null != jsonSender) {
            Closer.close(jsonSender);
            jsonSender = null;
        }

        return super.stop(timeout, timeUnit, changeLifeCycleState);
    }

    @Override
    public void start() {

        if (null == jsonSender) {
            jsonSender = createJsonSender();
        }

        super.start();
    }

    protected JsonSender createJsonSender() {
        return JsonSenderFactory.createSender(jsonMessageAssembler, errorReporter, Collections.<String, Object>emptyMap());
    }
}
