package eu.xenit.json.logback;

import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.core.AppenderBase;
import eu.xenit.RuntimeContainer;
import eu.xenit.json.LogMessageField;
import eu.xenit.json.MdcJsonMessageAssembler;
import eu.xenit.json.intern.Closer;
import eu.xenit.json.intern.ConfigurationSupport;
import eu.xenit.json.intern.ErrorReporter;
import eu.xenit.json.intern.JsonMessage;
import eu.xenit.json.intern.JsonSender;
import eu.xenit.json.intern.JsonSenderFactory;
import eu.xenit.json.intern.MessagePostprocessingErrorReporter;

import java.util.Collections;

/**
 * Logging-Handler for Json. This Logback Handler creates Json Messages and posts them using
 * UDP (default) or TCP. Following parameters are supported/needed:
 * <ul>
 * <li>host (Mandatory): Hostname/IP-Address of the Logstash Host
 * <ul>
 * <li>(the host) for UDP, e.g. 127.0.0.1 or some.host.com</li>
 * <li>See docs for more details</li>
 * </ul>
 * </li>
 * <li>port (Optional): Port, default 12201</li>
 * <li>version (Optional): JSON Version 1.0 or 1.1, default 1.0</li>
 * <li>originHost (Optional): Originating Hostname, default FQDN Hostname</li>
 * <li>extractStackTrace (Optional): Post Stack-Trace to StackTrace field (true/false/throwable reference [0 = throwable, 1 =
 * throwable.cause, -1 = root cause]), default false</li>
 * <li>filterStackTrace (Optional): Perform Stack-Trace filtering (true/false), default false</li>
 * <li>includeLocation (Optional): Include source code location, default true</li>
 * <li>mdcProfiling (Optional): Perform Profiling (Call-Duration) based on MDC Data. See <a href="#mdcProfiling">MDC
 * Profiling</a>, default false</li>
 * <li>facility (Optional): Name of the Facility, default json-java</li>
 * <li>filter (Optional): logback filter (incl. log level)</li>
 * <li>additionalFields(number) (Optional): Post additional fields. Eg.
 * .JsonLogHandler.additionalFields=fieldName=Value,field2=value2</li>
 * <li>additionalFieldTypes (Optional): Type specification for additional and MDC fields. Supported types: String, long, Long,
 * double, Double and discover (default if not specified, discover field type on parseability). Eg. field=String,field2=double</li>
 * <li>mdcFields (Optional): Post additional fields, pull Values from MDC. Name of the Fields are comma-separated
 * mdcFields=Application,Version,SomeOtherFieldName</li>
 * <li>dynamicMdcFields (Optional): Dynamic MDC Fields allows you to extract MDC values based on one or more regular
 * expressions. Multiple regex are comma-separated. The name of the MDC entry is used as JSON field name.</li>
 * <li>dynamicMdcFieldTypes (Optional): Pattern-based type specification for additional and MDC fields. Key-value pairs are
 * comma-separated. Supported types: String, long, Long, double, Double. Eg. my_field.*=String,business\..*\.field=double</li>
 * <li>includeFullMdc (Optional): Include all fields from the MDC, default false</li>
 * </ul>
 * <a name="mdcProfiling"></a> <h2>MDC Profiling</h2>
 * <p>
 * MDC Profiling allows to calculate the runtime from request start up to the time until the log message was generated. You must
 * set one value in the MDC:
 * <ul>
 * <li>profiling.requestStart.millis: Time Millis of the Request-Start (Long or String)</li>
 * </ul>
 * <p>
 * Two values are set by the Log Appender:
 * </p>
 * <ul>
 * <li>profiling.requestEnd: End-Time of the Request-End in Date.toString-representation</li>
 * <li>profiling.requestDuration: Duration of the request (e.g. 205ms, 16sec)</li>
 * </ul>
 * <p>
 * The {@link #append(ILoggingEvent)} method is thread-safe and may be called by different threads at any time.
 */
public class JsonLogbackAppender extends AppenderBase<ILoggingEvent> implements ErrorReporter {

    private final ErrorReporter errorReporter = new MessagePostprocessingErrorReporter(this);
    protected JsonSender jsonSender;
    protected MdcJsonMessageAssembler jsonMessageAssembler;

    public JsonLogbackAppender() {
        super();
        jsonMessageAssembler = new MdcJsonMessageAssembler();
        jsonMessageAssembler.addFields(LogMessageField.getDefaultMapping(LogMessageField.NamedLogField.Time, LogMessageField.NamedLogField.Severity, LogMessageField.NamedLogField.ThreadName, LogMessageField.NamedLogField.SourceClassName,
                LogMessageField.NamedLogField.SourceMethodName, LogMessageField.NamedLogField.SourceLineNumber, LogMessageField.NamedLogField.SourceSimpleClassName, LogMessageField.NamedLogField.LoggerName, LogMessageField.NamedLogField.Marker));
    }

    @Override
    protected void append(ILoggingEvent event) {

        if (event == null) {
            return;
        }

        try {
            JsonMessage message = createJsonMessage(event);
            if (!message.isValid()) {
                reportError("JSON Message is invalid: " + message.toJson(), null);
                return;
            }

            if (null == jsonSender || !jsonSender.sendMessage(message)) {
                reportError("Could not send JSON message", null);
            }
        } catch (Exception e) {
            reportError("Could not send JSON message: " + e.getMessage(), e);
        }
    }

    @Override
    public void start() {

        if (null == jsonSender) {
            RuntimeContainer.initialize(errorReporter);
            jsonSender = createJsonSender();
        }

        super.start();
    }

    @Override
    public void stop() {

        if (null != jsonSender) {
            Closer.close(jsonSender);
            jsonSender = null;
        }

        super.stop();
    }

    protected JsonSender createJsonSender() {
        return JsonSenderFactory.createSender(jsonMessageAssembler, errorReporter, Collections.<String, Object>emptyMap());
    }

    @Override
    public void reportError(String message, Exception exception) {
        addError(message, exception);
    }

    protected JsonMessage createJsonMessage(final ILoggingEvent loggingEvent) {
        return jsonMessageAssembler.createJsonMessage(new LogbackLogEvent(loggingEvent));
    }

    public void setAdditionalFields(String spec) {
        ConfigurationSupport.setAdditionalFields(spec, jsonMessageAssembler);
    }

    public void setAdditionalFieldTypes(String spec) {
        ConfigurationSupport.setAdditionalFieldTypes(spec, jsonMessageAssembler);
    }

    public void setMdcFields(String spec) {
        ConfigurationSupport.setMdcFields(spec, jsonMessageAssembler);
    }

    public void setDynamicMdcFields(String spec) {
        ConfigurationSupport.setDynamicMdcFields(spec, jsonMessageAssembler);
    }

    public void setDynamicMdcFieldTypes(String spec) {
        ConfigurationSupport.setDynamicMdcFieldTypes(spec, jsonMessageAssembler);
    }

    public String getOriginHost() {
        return jsonMessageAssembler.getOriginHost();
    }

    public void setOriginHost(String originHost) {
        jsonMessageAssembler.setOriginHost(originHost);
    }

    public String getHost() {
        return jsonMessageAssembler.getHost();
    }

    public void setHost(String host) {
        jsonMessageAssembler.setHost(host);
    }

    public int getPort() {
        return jsonMessageAssembler.getPort();
    }

    public void setPort(int port) {
        jsonMessageAssembler.setPort(port);
    }

    public String getFacility() {
        return jsonMessageAssembler.getFacility();
    }

    public void setFacility(String facility) {
        jsonMessageAssembler.setFacility(facility);
    }

    public String getExtractStackTrace() {
        return jsonMessageAssembler.getExtractStackTrace();
    }

    public void setExtractStackTrace(String extractStacktrace) {
        jsonMessageAssembler.setExtractStackTrace(extractStacktrace);
    }

    public boolean isFilterStackTrace() {
        return jsonMessageAssembler.isFilterStackTrace();
    }

    public void setFilterStackTrace(boolean filterStackTrace) {
        jsonMessageAssembler.setFilterStackTrace(filterStackTrace);
    }

    public boolean isIncludeLocation() {
        return jsonMessageAssembler.isIncludeLocation();
    }

    public void setIncludeLocation(boolean includeLocation) {
        jsonMessageAssembler.setIncludeLocation(includeLocation);
    }

    public boolean isMdcProfiling() {
        return jsonMessageAssembler.isMdcProfiling();
    }

    public void setMdcProfiling(boolean mdcProfiling) {
        jsonMessageAssembler.setMdcProfiling(mdcProfiling);
    }

    public String getTimestampPattern() {
        return jsonMessageAssembler.getTimestampPattern();
    }

    public void setTimestampPattern(String timestampPattern) {
        jsonMessageAssembler.setTimestampPattern(timestampPattern);
    }

    public int getMaximumMessageSize() {
        return jsonMessageAssembler.getMaximumMessageSize();
    }

    public void setMaximumMessageSize(int maximumMessageSize) {
        jsonMessageAssembler.setMaximumMessageSize(maximumMessageSize);
    }

    public boolean isIncludeFullMdc() {
        return jsonMessageAssembler.isIncludeFullMdc();
    }

    public void setIncludeFullMdc(boolean includeFullMdc) {
        jsonMessageAssembler.setIncludeFullMdc(includeFullMdc);
    }

    public String getVersion() {
        return jsonMessageAssembler.getVersion();
    }

    public void setVersion(String version) {
        jsonMessageAssembler.setVersion(version);
    }

}
