package eu.xenit.json.p6spy;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.p6spy.engine.logging.Category;
import com.p6spy.engine.spy.appender.P6Logger;

import java.io.IOException;
import java.io.PrintStream;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.HashMap;
import java.util.Map;

import static eu.xenit.json.XenitJsonUtil.formatLocalDateTime;
import static eu.xenit.json.XenitJsonUtil.getTimestamp;

public class JsonP6spyAppender implements P6Logger {
    private static final String TYPE = "sql";
    private static final String COMPONENT = "p6spy";

    private PrintStream printStream;

    public JsonP6spyAppender() {
        this.printStream = System.out;
    }


    protected PrintStream getPrintStream() {
        return printStream;
    }

    public void setPrintStream(PrintStream printStream) {
        this.printStream = printStream;
    }


    @Override
    public void logException(Exception e) {
        Map<String, Object> root = createRootLogging(LocalDateTime.now());
        root.put("severity", "ERROR");
        root.put("exceptionMessage", e.getMessage());
        root.put("thread", Thread.currentThread().getName());
        root.put("exceptionStackTrace", e.getStackTrace());
        try {
            getPrintStream().println(new ObjectMapper().writeValueAsString(root));
        } catch (IOException exception) {
            exception.printStackTrace(getPrintStream());
        }
    }

    @Override
    public void logSQL(int connectionId, String now, long elapsed, Category category, String prepared, String sql, String url) {
        Map<String, Object> root = createRootLogging(
                Instant.ofEpochMilli(
                                Long.parseLong(now))
                        .atOffset(ZoneOffset.UTC)
                        .toLocalDateTime());
        root.put("sql", sql);
        root.put("requestTime", getTimestamp(elapsed));
        root.put("prepared", prepared);
        root.put("severity", category);
        root.put("connectionId", connectionId);
        root.put("jdbcUrl", url);
        root.put("thread", Thread.currentThread().getName());
        try {
            getPrintStream().println(new ObjectMapper().writeValueAsString(root));
        } catch (IOException e) {
            logException(e);
        }
    }

    @Override
    public void logText(String text) {
        if (!text.trim().isEmpty()) {
            Map<String, Object> root = createRootLogging(LocalDateTime.now());
            root.put("severity", "INFO");
            root.put("text", text);
            root.put("thread", Thread.currentThread().getName());
            try {
                getPrintStream().println(new ObjectMapper().writeValueAsString(root));
            } catch (IOException exception) {
                logException(exception);
            }
        }
    }

    @Override
    public boolean isCategoryEnabled(Category category) {
        // no restrictions on logger side
        return true;
    }

    private Map<String, Object> createRootLogging(LocalDateTime now) {
        Map<String, Object> root = new HashMap<>();
        root.put("time", formatLocalDateTime(now));
        root.put("timestamp", getTimestamp(now.toInstant(ZoneOffset.UTC).toEpochMilli()));
        root.put("type", TYPE);
        root.put("component", COMPONENT);
        return root;
    }

}