/*
 * Decompiled with CFR 0.152.
 */
package fi.evolver.ai.vaadin.admin;

import fi.evolver.ai.vaadin.entity.Chat;
import fi.evolver.ai.vaadin.entity.ChatMessage;
import java.io.IOException;
import java.io.OutputStream;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.TreeMap;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.CreationHelper;
import org.apache.poi.ss.usermodel.Font;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.VerticalAlignment;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.xssf.usermodel.XSSFFont;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;

public class ChatReportGenerator {
    private static final List<HeaderConfig> EXPORT_HEADERS = List.of(new HeaderConfig("Chat date time"), new HeaderConfig("First user message", 30), new HeaderConfig("First assistant response", 30), new HeaderConfig("Summary", 30), new HeaderConfig("View", 20), new HeaderConfig("LLM model", 14), new HeaderConfig("Chat messages", 30), new HeaderConfig("User message count"), new HeaderConfig("Assistant message count"), new HeaderConfig("Total message count"), new HeaderConfig("Token count"), new HeaderConfig("Chat rating"));
    private static final List<HeaderConfig> ANALYTICS_HEADERS = List.of(new HeaderConfig("Start date", 20), new HeaderConfig("End date", 20), new HeaderConfig("View", 20), new HeaderConfig("LLM model", 14), new HeaderConfig("Chat count"), new HeaderConfig("Distinct users"), new HeaderConfig("Average user message count"), new HeaderConfig("Average assistant message count"), new HeaderConfig("Average total message count"), new HeaderConfig("Total token count"), new HeaderConfig("Average chat rating"));

    private ChatReportGenerator() {
    }

    public static void generateChatReport(List<Chat> chats, LocalDate startDate, LocalDate endDate, OutputStream out) throws IOException {
        try (XSSFWorkbook workbook = new XSSFWorkbook();){
            CellFormat cellFormat = new CellFormat((Workbook)workbook);
            ChatReportGenerator.generateChatExportSheet(chats, (Workbook)workbook, cellFormat);
            ChatReportGenerator.generateChatAnalyticsSheet(chats, startDate, endDate, (Workbook)workbook, cellFormat);
            workbook.write(out);
        }
    }

    private static void generateChatAnalyticsSheet(List<Chat> chats, LocalDate startDate, LocalDate endDate, Workbook workbook, CellFormat cellFormat) {
        Sheet sheet = workbook.createSheet("Chat analytics");
        ChatReportGenerator.generateHeaderRow(ANALYTICS_HEADERS, sheet, cellFormat.headerStyle());
        TreeMap<String, Map<String, List<Chat>>> chatsByTypeAndModel = new TreeMap<String, Map<String, List<Chat>>>(chats.stream().collect(Collectors.groupingBy(Chat::getChatType, Collectors.groupingBy(ChatReportGenerator::getLlmModel))));
        int rowIndex = 0;
        for (String type : chatsByTypeAndModel.keySet()) {
            for (String model : chatsByTypeAndModel.get(type).keySet()) {
                List<Chat> chatEntries = chatsByTypeAndModel.get(type).get(model);
                int averageUserMessageCount = ChatReportGenerator.calculateAverageMessageCount(chatEntries, ChatMessage.ChatMessageRole.USER);
                int averageAssistantMessageCount = ChatReportGenerator.calculateAverageMessageCount(chatEntries, ChatMessage.ChatMessageRole.ASSISTANT);
                ChatReportGenerator.createContentCells(sheet.createRow(rowIndex + 1), List.of(startDate, endDate, type, model, chatEntries.size(), chatEntries.stream().map(Chat::getUsername).collect(Collectors.toSet()).size(), averageUserMessageCount, averageAssistantMessageCount, averageUserMessageCount + averageAssistantMessageCount, ChatReportGenerator.calculateTotalTokenCount(chatEntries), ChatReportGenerator.calculateAverageChatRating(chatEntries)), cellFormat);
                ++rowIndex;
            }
        }
    }

    private static int calculateTotalTokenCount(List<Chat> chats) {
        return chats.stream().map(chat -> chat.getChatMessages().stream().filter(cm -> cm.getRole() == ChatMessage.ChatMessageRole.USER || cm.getRole() == ChatMessage.ChatMessageRole.ASSISTANT).toList()).flatMap(Collection::stream).mapToInt(cm -> cm.getTokenCount() != null ? cm.getTokenCount() : 0).sum();
    }

    private static int calculateAverageMessageCount(List<Chat> chats, ChatMessage.ChatMessageRole role) {
        return (int)Math.round(chats.stream().map(chat -> chat.getChatMessages().stream().filter(cm -> cm.getRole() == role).count()).mapToDouble(Long::doubleValue).average().orElse(0.0));
    }

    private static String getLlmModel(Chat chat) {
        return chat.getChatMessages().stream().filter(cm -> cm.getRole() == ChatMessage.ChatMessageRole.USER).findFirst().map(ChatMessage::getModel).orElse("");
    }

    private static double calculateAverageChatRating(List<Chat> chats) {
        return chats.stream().map(Chat::getChatRating).filter(Objects::nonNull).mapToDouble(Integer::doubleValue).average().orElse(0.0);
    }

    private static void generateChatExportSheet(List<Chat> chats, Workbook workbook, CellFormat cellFormat) {
        Sheet sheet = workbook.createSheet("Chat export");
        ChatReportGenerator.generateHeaderRow(EXPORT_HEADERS, sheet, cellFormat.headerStyle());
        for (int i = 0; i < chats.size(); ++i) {
            Chat chat = chats.get(i);
            Row row = sheet.createRow(i + 1);
            row.setHeightInPoints(30.0f);
            List<ChatMessage> userMessages = chat.getChatMessages().stream().filter(cm -> cm.getRole() == ChatMessage.ChatMessageRole.USER).collect(Collectors.toList());
            List<ChatMessage> assistantMessages = chat.getChatMessages().stream().filter(cm -> cm.getRole() == ChatMessage.ChatMessageRole.ASSISTANT).collect(Collectors.toList());
            ChatReportGenerator.createContentCells(row, List.of(!userMessages.isEmpty() ? ((ChatMessage)userMessages.get(0)).getSendTime() : "", !userMessages.isEmpty() ? ((ChatMessage)userMessages.get(0)).getMessage() : "", !assistantMessages.isEmpty() ? ((ChatMessage)assistantMessages.get(0)).getMessage() : "", chat.getSummary(), chat.getChatType(), !userMessages.isEmpty() ? Optional.ofNullable(((ChatMessage)userMessages.get(0)).getModel()).orElse("") : "", ChatReportGenerator.printChatMessages(chat), userMessages.size(), assistantMessages.size(), userMessages.size() + assistantMessages.size(), ChatReportGenerator.calculateTokenCount(userMessages, assistantMessages), Optional.ofNullable(chat.getChatRating())), cellFormat);
        }
    }

    private static void createContentCells(Row row, List<Object> data, CellFormat cellFormat) {
        for (int i = 0; i < data.size(); ++i) {
            ChatReportGenerator.createContentCell(row, data.get(i), i, cellFormat);
        }
    }

    private static String printChatMessages(Chat chat) {
        return chat.getChatMessages().stream().map(cm -> "%s\n%s".formatted(ChatReportGenerator.getRoleHeader(cm.getRole()), cm.getMessage())).collect(Collectors.joining("\n\n"));
    }

    private static String getRoleHeader(ChatMessage.ChatMessageRole role) {
        return switch (role) {
            case ChatMessage.ChatMessageRole.ASSISTANT -> "***ASSISTANT_MESSAGE***";
            case ChatMessage.ChatMessageRole.SYSTEM -> "***SYSTEM_MESSAGE***";
            case ChatMessage.ChatMessageRole.USER -> "***USER_MESSAGE***";
            case ChatMessage.ChatMessageRole.ERROR -> "***ERROR_MESSAGE***";
            default -> throw new IllegalArgumentException("Unknown ChatMessageRole role: %s".formatted(new Object[]{role}));
        };
    }

    private static int calculateTokenCount(List<ChatMessage> userMessages, List<ChatMessage> assistantMessages) {
        return Stream.concat(userMessages.stream(), assistantMessages.stream()).filter(cm -> cm.getTokenCount() != null).mapToInt(ChatMessage::getTokenCount).sum();
    }

    private static void createContentCell(Row row, Object value, int contentCellIndex, CellFormat cellFormat) {
        Cell cell = row.createCell(contentCellIndex);
        if (value instanceof Optional) {
            Optional optionalValue = (Optional)value;
            optionalValue.ifPresent(val -> ChatReportGenerator.createContentCell(row, val, contentCellIndex, cellFormat));
            return;
        }
        if (value instanceof LocalDate) {
            cell.setCellValue((LocalDate)value);
            cell.setCellStyle(cellFormat.dateStyle());
            return;
        }
        if (value instanceof LocalDateTime) {
            cell.setCellValue((LocalDateTime)value);
            cell.setCellStyle(cellFormat.dateTimeStyle());
            return;
        }
        if (value instanceof Integer) {
            cell.setCellValue(Double.valueOf(((Integer)value).intValue()).doubleValue());
        } else if (value instanceof Long) {
            cell.setCellValue(Double.valueOf(((Long)value).longValue()).doubleValue());
        } else if (value instanceof Double) {
            cell.setCellValue(((Double)value).doubleValue());
        } else {
            cell.setCellValue(String.valueOf(value));
        }
        cell.setCellStyle(cellFormat.contentStyle());
    }

    private static void generateHeaderRow(List<HeaderConfig> headers, Sheet sheet, CellStyle cellStyle) {
        Row header = sheet.createRow(0);
        for (int i = 0; i < headers.size(); ++i) {
            Cell headerCell = header.createCell(i);
            headerCell.setCellValue(headers.get(i).header());
            headerCell.setCellStyle(cellStyle);
            HeaderConfig config = headers.get(i);
            if (config.width() != null) {
                sheet.setColumnWidth(i, config.width() * 256);
                continue;
            }
            sheet.autoSizeColumn(i);
        }
    }

    private record CellFormat(CellStyle headerStyle, CellStyle contentStyle, CellStyle dateStyle, CellStyle dateTimeStyle) {
        public CellFormat(Workbook workbook) {
            this(CellFormat.createHeaderStyle(workbook), CellFormat.createContentStyle(workbook), CellFormat.createDateStyle(workbook, "yyyy-mm-dd"), CellFormat.createDateStyle(workbook, "yyyy-mm-dd hh:mm:ss"));
        }

        private static CellStyle createDateStyle(Workbook workbook, String format) {
            CreationHelper creationHelper = workbook.getCreationHelper();
            CellStyle dateStyle = workbook.createCellStyle();
            dateStyle.setWrapText(true);
            dateStyle.setVerticalAlignment(VerticalAlignment.TOP);
            dateStyle.setDataFormat(creationHelper.createDataFormat().getFormat(format));
            return dateStyle;
        }

        private static CellStyle createContentStyle(Workbook workbook) {
            CellStyle contentStyle = workbook.createCellStyle();
            contentStyle.setWrapText(true);
            contentStyle.setVerticalAlignment(VerticalAlignment.TOP);
            return contentStyle;
        }

        private static CellStyle createHeaderStyle(Workbook workbook) {
            XSSFFont font = ((XSSFWorkbook)workbook).createFont();
            font.setFontName("Arial");
            font.setFontHeightInPoints((short)13);
            font.setBold(true);
            CellStyle headerStyle = workbook.createCellStyle();
            headerStyle.setFont((Font)font);
            return headerStyle;
        }
    }

    private record HeaderConfig(String header, Integer width) {
        public HeaderConfig(String header) {
            this(header, null);
        }
    }
}

