package fr.ird.observe.maven.plugins.toolbox;

/*-
 * #%L
 * ObServe Toolkit :: Maven plugin
 * %%
 * Copyright (C) 2017 - 2021 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import io.ultreia.java4all.i18n.spi.builder.I18nKeySet;
import io.ultreia.java4all.i18n.spi.builder.I18nModule;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;

import java.io.File;
import java.net.URLClassLoader;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * To execute the given
 * Created on 22/10/2020.
 *
 * @author Tony Chemit - dev@tchemit.fr
 * @since 8.0.1
 */
@Mojo(name = "execute-runner", threadSafe = true, requiresDependencyCollection = ResolutionScope.COMPILE)
public class ExecuteRunnerMojo extends ToolboxMojoSupport {

    /**
     * To activate verbose mode.
     */
    @Parameter(property = "executeRunner.verbose", defaultValue = "${maven.verbose}")
    private boolean verbose;
    /**
     * To skip the goal.
     */
    @Parameter(property = "executeRunner.skip", defaultValue = "false")
    private boolean skip;
    /**
     * To force some stuff in runners.
     */
    @Parameter(property = "executeRunner.force", defaultValue = "false")
    private boolean force;
    /**
     * To add project class path to the plugin class path.
     */
    @Parameter(property = "executeRunner.addProjectClassPath", defaultValue = "true")
    private boolean addProjectClassPath;
    /**
     * To add Java generated source root directory to class-path.
     */
    @Parameter(property = "executeRunner.addTargetRootSource", defaultValue = "false")
    private boolean addTargetRootSource;
    /**
     * Optional i18n getter file.
     */
    @Parameter(property = "executeRunner.i18n")
    private String i18n;
    /**
     * Fully qualified name of {@link Runnable} to execute.
     */
    @Parameter(required = true)
    private List<String> runnerTypes;
    /**
     * Java source root directory.
     */
    @Parameter(defaultValue = "${project.basedir}/src/main/java", required = true)
    private File sourceRoot;
    /**
     * Resources source root directory.
     */
    @Parameter(defaultValue = "${project.basedir}/src/main/resources", required = true)
    private File resourcesRoot;
    /**
     * Compile classes directory.
     */
    @Parameter(defaultValue = "${project.build.directory}/classes", required = true)
    private File compileRoot;
    /**
     * Java generated source root directory.
     */
    @Parameter(defaultValue = "${project.build.directory}/generated-sources/java", required = true)
    private File targetRoot;
    /**
     * Extra properties to inject in runners.
     */
    @Parameter
    private Map<String, String> extraProperties;

    public static abstract class MojoRunnable implements Runnable {

        public static final String COMPILE_SOURCE_DIRECTORY = "compile.source.directory";
        public static final String COMPILE_RESOURCES_DIRECTORY = "compile.resources.directory";
        public static final String GENERATED_JAVA_DIRECTORY = "target.directory";
        public static final String CLASSES_DIRECTORY = "compile.target.directory";
        protected I18nKeySet getterFile;
        protected Log log;
        private boolean force;
        private Map<String, String> extraProperties;

        public I18nKeySet getGetterFile() {
            return getterFile;
        }

        public void setGetterFile(I18nKeySet getterFile) {
            this.getterFile = getterFile;
        }

        public Log getLog() {
            return log;
        }

        public void setLog(Log log) {
            this.log = log;
        }

        public boolean isForce() {
            return force;
        }

        public void setForce(boolean force) {
            this.force = force;
        }

        public Map<String, String> getExtraProperties() {
            return extraProperties == null ? Collections.emptyMap() : extraProperties;
        }

        public void setExtraProperties(Map<String, String> extraProperties) {
            this.extraProperties = extraProperties;
        }
    }

    @Override
    protected void doAction() throws Exception {

        I18nKeySet getterFile = null;
        I18nModule i18nModule = null;
        if (i18n != null) {
            i18nModule = I18nModule.forGetter(getProject().getProperties());
            getterFile = i18nModule.getModuleKeySet(i18n);
        }
        ClassLoader contextClassLoader = Thread.currentThread().getContextClassLoader();
        try {
            URLClassLoader urlClassLoader = initClassLoader(getProject(), compileRoot == null ? sourceRoot : compileRoot, false, false, true, true, addProjectClassPath);
            Thread.currentThread().setContextClassLoader(urlClassLoader);

            for (String runnerType : runnerTypes) {
                getLog().info("Will execute runner: " + runnerType);
                System.setProperty(MojoRunnable.GENERATED_JAVA_DIRECTORY, targetRoot.getAbsolutePath());
                System.setProperty(MojoRunnable.CLASSES_DIRECTORY, compileRoot.getAbsolutePath());
                System.setProperty(MojoRunnable.COMPILE_SOURCE_DIRECTORY, sourceRoot.getAbsolutePath());
                System.setProperty(MojoRunnable.COMPILE_RESOURCES_DIRECTORY, resourcesRoot.getAbsolutePath());
                System.setProperty("base.directory", getProject().getBasedir().getAbsolutePath());

                MojoRunnable o = (MojoRunnable) urlClassLoader.loadClass(runnerType).getConstructor().newInstance();

                o.setGetterFile(getterFile);
                o.setExtraProperties(extraProperties);
                o.setLog(getLog());
                o.run();
            }

        } finally {
            if (getterFile != null) {
                i18nModule.storeModuleKeySet(getterFile);
            }
            Thread.currentThread().setContextClassLoader(contextClassLoader);
        }

        if (addTargetRootSource) {
            addCompileSourceRoots(targetRoot);
        }
    }

    @Override
    public boolean isVerbose() {
        return verbose;
    }

    @Override
    public void setVerbose(boolean verbose) {
        this.verbose = verbose;
    }

    @Override
    public boolean isSkip() {
        return skip;
    }

    public boolean isForce() {
        return force;
    }

    public void setForce(boolean force) {
        this.force = force;
    }
}
