package fr.ird.observe.toolkit.runner.navigation.id;

/*-
 * #%L
 * ObServe Toolkit :: Runner for Navigation
 * %%
 * Copyright (C) 2017 - 2021 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import fr.ird.observe.toolkit.runner.navigation.Template;
import fr.ird.observe.toolkit.templates.navigation.NodeLinkModel;
import fr.ird.observe.toolkit.templates.navigation.NodeModel;
import org.apache.commons.lang3.StringUtils;
import org.apache.maven.plugin.logging.Log;

import java.io.IOException;
import java.nio.file.Path;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * Created on 02/04/2021.
 *
 * @author Tony Chemit - dev@tchemit.fr
 * @since 5.0.17
 */
public abstract class ModelTemplate extends Template {

    public static final String PARENT_MODEL = "" +
            "    @Override\n" +
            "    public final %1$s getParent() {\n" +
            "        return (%1$s) super.getParent();\n" +
            "    }\n\n";
    public static final String CAPABILITY_MODEL = "" +
            "    public final %1$s get%2$s() {\n" +
            "        return getNode(%3$d);\n" +
            "    }\n\n";
    public static final String CAPABILITY_CONSTRUCTOR_MODEL = "" +
            "              new %1$s()%2$s";
    public static final String CAPABILITY_FIELD_MODEL = "" +
            "    private final transient %1$s %2$s;\n\n";
    public static final String CAPABILITY_FIELD_INIT_MODEL = "" +
            "        this.%1$s = forModelType(%2$s.class).orElse(null);\n";
    public static final String GET_MODULE = "" +
            "    public final %1$s get%2$s() {\n" +
            "        return %3$s;\n" +
            "    }\n\n";

    private final Map<String, String> modules = new LinkedHashMap<>();

    public ModelTemplate(Log log, String prefix, String suffix, Path targetDirectory) {
        super(log, prefix, suffix, targetDirectory);
    }

    public abstract String getNodeTemplate();

    public abstract String getModuleTemplate();

    public abstract String getProjectTemplate();

    @Override
    public void generate(NodeModel model) throws IOException {
        String content = generateNodeContent(model);
        Path generate = store(targetDirectory, model.getPackageName("").replace("data.", ""), model.getSimpleName(suffix), content);
        if (model.getLevel() == 0) {
            String moduleName = generate.getParent().getParent().toFile().getName();
            generateModule(generate, model, moduleName);
        }
        generateProject();
    }

    public void generateModule(Path generate, NodeModel model, String moduleName) throws IOException {
        String content = generateModuleContent(generate, model);
        String moduleClassName = "Module" + suffix.replace("Node", "Model");
        modules.put(moduleName, prefix + "." + moduleName + "." + moduleClassName);
        store(targetDirectory, moduleName, moduleClassName, content);
    }

    public void generateProject() throws IOException {
        String projectContent = generateProjectContent();
        store(targetDirectory, "", "Project" + suffix.replace("Node", "Model"), projectContent);
    }

    public String generateNodeContent(NodeModel model) {
        StringBuilder constructorBuilder = new StringBuilder();
        StringBuilder capabilityBuilder = new StringBuilder();
        Iterator<NodeLinkModel> iterator = model.getChildren().iterator();
        while (iterator.hasNext()) {
            NodeLinkModel child = iterator.next();
            boolean hasNext = iterator.hasNext();
            constructorBuilder.append(String.format(CAPABILITY_CONSTRUCTOR_MODEL, child.getName(prefix, suffix).replace(".data", ""), hasNext ? ",\n" : ""));
            capabilityBuilder.append(String.format(CAPABILITY_MODEL, child.getName(prefix, suffix).replace(".data", ""), StringUtils.capitalize(child.getPropertyName()), child.getOrder()));
        }
        int level = model.getLevel();
        return String.format(getNodeTemplate()
                , model.getPackageName(prefix).replace(".data", "")
                , now
                , model.getSimpleName(suffix)
                , model.getDtoType()
                , constructorBuilder.toString().trim()
                , level
                , level == -1 ? "" : String.format(PARENT_MODEL, model.getParentName(prefix, suffix).replace(".data", ""))
                , capabilityBuilder.toString()
        );
    }

    public String generateModuleContent(Path generate, NodeModel model) {
        String moduleName = generate.getParent().getParent().toFile().getName();
        List<NodeModel> nodes = model.collectNodes();
        StringBuilder capabilityBuilder = new StringBuilder();
        int index = 0;
        for (NodeModel child : nodes) {
            String[] nodeName = child.getPackageName("").replace("data.", "").split("\\.");
            String getterName = StringUtils.capitalize(nodeName[1]) + child.getSimpleName("");
            capabilityBuilder.append(String.format(CAPABILITY_MODEL, child.getName(prefix, suffix).replace(".data", ""), getterName, index++));
        }
        return String.format(getModuleTemplate()
                , prefix + "." + moduleName
                , now
                , model.getName(prefix, suffix).replace(".data", "")
                , capabilityBuilder.toString()
        );
    }

    public String generateProjectContent() {
        StringBuilder fieldBuilder = new StringBuilder();
        StringBuilder constructorBuilder = new StringBuilder();
        StringBuilder constructor2Builder = new StringBuilder();
        StringBuilder capabilityBuilder = new StringBuilder();
        Iterator<String> iterator = modules.keySet().iterator();
        while (iterator.hasNext()) {
            String moduleName = iterator.next();
            boolean hasNext = iterator.hasNext();
            String moduleFqn = modules.get(moduleName);
            fieldBuilder.append(String.format(CAPABILITY_FIELD_MODEL, moduleFqn, moduleName));
            constructorBuilder.append(String.format(CAPABILITY_CONSTRUCTOR_MODEL, moduleFqn, hasNext ? ",\n" : ""));
            constructor2Builder.append(String.format(CAPABILITY_FIELD_INIT_MODEL, moduleName, moduleFqn));
            capabilityBuilder.append(String.format(GET_MODULE, moduleFqn, StringUtils.capitalize(moduleName), moduleName));
        }
        return String.format(getProjectTemplate()
                , prefix
                , now
                , fieldBuilder.toString()
                , constructorBuilder.toString().trim()
                , constructor2Builder.toString().trim()
                , capabilityBuilder.toString()
        );
    }

}
