package fr.ird.observe.toolkit.runner.navigation.tree;

/*-
 * #%L
 * ObServe Toolkit :: Runner for Navigation
 * %%
 * Copyright (C) 2017 - 2021 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import fr.ird.observe.dto.BusinessDto;
import fr.ird.observe.dto.data.ContainerDto;
import fr.ird.observe.dto.data.SimpleDto;
import fr.ird.observe.spi.module.BusinessProject;
import fr.ird.observe.toolkit.navigation.spi.tree.descriptor.NavigationNodeType;
import fr.ird.observe.toolkit.runner.navigation.Template;
import fr.ird.observe.toolkit.templates.navigation.NodeLinkModel;
import fr.ird.observe.toolkit.templates.navigation.NodeModel;
import io.ultreia.java4all.i18n.spi.builder.I18nKeySet;
import io.ultreia.java4all.util.ServiceLoaders;
import org.apache.commons.lang3.StringUtils;
import org.apache.maven.plugin.logging.Log;

import java.beans.Introspector;
import java.io.IOException;
import java.nio.file.Path;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.stream.Collectors;

/**
 * Created on 03/04/2021.
 *
 * @author Tony Chemit - dev@tchemit.fr
 * @since 5.0.17
 */
public abstract class ModelTemplate extends Template {
    public static final String PATH_MAPPING_MODEL = "" +
            "package %1$s;\n\n" +
            "import com.google.common.collect.ImmutableMap;\n" +
            "import fr.ird.observe.dto.BusinessDto;\n" +
            "import fr.ird.observe.toolkit.navigation.spi.tree.DtoToToolkitTreePathMapping;\n\n" +
            "import javax.annotation.Generated;\n" +
            "import java.nio.file.Path;\n\n" +
            "@Generated(value = \"fr.ird.observe.toolkit.runner.navigation.GenerateNavigationModel\", date = \"%2$s\")\n" +
            "public class %3$s extends DtoToToolkitTreePathMapping {\n\n" +
            "    private static %3$s INSTANCE;\n\n" +
            "    public static %3$s get() {\n" +
            "        return INSTANCE == null ? INSTANCE = new %3$s() : INSTANCE;\n" +
            "    }\n\n" +
            "    protected %3$s() {\n" +
            "        super(ImmutableMap\n" +
            "                      .<Class<? extends BusinessDto>, Path>builder()\n" +
            "                      %4$s\n" +
            "                      .build());\n" +
            "    }\n\n" +
            "}";

    public static final String GET_USER_OBJECT = "" +
            "    @Override\n" +
            "    public final %1$s getUserObject() {\n" +
            "        return (%1$s) super.getUserObject();\n" +
            "    }\n\n";
    public static final String PARENT_MODEL = "" +
            "    @Override\n" +
            "    public final %1$s getParent() {\n" +
            "        return (%1$s) super.getParent();\n" +
            "    }\n\n";
    public static final String CAPABILITY_SINGLE = "" +
            "    public final %1$s get%2$s() {\n" +
            "        return find(%1$s.class);\n" +
            "    }\n\n";
    public static final String CAPABILITY_MULTIPLE = "" +
            "    public final %1$s get%2$s(fr.ird.observe.dto.ToolkitIdDtoBean %3$s) {\n" +
            "        return find(%1$s.class, %3$s);\n" +
            "    }\n\n";
    public static final String DEFAULT_CAPABILITY_MULTIPLE = "" +
            "    public final %1$s get%2$s(fr.ird.observe.dto.ToolkitIdDtoBean %3$s) {\n" +
            "        return get%2$s().getChildren(%3$s);\n" +
            "    };\n\n";
    public static final String ROOT_PATH = "" +
            "    public final static Path PATH = Path.of(\"/\");\n\n";
    public static final String NODE_PATH = "" +
            "    public final static Path PATH = %1$s.PATH.resolve(\"%2$s\");\n\n";
    public static final String INTERCEPTOR_CASE = "           case \"%1$s\":\n" +
            "                provider.intercept((%1$s) node);\n" +
            "                break;\n";
    public static final String INTERCEPTOR_METHOD = "" +
            "   default void intercept(%1$s node) {\n" +
            "    }\n\n";
    protected final String modelName;
    protected final BusinessProject businessProject;
    protected final Set<Class<? extends ContainerDto<?>>> containerDataTypes;
    private final Map<String, String> pathMapping = new LinkedHashMap<>();
    private final I18nKeySet getterFile;
    private final Set<Class<? extends SimpleDto>> simpleDataTypes;
    protected StringBuilder interceptorCaseBuilder = new StringBuilder();
    protected StringBuilder interceptorMethodsBuilder = new StringBuilder();

    public ModelTemplate(Log log, I18nKeySet getterFile, String prefix, String suffix, Path targetDirectory, String modelName) {
        super(log, prefix, suffix, targetDirectory);
        this.getterFile = getterFile;
        this.modelName = modelName;
        businessProject = ServiceLoaders.loadUniqueService(BusinessProject.class);
        containerDataTypes = businessProject.getContainerDataTypes();
        simpleDataTypes = businessProject.getSimpleDataTypes();
    }

    public abstract String getNodeTemplate();

    public abstract String getBeanTemplate();

    protected void registerInterceptorNode(NodeModel model) {
        String childName = model.getName(prefix, suffix);
        interceptorCaseBuilder.append(" ").append(String.format(INTERCEPTOR_CASE, childName));
        interceptorMethodsBuilder.append(" ").append(String.format(INTERCEPTOR_METHOD, childName));
    }

    public abstract void generateMapping() throws IOException;

    protected void generateMapping(String path) throws IOException {
        String data = pathMapping.entrySet().stream()
                .map(e -> String.format("                     .put(%s.class, %s)\n", e.getKey(), e.getValue()))
                .collect(Collectors.joining(" "));
        String className = "DtoTo" + suffix.replace("TreeNode", "") + "TreePathMapping";
        String content = String.format(PATH_MAPPING_MODEL
                , "fr.ird.observe.spi.tree." + path
                , now
                , className
                , data.substring(1).trim());
        store(targetDirectory.getParent().getParent(), "spi.tree." + path, className, content);
    }

    public void generate(NodeModel model) throws IOException {
        if (model.isRoot()) {
            registerInterceptorNode(model);
        }
        if (getNodeTemplate() != null) {
            generateNode(model);
        }
        if (getBeanTemplate() != null) {
            generateBean(model);
        }
        registerPath(pathMapping, model);
    }

    protected void registerPath(Map<String, String> pathMapping, NodeModel model) {
        String dtoType = model.getDtoType();
        if (model.getLevel() < 1 || dtoType == null) {
            return;
        }
        if (!model.isOpenList()) {
            this.pathMapping.put(dtoType, model.getName(prefix, suffix) + ".PATH");
        }
    }

    public void generateNode(NodeModel model) throws IOException {
        String content = generateNodeContent(model);
        store(targetDirectory, model.getPackageName(""), model.getSimpleName(suffix), content);
    }

    public void generateBean(NodeModel model) throws IOException {
        String content = generateBeanContent(model);
        store(targetDirectory, model.getPackageName(""), model.getSimpleName(suffix + "Bean"), content);
    }

    protected String generateNodeContent(NodeModel model) {
        StringBuilder fieldsBuilder = new StringBuilder();
        StringBuilder capabilityBuilder = new StringBuilder();
        String beanSimpleName = model.getSimpleName(suffix + "Bean");
        capabilityBuilder.append(String.format(GET_USER_OBJECT, beanSimpleName));
        if (model.getLevel() > -1) {
            String parentName = model.getParentName(prefix, suffix);
            capabilityBuilder.append(String.format(PARENT_MODEL, parentName));
            fieldsBuilder.append(String.format(NODE_PATH, parentName, getPath(model)));
        } else {
            fieldsBuilder.append(ROOT_PATH);
        }
//        fieldsBuilder.append(String.format(CAPABILITY_SINGLE_FIELD, beanSimpleName, "bean"));
        for (NodeLinkModel child : model.getChildren()) {
            String type = child.getName(prefix, suffix);
            String propertyName = child.getPropertyName();

            String getterName = StringUtils.capitalize(propertyName);
            if (child.getCapability().isMultiple()) {
                String parameterId = Introspector.decapitalize(child.getSimpleName("").replace("List", "") + "Id");
                capabilityBuilder.append(String.format(CAPABILITY_MULTIPLE, type, getterName, parameterId));
            } else {
                capabilityBuilder.append(String.format(CAPABILITY_SINGLE, type, getterName));
                if (child.isOpenList()) {
                    String parameterId = Introspector.decapitalize(getterName + "Id");
                    capabilityBuilder.append(String.format(DEFAULT_CAPABILITY_MULTIPLE, type.replace("List", ""), getterName, parameterId));
                }
            }
        }

        return String.format(getNodeTemplate()
                , model.getPackageName(prefix)
                , now
                , model.getSimpleName(suffix)
                , fieldsBuilder
                , capabilityBuilder);
    }

    protected String generateBeanContent(NodeModel model) {
        StringBuilder capabilityBuilder = new StringBuilder();
        Map<String, String> defaultStates = new TreeMap<>();
        addType(defaultStates, model);
        addMultiplicity(defaultStates, model);
        addDefaultMapping(defaultStates, model);
        switch (model.getType()) {
            case Root:
                break;
            case ReferentialPackage:
                generateReferentialPackageBeanContent(model, defaultStates);
                break;
            case ReferentialType:
                generateReferentialTypeBeanContent(model, defaultStates);
                break;
            case OpenList:
                generateOpenListBeanContent(model, defaultStates);
                break;
            case Open:
                generateOpenBeanContent(model, defaultStates);
                break;
            case Edit:
                generateEditBeanContent(model, defaultStates);
                break;
            case Table:
                generateTableBeanContent(model, defaultStates);
                break;
            case Simple:
                generateSimpleBeanContent(model, defaultStates);
                break;
        }
        addDefaultStates(capabilityBuilder, defaultStates);
        return generate0(model, capabilityBuilder);
    }

    protected String generate0(NodeModel model, StringBuilder capabilityBuilder) {
        return String.format(getBeanTemplate()
                , model.getPackageName(prefix)
                , now
                , model.getSimpleName(suffix + "Bean")
                , model.getType().name()
                , capabilityBuilder.toString()
        );
    }

    protected void addMapping(Map<String, String> capabilityBuilder, Map<String, String> mapping) {
        String map;
        if (mapping == null || mapping.isEmpty()) {
            map = "ImmutableMap.of()";
        } else {
            List<String> collect = mapping.entrySet().stream().map(e -> "                      .put(\"" + e.getKey() + "\", " + e.getValue() + ")\n").collect(Collectors.toList());
            map = String.format("ImmutableMap.<String, Class<?>>builder()\n" +
                                        "                        %1$s\n" +
                                        "                        .build()", String.join("  ", collect).substring(2).trim());
        }
        capabilityBuilder.put("STATE_NODE_MAPPING", map);
    }

    protected void addType(Map<String, String> capabilityBuilder, NodeModel model) {
        capabilityBuilder.put("STATE_TYPE", NavigationNodeType.class.getName() + "." + model.getType().name());
    }

    protected void addMultiplicity(Map<String, String> capabilityBuilder, NodeModel model) {
        NodeModel parent = model.getParent();
        NodeLinkModel nodeLinkModel = parent == null ? null : parent.getChildren().stream().filter(l -> l.getTargetModel().equals(model)).findFirst().orElseThrow();
        boolean multiple = nodeLinkModel != null && nodeLinkModel.getCapability().isMultiple();
        capabilityBuilder.put("STATE_MULTIPLICITY", multiple + "");
    }

    protected void addIconPath(Map<String, String> capabilityBuilder, NodeModel model) {
        capabilityBuilder.put("STATE_ICON_PATH", String.format("\"navigation.%1$s\"", model.getIconPath()));
    }

    protected void addPath(Map<String, String> capabilityBuilder, NodeModel model) {
        capabilityBuilder.put("STATE_PATH", model.getPath());
    }

    protected void addReferentialType(Map<String, String> capabilityBuilder, NodeModel model) {
        capabilityBuilder.put("STATE_REFERENTIAL_TYPE", model.getDtoType() + ".class");
    }

    protected void addDataType(Map<String, String> capabilityBuilder, NodeModel model) {
        capabilityBuilder.put("STATE_DATA_TYPE", model.getDtoType() + ".class");
    }

    private void addDefaultStates(StringBuilder capabilityBuilder, Map<String, String> defaultStates) {
        String extra = "    @Override\n" +
                "    protected final Map<ToolkitTreeNodeBeanState<?>, Object> defaultStates() {\n" +
                "        return %1$s;\n" +
                "    }\n\n";
        String map;
        if (defaultStates == null || defaultStates.isEmpty()) {
            map = "ImmutableMap.of()";
        } else {
            List<String> collect = defaultStates.entrySet().stream().map(e -> "              .put(" + e.getKey() + ", " + e.getValue() + ")\n").collect(Collectors.toList());
            map = String.format("ImmutableMap.<ToolkitTreeNodeBeanState<?>, Object>builder()\n" +
                                        "                %1$s\n" +
                                        "                .build()", String.join("  ", collect).substring(2).trim());
        }
        capabilityBuilder.append(String.format(extra, map));
    }

    protected void addText(Map<String, String> capabilityBuilder, NodeModel model) {
        capabilityBuilder.put("STATE_TEXT", model.getText(getterFile));
    }

    private void addDefaultMapping(Map<String, String> defaultStates, NodeModel model) {
        Map<String, String> mappingState = new LinkedHashMap<>();
        List<NodeLinkModel> children = model.getChildren();
        boolean capabilityLeaf = children.isEmpty();
        boolean capabilitySelectReference = model.isEdit() || (model.isOpenList() && model.getLevel() == 0);
        boolean capabilityEditReference = model.isOpen();
        boolean capabilitySelectReferenceContainer = false;
        boolean capabilityEditReferenceContainer = false;
        for (NodeLinkModel child : children) {
            String propertyName = child.getPropertyName();
            if (child.isReferentialPackage()) {
                propertyName = "referential-" + propertyName;
            }
            mappingState.put(propertyName, child.getName(prefix, suffix) + ".class");
            if (child.isEdit()) {
                capabilitySelectReferenceContainer = true;
            }
            if (child.isOpen()) {
                capabilityEditReferenceContainer = true;
            }
        }
        addMapping(defaultStates, mappingState);
        addCapability(defaultStates, capabilityLeaf, capabilitySelectReference || capabilityEditReference, capabilityEditReference, capabilitySelectReferenceContainer, capabilityEditReferenceContainer);
    }

    protected void addCapability(Map<String, String> defaultStates, boolean capabilityLeaf, boolean capabilitySelectReference, boolean capabilityEditReference, boolean capabilitySelectReferenceContainer, boolean capabilityEditReferenceContainer) {
        if (capabilitySelectReference) {
            defaultStates.put("STATE_CAPABILITY_SELECT_REFERENCE", "true");
        }
        if (capabilityEditReference) {
            defaultStates.put("STATE_CAPABILITY_EDIT_REFERENCE", "true");
        }
        if (capabilityLeaf) {
            defaultStates.put("STATE_CAPABILITY_LEAF", "true");
        } else {
            defaultStates.put("STATE_CAPABILITY_CONTAINER", "true");
            if (capabilitySelectReferenceContainer) {
                defaultStates.put("STATE_CAPABILITY_SELECT_REFERENCE_CONTAINER", "true");
            }
            if (capabilityEditReferenceContainer) {
                defaultStates.put("STATE_CAPABILITY_EDIT_REFERENCE_CONTAINER", "true");
            }
        }
    }

    protected String getPath(NodeModel model) {
        NodeModel parent = model.getParent();
        if (parent == null) {
            return "/";
        }
        for (NodeLinkModel child : parent.getChildren()) {
            if (child.getTargetModel().equals(model)) {
                String propertyName = child.getPropertyName();
                if (child.isEdit() || ((child.isOpenList() || child.isOpen()) && model.getLevel() == 0)) {
                    return String.format("%s:${%s}", propertyName, child.getSimpleName(""));
                }
                if (child.isOpen()) {
                    return String.format("${%s}", child.getSimpleName(""));
                }
                if (child.isReferentialPackage()) {
                    propertyName = "referential-" + propertyName;
                }
                return propertyName;
            }
        }
        throw new IllegalStateException("Can't find path for model: " + model);
    }

    protected void generateReferentialPackageBeanContent(NodeModel model, Map<String, String> defaultStates) {
        addIconPath(defaultStates, model);
        addText(defaultStates, model);
        addPath(defaultStates, model);
    }

    protected void generateReferentialTypeBeanContent(NodeModel model, Map<String, String> defaultStates) {
        addIconPath(defaultStates, model);
        addText(defaultStates, model);
        addReferentialType(defaultStates, model);
        addPath(defaultStates, model);
    }

    protected void generateOpenBeanContent(NodeModel model, Map<String, String> defaultStates) {
        addIconPath(defaultStates, model);
        addDataType(defaultStates, model);
        if (model.getLevel() == 0) {
            addPath(defaultStates, model);
        }
    }

    protected void generateOpenListBeanContent(NodeModel model, Map<String, String> defaultStates) {
        addIconPath(defaultStates, model);
        if (model.getLevel() == 0) {
            addDataType(defaultStates, model);
        }
        addText(defaultStates, model);
        addPath(defaultStates, model);
    }

    protected void generateEditBeanContent(NodeModel model, Map<String, String> defaultStates) {
        addIconPath(defaultStates, model);
        addDataType(defaultStates, model);
        addText(defaultStates, model);
        addPath(defaultStates, model);
    }

    protected void generateSimpleBeanContent(NodeModel model, Map<String, String> defaultStates) {
        addIconPath(defaultStates, model);
        addDataType(defaultStates, model);
        addText(defaultStates, model);
        addPath(defaultStates, model);
    }

    protected void generateTableBeanContent(NodeModel model, Map<String, String> defaultStates) {
        addIconPath(defaultStates, model);
        addDataType(defaultStates, model);
        addText(defaultStates, model);
        addPath(defaultStates, model);
    }

    protected boolean rejectDtoType(Class<? extends BusinessDto> dtoType) {
        return BusinessProject.isReferential(dtoType)
                || businessProject.getMapping().getReferenceType(dtoType) == null
                || simpleDataTypes.contains(dtoType)
                || (containerDataTypes.contains(dtoType) && businessProject.getMapping().getMainDtoType(dtoType) != dtoType);
//                || (!containerChildDataTypes.contains(dtoType)
//                && !openableDataTypes.contains(dtoType)
//                && !editableDataTypes.contains(dtoType))
//                ;
    }
}
