package fr.ird.observe.toolkit.runner.navigation.tree;

/*-
 * #%L
 * ObServe Toolkit :: Runner for Navigation
 * %%
 * Copyright (C) 2017 - 2021 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import fr.ird.observe.dto.data.DataDto;
import fr.ird.observe.spi.module.BusinessDataPackage;
import fr.ird.observe.spi.module.BusinessModule;
import fr.ird.observe.spi.module.BusinessProjectVisitor;
import fr.ird.observe.spi.module.BusinessSubModule;
import fr.ird.observe.spi.parent.ToParentIdProvider;
import fr.ird.observe.toolkit.templates.navigation.NodeLinkModel;
import fr.ird.observe.toolkit.templates.navigation.NodeModel;
import io.ultreia.java4all.i18n.spi.builder.I18nKeySet;
import org.apache.commons.lang3.StringUtils;
import org.apache.maven.plugin.logging.Log;

import java.beans.Introspector;
import java.io.IOException;
import java.nio.file.Path;
import java.util.Map;

/**
 * Created on 03/04/2021.
 *
 * @author Tony Chemit - dev@tchemit.fr
 * @since 5.0.17
 */
public class NavigationTemplate extends ModelTemplate {
    public static final String NODE_MODEL = "" +
            "package %1$s;\n\n" +
            "import fr.ird.observe.toolkit.navigation.tree.navigation.NavigationTreeNode;\n\n" +
            "import javax.annotation.Generated;\n" +
            "import java.nio.file.Path;\n\n" +
            "@Generated(value = \"fr.ird.observe.toolkit.runner.navigation.tree.NavigationTemplate\", date = \"%2$s\")\n" +
            "public class %3$s extends NavigationTreeNode {\n\n" +
            "%4$s" +
            "    public %3$s() {\n" +
            "        super(new %3$sBean());\n" +
            "    }\n\n" +
            "%5$s" +
            "}\n";
    public static final String BEAN_MODEL = "" +
            "package %1$s;\n\n" +
            "import com.google.common.collect.ImmutableMap;\n" +
            "import fr.ird.observe.spi.module.ObserveBusinessProject;\n" +
            "import fr.ird.observe.toolkit.navigation.spi.tree.descriptor.NavigationNodeType;\n" +
            "import fr.ird.observe.toolkit.navigation.tree.ToolkitTreeNodeBeanState;\n\n" +
            "import javax.annotation.Generated;\n" +
            "import java.util.Map;\n" +
            "import java.util.function.Supplier;\n\n" +
            "@Generated(value = \"fr.ird.observe.toolkit.runner.navigation.tree.NavigationTemplate\", date = \"%2$s\")\n" +
            "public class %3$s extends fr.ird.observe.toolkit.navigation.tree.navigation.bean.%4$sNavigationTreeNodeBean {\n" +
            "%5$s" +
            "    @Override\n" +
            "    protected Supplier<ObserveBusinessProject> getProjectSupplier() {\n" +
            "        return ObserveBusinessProject::get;\n" +
            "    }\n}\n";

    public static final String ID_PROVIDER_MODEL = "" +
            "package %1$s;\n\n" +
            "import fr.ird.observe.dto.ToolkitIdDtoBean;\n" +
            "import fr.ird.observe.dto.ToolkitParentIdDtoBean;\n\n" +
            "import javax.annotation.Generated;\n" +
            "import java.util.Objects;\n\n" +
            "@Generated(value = \"fr.ird.observe.toolkit.runner.navigation.tree.NavigationTemplate\", date = \"%2$s\")\n" +
            "public interface %3$s extends ToParentIdProvider {\n" +
            "\n" +
            "    static ToolkitParentIdDtoBean getParent(%3$s provider, ToolkitIdDtoBean shortId) {\n" +
            "        String id = Objects.requireNonNull(shortId.getId());\n" +
            "        switch (shortId.getType().getName()) {\n" +
            "            %4$s\n" +
            "        }\n" +
            "        return null;\n" +
            "    }\n\n" +
            "    @Override\n" +
            "    default ToolkitParentIdDtoBean getParent(ToolkitIdDtoBean shortId) {\n" +
            "        return getParent(this, shortId);\n" +
            "    }\n\n" +
            "    %5$s\n" +
            "}";

    public static final String ID_CASE = "           case \"%1$s\":\n" +
            "                return provider.get%2$sParent(id);\n";
    public static final String ID_METHOD = "" +
            "   ToolkitParentIdDtoBean get%1$sParent(String %2$s);\n\n";
    public static final String INTERCEPTOR_MODEL = "" +
            "package %1$s;\n\n" +
            "import fr.ird.observe.toolkit.navigation.tree.ToolkitTreeNode;\n" +
            "import fr.ird.observe.toolkit.navigation.spi.tree.ToolkitTreeNodeInterceptor;\n\n" +
            "import javax.annotation.Generated;\n" +
            "import java.util.Objects;\n\n" +
            "@Generated(value = \"fr.ird.observe.toolkit.runner.navigation.tree.GenerateNavigationModel\", date = \"%2$s\")\n" +
            "public interface %3$s extends ToolkitTreeNodeInterceptor {\n" +
            "\n" +
            "    static void intercept(%3$s provider, ToolkitTreeNode node) {\n" +
            "        switch (Objects.requireNonNull(node.getClass().getName())) {\n" +
            "            %4$s\n" +
            "        }\n" +
            "    }\n\n" +
            "    @Override\n" +
            "    default void intercept(ToolkitTreeNode node) {\n" +
            "        intercept(this, node);\n" +
            "    }\n\n" +
            "    %5$s\n" +
            "}";

    StringBuilder idCaseBuilder = new StringBuilder();
    StringBuilder idMethodsBuilder = new StringBuilder();

    public NavigationTemplate(Log log, I18nKeySet getterFile, Path targetDirectory, String modelName) {
        super(log, getterFile, GenerateNavigationModel.PACKAGE, GenerateNavigationModel.SUFFIX, targetDirectory, modelName);
    }

    @Override
    protected void registerPath(Map<String, String> pathMapping, NodeModel model) {
        super.registerPath(pathMapping, model);
        String dtoType = model.getDtoType();
        if (model.getLevel() == 0 && dtoType != null && model.isOpenList()) {
            pathMapping.put(dtoType, model.getName(prefix, suffix) + ".PATH");
        }
    }

    @Override
    public void generate(NodeModel model) throws IOException {
        super.generate(model);
        for (NodeLinkModel child : model.getChildren()) {
            registerInterceptorNode(child.getTargetModel());
        }
    }

    @Override
    public void generateMapping() throws IOException {
        generateMapping("navigation");
        BusinessProjectVisitor visitor = new BusinessProjectVisitor() {
            @Override
            public void enterSubModuleDataType(BusinessModule module, BusinessSubModule subModule, BusinessDataPackage dataPackage, Class<? extends DataDto> dtoType) {
                if (rejectDtoType(dtoType)) {
                    return;
                }
                StringBuilder getterName = new StringBuilder();
                getterName.append(StringUtils.capitalize(module.getName()));
                getterName.append(StringUtils.capitalize(subModule.getName()));
                String simpleName = dtoType.getSimpleName().replace("Dto", "");
                getterName.append(simpleName);
                idCaseBuilder.append(" ").append(String.format(ID_CASE, dtoType.getName(), getterName));
                idMethodsBuilder.append(" ").append(String.format(ID_METHOD, getterName, Introspector.decapitalize(simpleName) + "Id"));
            }
        };
        businessProject.accept(visitor);

        {
            String className = modelName + "ToParentIdProvider";
            String content = String.format(ID_PROVIDER_MODEL
                    , ToParentIdProvider.class.getPackageName()
                    , now
                    , className
                    , idCaseBuilder.substring(1).trim()
                    , idMethodsBuilder.substring(1).trim());
            store(targetDirectory.getParent().getParent(), "spi.parent", className, content);
        }
        {
            String className = modelName + "NavigationTreeNodeInterceptor";
            String content = String.format(INTERCEPTOR_MODEL
                    , "fr.ird.observe.spi.tree.navigation"
                    , now
                    , className
                    , interceptorCaseBuilder.substring(1).trim()
                    , interceptorMethodsBuilder.substring(1).trim());
            store(targetDirectory.getParent().getParent(), "spi.tree.navigation", className, content);
        }
    }

    @Override
    public String getNodeTemplate() {
        return NODE_MODEL;
    }

    @Override
    public String getBeanTemplate() {
        return BEAN_MODEL;
    }

}
