package fr.ird.observe.toolkit.templates;

/*-
 * #%L
 * ObServe Toolkit :: Templates
 * %%
 * Copyright (C) 2017 - 2021 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import fr.ird.observe.toolkit.templates.io.TagValuesExtractor;
import io.ultreia.java4all.util.SortedProperties;
import org.codehaus.plexus.component.annotations.Component;
import org.nuiton.eugene.Template;
import org.nuiton.eugene.models.object.ObjectModel;
import org.nuiton.eugene.models.object.ObjectModelGenerator;
import org.nuiton.eugene.models.object.ObjectModelType;

import java.io.File;
import java.io.IOException;
import java.io.Writer;
import java.nio.file.Path;
import java.util.Map;
import java.util.Set;

/**
 * Created on 26/01/2021.
 *
 * @author Tony Chemit - dev@tchemit.fr
 * @since 8.0.5
 */
@Component(role = Template.class, hint = "fr.ird.observe.toolkit.templates.ExtractTagValues")
public class ExtractTagValues extends ObjectModelGenerator {

    private String filePrefix;
    private Path outputFile;
    private Path resourceDirectory;
    private TagValuesExtractor extractor;

    @Override
    public void applyTemplate(ObjectModel model, File destDir) throws IOException {
        String packageName = getConfiguration().getProperty(PROP_DEFAULT_PACKAGE);

        Path rootResourcePath = getConfiguration().getProperty(PROP_RESOURCE_DIRECTORY, File.class).toPath();
        Path rootOutputPath = getConfiguration().getProperty(PROP_OUTPUT_DIRECTORY, File.class).toPath().getParent().resolve("java");

        String classifier;
        if (packageName.contains(".dto")) {
            classifier = "dto";
        } else if (packageName.contains(".entities")) {
            classifier = "persistence";
        } else {
            throw new IllegalStateException("Can't manage packageName: " + packageName);
        }
        resourceDirectory = TagValuesExtractor.getModelPath(rootResourcePath, model, classifier);
        getLog().info(String.format("[%s] Will generate tag values at %s", classifier, resourceDirectory));

        extractor = new TagValuesExtractor(model, packageName, classifier, getLog());
        extractor.load(getClassLoader(), resourceDirectory);

        Set<String> tagValueKeys = extractor.getModelTagValueKeys();
        for (String tagValuesKey : tagValueKeys) {
            filePrefix = tagValuesKey;
            super.applyTemplate(model, destDir);
        }
        extractor.store(rootOutputPath);
    }

    @Override
    protected void generateFromElement(Object element, File destDir, String filename, ObjectModelType type) {
        if (ObjectModelType.OBJECT_MODEL != type) {
            // only generate on model
            return;
        }
        super.generateFromElement(element, destDir, filename, type);
    }

    @Override
    public void generateFromModel(Writer output, ObjectModel input) throws IOException {
        // get existing tag values
        SortedProperties fromFile = extractor.createOrLoadFromFile(outputFile, filePrefix);

        // get tag values from model
        Map<String, String> fromModel = extractor.loadFromModel(filePrefix);

        // merge and store
        extractor.mergeAndStore(fromFile, fromModel, outputFile, output, filePrefix);
    }

    @Override
    public String getFilenameForModel(ObjectModel model) {
        return filePrefix + ".properties";
    }

    @Override
    protected File getDestinationFile(File destDir, String filename) {
        int endIndex = filename.indexOf("-");
        outputFile = resourceDirectory.resolve(filename.substring(0, endIndex)).resolve(filename.substring(filename.lastIndexOf("-") + 1));
        return outputFile.toFile();
    }
}
