package fr.ird.observe.toolkit.templates;

/*-
 * #%L
 * ObServe Toolkit :: Templates
 * %%
 * Copyright (C) 2017 - 2021 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import fr.ird.observe.dto.data.DataDto;
import fr.ird.observe.dto.referential.ReferentialDto;
import fr.ird.observe.spi.type.TypeTranslators;
import io.ultreia.java4all.classmapping.ImmutableClassMapping;
import org.nuiton.eugene.GeneratorUtil;
import org.nuiton.eugene.java.extension.ImportsManager;
import org.nuiton.eugene.java.extension.ObjectModelAnnotation;
import org.nuiton.eugene.models.object.ObjectModel;
import org.nuiton.eugene.models.object.ObjectModelAttribute;
import org.nuiton.eugene.models.object.ObjectModelClass;
import org.nuiton.eugene.models.object.ObjectModelClassifier;
import org.nuiton.eugene.models.object.ObjectModelElement;
import org.nuiton.eugene.models.object.ObjectModelInterface;
import org.nuiton.eugene.models.object.ObjectModelJavaModifier;
import org.nuiton.eugene.models.object.ObjectModelModifier;
import org.nuiton.eugene.models.object.ObjectModelOperation;
import org.nuiton.eugene.models.object.ObjectModelParameter;

import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;




/**
 * @author Tony Chemit - dev@tchemit.fr
 * @since 5
 */
@SuppressWarnings({"StringOperationCanBeSimplified", "UnusedAssignment", "UnusedReturnValue", "unused"})
public interface TemplateContract {

    static boolean isReferentialFromPackageName(String packageName) {
        return packageName.contains(".referential");
    }

    static String cleanDtoType(boolean referential, String fqn) {
        return TypeTranslators.getTranslator(referential ? ReferentialDto.class : DataDto.class).cleanType(fqn);
    }

    ObjectModelClass createClass(String className, String packageName);

    ObjectModelParameter addParameter(ObjectModelOperation operation, Class<?> type, String name);

    void addImport(ObjectModelClass output, Class<?> type);

    void addImport(ObjectModelClass output, String type);

    void setSuperClass(ObjectModelClass output, Class<?> superClass);

    ObjectModelAnnotation addAnnotation(ObjectModelClass output, ObjectModelElement output1, Class<?> annotationType);

    void addAnnotationParameter(ObjectModelClass output, ObjectModelAnnotation annotation, String name, String value);

    ObjectModelOperation addConstructor(ObjectModelClass output, ObjectModelJavaModifier modifiers);

    ObjectModelOperation addOperation(
            ObjectModelClassifier classifier,
            String name,
            String type,
            ObjectModelModifier... modifiers);

    void setOperationBody(ObjectModelOperation constructor, String body);

    ObjectModelAttribute addAttribute(ObjectModelClassifier classifier, String name, String type, String value, ObjectModelModifier... modifiers);

    ObjectModel getModel();

    default void addAnnotationClassParameter(ImportsManager manager, ObjectModelClass output, ObjectModelAnnotation annotation, String name, Class<?> value) {
        String importAndSimplify = manager.importAndSimplify(value.getName());
        addAnnotationParameter(output, annotation, name, "{" + importAndSimplify + ".class}");
    }

    default void addAnnotationClassParameter(ImportsManager manager, ObjectModelClass output, ObjectModelAnnotation annotation, String name, String value) {
        String importAndSimplify = manager.importAndSimplify(value);
        addAnnotationParameter(output, annotation, name, "{" + importAndSimplify + ".class}");
    }

    default void addAnnotationClassParameter(ImportsManager manager, ObjectModelClass output, ObjectModelAnnotation annotation, String name, Class<?>... values) {
        StringBuilder content = new StringBuilder("{");
        int max = values.length;
        for (int i = 0; i < max; i++) {
            Class<?> value = values[i];

            addImport(output, value);
            String importAndSimplify = manager.importAndSimplify(value.getName());
            content.append(importAndSimplify).append(".class");
            if (i + 1 < max) {
                content.append(", ");
            }
        }
        content.append("}");
        addAnnotationParameter(output, annotation, name, content.toString());
    }

    default void addAnnotationClassParameter(ObjectModelClass output, ObjectModelAnnotation annotation, String name, Iterable<String> values) {
        StringBuilder content = new StringBuilder("{");
        Iterator<String> iterator = values.iterator();
        while (iterator.hasNext()) {
            String value = iterator.next();
            addImport(output, value);
            content.append(value).append(".class");
            if (iterator.hasNext()) {
                content.append(", ");
            }
        }
        content.append("}");
        addAnnotationParameter(output, annotation, name, content.toString());
    }

    default void addStaticFactory(ObjectModelClass aClass, String realClassName) {
        String className = realClassName == null ? aClass.getQualifiedName() : realClassName;
        addAttribute(aClass, "INSTANCE", className, null, ObjectModelJavaModifier.PRIVATE, ObjectModelJavaModifier.STATIC);
        ObjectModelOperation get = addOperation(aClass, "get", className, ObjectModelJavaModifier.PUBLIC, ObjectModelJavaModifier.STATIC);
        setOperationBody(get, ""+"\n"
+"        return INSTANCE == null ? INSTANCE = new "+className+"() : INSTANCE;\n"
+"    ");
    }

    default void generateClassMapping(boolean useSimpleName, Class<?> superClass, String sourceType, String targetType, Class<?> builderType, String buildMethod, Map<String, String> dtoToReferenceMapping) {

        String className = getModel().getName() + superClass.getSimpleName();
        ObjectModelClass output = createClass(className, superClass.getPackage().getName());
        addImport(output, ImmutableClassMapping.class);
        String initMethod = builderType.getName();
        if (useSimpleName) {

            addImport(output, builderType);
            addImport(output, sourceType);
            addImport(output, targetType);
            sourceType = GeneratorUtil.getSimpleName(sourceType);
            targetType = GeneratorUtil.getSimpleName(targetType);
            initMethod = GeneratorUtil.getSimpleName(initMethod);
        }
        setSuperClass(output, superClass);

        StringBuilder body = new StringBuilder(""+"\n"
+"        super("
        );
        body.append(""+""+initMethod+".<"+sourceType+", "+targetType+">builder()"
        );
        dtoToReferenceMapping.forEach((source, target) -> body.append(""+"\n"
+"                      .put("+source+".class, "+target+")"
        ));
        body.append(""+"\n"
+"                      ."+buildMethod+"());\n"
+"    "
        );
        ObjectModelOperation constructor = addConstructor(output, ObjectModelJavaModifier.PUBLIC);
        setOperationBody(constructor, body.toString());
        addStaticFactory(output, null);
    }

    default Set<String> getInterfaces(ObjectModelClassifier classifier, ObjectModelInterface... interfaces) {
        Set<String> result = new LinkedHashSet<>();
        for (ObjectModelInterface objectModelInterface : interfaces) {
            getInterfaces(classifier, objectModelInterface, result);
        }
        result.remove(classifier.getQualifiedName());
        return result;
    }

    default void getInterfaces(ObjectModelClassifier classifier, ObjectModelInterface anInterface, Set<String> result) {

        for (ObjectModelInterface classifierInterface : classifier.getInterfaces()) {
            if (anInterface.equals(classifierInterface)) {
                result.add(classifier.getQualifiedName());
                result.add(anInterface.getQualifiedName());
                continue;
            }
            if (classifierInterface.getInterfaces() != null) {
                getInterfaces(classifierInterface, anInterface, result);
            }
        }
        if (classifier instanceof ObjectModelClass) {
            ObjectModelClass classifier1 = (ObjectModelClass) classifier;
            classifier1.getSuperclasses().forEach(superclass -> getInterfaces(superclass, anInterface, result));
        }

    }
}
