package fr.ird.observe.toolkit.templates;

/*
 * #%L
 * ObServe Toolkit :: Templates
 * %%
 * Copyright (C) 2017 - 2021 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.auto.service.AutoService;
import com.google.common.collect.ImmutableSet;
import org.codehaus.plexus.component.annotations.Component;
import org.nuiton.eugene.models.extension.tagvalue.TagValueMetadata;
import org.nuiton.eugene.models.extension.tagvalue.TagValueUtil;
import org.nuiton.eugene.models.extension.tagvalue.matcher.EqualsTagValueNameMatcher;
import org.nuiton.eugene.models.extension.tagvalue.provider.DefaultTagValueMetadatasProvider;
import org.nuiton.eugene.models.extension.tagvalue.provider.TagValueMetadatasProvider;
import org.nuiton.eugene.models.object.ObjectModelAttribute;
import org.nuiton.eugene.models.object.ObjectModelClass;
import org.nuiton.eugene.models.object.ObjectModelClassifier;
import org.nuiton.eugene.models.object.ObjectModelPackage;
import org.nuiton.eugene.models.tagvalue.ObjectModelTagValuesStore;

import java.util.Objects;
import java.util.Set;

import static io.ultreia.java4all.i18n.I18n.t;

/**
 * Defines all tag values managed by Java templates.
 *
 * @author Tony Chemit - dev@tchemit.fr
 * @since 4.0
 */
@Component(role = TagValueMetadatasProvider.class, hint = "toolkit")
@AutoService(TagValueMetadatasProvider.class)
public class ToolkitTagValues extends DefaultTagValueMetadatasProvider {

    public ToolkitTagValues() {
        super(Store.values());
    }

    @Override
    public String getDescription() {
        return "Tag values Toolkit";
    }

    /**
     * Obtain the value of the {@link Store#references} tag value on the given classifier.
     *
     * @param classifier classifier to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see Store#references
     * @since 3.0
     */
    public String getReferencesTagValue(ObjectModelClassifier classifier) {
        String tagValue = TagValueUtil.findTagValue(Store.references, classifier);
        return tagValue == null ? null : tagValue.trim();
    }

    /**
     * Obtain the value of the {@link Store#mainDto} tag value on the given classifier.
     *
     * @param classifier classifier to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see Store#mainDto
     * @since 3.0
     */
    public String getMainDtoTagValue(ObjectModelClassifier classifier) {
        String tagValue = TagValueUtil.findTagValue(Store.mainDto, classifier);
        return tagValue == null ? null : tagValue.trim();
    }

    public String getNavigationParentTagValue(ObjectModelClassifier classifier) {
        String tagValue = TagValueUtil.findTagValue(Store.navigationParent, classifier);
        return tagValue == null ? null : tagValue.trim();
    }

    public String getNavigationNodeTypeTagValue(ObjectModelClassifier classifier, ObjectModelAttribute attribute) {
        String tagValue = TagValueUtil.findTagValue(Store.navigationNodeType, attribute, classifier);
        return tagValue == null ? null : tagValue.trim();
    }

    public String getNavigationNodeExtraTypeTagValue(ObjectModelClassifier classifier) {
        String tagValue = TagValueUtil.findTagValue(Store.navigationNodeExtraType, classifier);
        return tagValue == null ? null : tagValue.trim();
    }

    /**
     * Obtain the value of the {@link Store#comparator} tag value on the given classifier.
     *
     * @param classifier classifier to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see Store#comparator
     * @since 3.0
     */
    public String getComparatorTagValue(ObjectModelClassifier classifier) {
        String tagValue = TagValueUtil.findTagValue(Store.comparator, classifier);
        return tagValue == null ? null : tagValue.trim();
    }

    /**
     * Obtain the value of the {@link Store#form} tag value on the given classifier.
     *
     * @param classifier classifier to seek
     * @param aPackage   package to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see Store#form
     * @since 3.0
     */
    public String getFormTagValue(ObjectModelClassifier classifier, ObjectModelPackage aPackage) {
        String tagValue = TagValueUtil.findTagValue(Store.form, classifier, aPackage);
        return tagValue == null ? null : tagValue.trim();
    }

    /**
     * Obtain the value of the {@link Store#packagePriority} tag value on the given package.
     *
     * @param aPackage package to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see Store#packagePriority
     * @since 3.0
     */
    public String getPackagePriorityTagValue(ObjectModelPackage aPackage) {
        String tagValue = TagValueUtil.findTagValue(Store.packagePriority, aPackage);
        return tagValue == null ? null : tagValue.trim();
    }

    public boolean isSkipCopyToEntity(ObjectModelAttribute attribute) {
        return TagValueUtil.findBooleanTagValue(Store.skipCopyToEntity, attribute);
    }

    public boolean isSkipCopyToEntity(ObjectModelTagValuesStore store, ObjectModelClass clazz, ObjectModelAttribute attribute) {
        return store.findAttributeBooleanTagValue(Store.skipCopyToEntity, clazz, attribute);
    }

    public String notSkip(String tagValue) {
        return Objects.equals("skip", tagValue) ? null : tagValue;
    }

    public enum Store implements TagValueMetadata {

        /**
         * To add a prefix on the name of each generated bean class.
         * <p>
         * You can globally use it on the complete model or to a specific classifier.
         *
         * @see #getReferencesTagValue(ObjectModelClassifier)
         * @since 3.0
         */
        references("Pour qualifier les propriété à générer", String.class, null, ObjectModelClassifier.class),

        mainDto("Pour qualifier le dto principal", String.class, null, ObjectModelClassifier.class),

        navigationParent("Pour indiquer quelle est le parent dans la navigation", String.class, null, ObjectModelClassifier.class),

        navigationNodeExtraType("Pour indiquer des nœuds supplémentaires de type SIMPLE au tout au début", String.class, null, ObjectModelClassifier.class),

        navigationNodeType("Pour indiquer quelle est le type de nœud associé dans la navigation", String.class, null, ObjectModelClassifier.class, ObjectModelAttribute.class),

        skipCopyToEntity("Pour indiquer que l'on ne recopie pas d'un dto vers l'entité", String.class, null, ObjectModelAttribute.class),

        comparator("Pour qualifier l'attribut de comparaison", String.class, null, ObjectModelClassifier.class),
        Static("Pour indiquer si une classe est static (ici c'est un hack pour ne pas generer la persistence de celle-ci", Boolean.class, "false", ObjectModelClassifier.class),

        /**
         * To add a prefix on the name of each generated bean class.
         * <p>
         * You can globally use it on the complete model or to a specific classifier.
         *
         * @see #getFormTagValue(ObjectModelClassifier, ObjectModelPackage)
         * @since 3.0
         */
        form("Pour qualifier les propriétés à générer", String.class, null, ObjectModelClassifier.class, ObjectModelPackage.class),
        /**
         * To add priority on packages.
         * <p>
         * You can only use it on a package.
         *
         * @see #getPackagePriorityTagValue(ObjectModelPackage)
         * @since 3.0
         */
        packagePriority("Pour prioriser les paquetages", int.class, null, ObjectModelPackage.class);

        private final Set<Class<?>> targets;
        private final Class<?> type;
        private final String i18nDescriptionKey;
        private final String defaultValue;

        Store(String i18nDescriptionKey, Class<?> type, String defaultValue, Class<?>... targets) {
            this.targets = ImmutableSet.copyOf(targets);
            this.type = type;
            this.i18nDescriptionKey = i18nDescriptionKey;
            this.defaultValue = defaultValue;
        }

        @Override
        public String getName() {
            return name();
        }

        @Override
        public Set<Class<?>> getTargets() {
            return targets;
        }

        @Override
        public Class<?> getType() {
            return type;
        }

        @Override
        public Class<EqualsTagValueNameMatcher> getMatcherClass() {
            return EqualsTagValueNameMatcher.class;
        }

        @Override
        public String getDescription() {
            return t(i18nDescriptionKey);
        }

        @Override
        public String getDefaultValue() {
            return defaultValue;
        }

        @Override
        public boolean isDeprecated() {
            return false;
        }

    }
}
