package fr.ird.observe.toolkit.templates.dto;

/*
 * #%L
 * ObServe Toolkit :: Templates
 * %%
 * Copyright (C) 2017 - 2021 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */




import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import fr.ird.observe.dto.DtoAndReferenceAware;
import fr.ird.observe.dto.IdDto;
import fr.ird.observe.dto.ToolkitDtoIdBean;
import fr.ird.observe.dto.reference.DataDtoReference;
import fr.ird.observe.dto.reference.DataDtoReferenceDefinition;
import fr.ird.observe.dto.reference.DtoReference;
import fr.ird.observe.dto.reference.DtoReferenceDefinition;
import fr.ird.observe.dto.reference.ReferentialDtoReference;
import fr.ird.observe.dto.reference.ReferentialDtoReferenceDefinition;
import fr.ird.observe.dto.reference.ReferentialDtoReferenceWithNoCodeAware;
import fr.ird.observe.dto.referential.ReferentialLocale;
import fr.ird.observe.spi.ProjectPackagesDefinition;
import fr.ird.observe.spi.mapping.DtoToReferenceDtoMapping;
import fr.ird.observe.spi.mapping.ReferenceDtoToDtoClassMapping;
import fr.ird.observe.toolkit.templates.ToolkitTagValues;
import fr.ird.observe.toolkit.templates.TemplateContract;
import io.ultreia.java4all.bean.spi.GenerateJavaBeanDefinition;
import io.ultreia.java4all.classmapping.ImmutableClassMapping;
import io.ultreia.java4all.i18n.spi.builder.I18nKeySet;
import org.nuiton.eugene.EugeneCoreTagValues;
import org.nuiton.eugene.java.BeanTransformerContext;
import org.nuiton.eugene.java.BeanTransformerTagValues;
import org.nuiton.eugene.java.EugeneJavaTagValues;
import org.nuiton.eugene.java.JavaGeneratorUtil;
import org.nuiton.eugene.java.ObjectModelTransformerToJava;
import org.nuiton.eugene.java.extension.ImportsManager;
import org.nuiton.eugene.java.extension.ObjectModelAnnotation;
import org.nuiton.eugene.models.object.ObjectModel;
import org.nuiton.eugene.models.object.ObjectModelAttribute;
import org.nuiton.eugene.models.object.ObjectModelClass;
import org.nuiton.eugene.models.object.ObjectModelClassifier;
import org.nuiton.eugene.models.object.ObjectModelElement;
import org.nuiton.eugene.models.object.ObjectModelJavaModifier;
import org.nuiton.eugene.models.object.ObjectModelModifier;
import org.nuiton.eugene.models.object.ObjectModelOperation;
import org.nuiton.eugene.models.object.ObjectModelPackage;
import org.nuiton.eugene.models.object.ObjectModelParameter;
import org.nuiton.eugene.models.object.xml.ObjectModelAttributeImpl;

import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.TreeMap;
import java.util.function.Function;

/**
 * Generates a reference of a bean.
 * <p>
 * For example:
 * <pre>
 *     GeneratedBoatReference
 *     BoatReference
 * </pre>
 *
 * @author Tony Chemit - dev@tchemit.Fr
 * @since 1.04
 */
@SuppressWarnings({"unused", "StringOperationCanBeSimplified"})
public class DtoReferenceTransformer extends ObjectModelTransformerToJava implements TemplateContract {

    private final EugeneCoreTagValues coreTagValues;
    private final EugeneJavaTagValues javaTemplatesTagValues;
    private final BeanTransformerTagValues beanTagValues;
    private final ToolkitTagValues observeTagValues;

    private BeanTransformerContext context;

    static Map<ObjectModelAttribute, ObjectModelAttribute> getReferenceProperties(ImmutableList<String> selectedClassesFqn, ObjectModelClass input, Set<String> availableProperties, Function<String, String> classMapping) {
        Collection<ObjectModelAttribute> attributes = new LinkedList<>(input.getAttributes());
        attributes.addAll(input.getAllOtherAttributes());
        Map<String, ObjectModelAttribute> properties = new TreeMap<>();
        Map<ObjectModelAttribute, ObjectModelAttribute> resultMap = new LinkedHashMap<>();
        for (ObjectModelAttribute attr : attributes) {

            if (attr.isNavigable()) {

                String attrName = attr.getName();
                String type = attr.getType();

                if (selectedClassesFqn.contains(ProjectPackagesDefinition.cleanType(type))) {

                    // Dto

                    if (availableProperties.contains(attrName + "Label")) {
                        ObjectModelAttributeImpl newAttr = new ObjectModelAttributeImpl();
                        newAttr.setName(attrName + "Label");
                        newAttr.setType("String");
                        // only keep navigable attributes
                        properties.put(newAttr.getName(), newAttr);
                        resultMap.put(newAttr, attr);
                    }

                    if (availableProperties.contains(attrName + "Id")) {
                        ObjectModelAttributeImpl newAttr = new ObjectModelAttributeImpl();
                        newAttr.setName(attrName + "Id");
                        newAttr.setType("String");
                        // only keep navigable attributes
                        properties.put(newAttr.getName(), newAttr);
                        resultMap.put(newAttr, attr);
                    }
                    if (availableProperties.contains(attrName + "Code")) {
                        ObjectModelAttributeImpl newAttr = new ObjectModelAttributeImpl();
                        newAttr.setName(attrName + "Code");
                        newAttr.setType("String");
                        // only keep navigable attributes
                        properties.put(newAttr.getName(), newAttr);
                        resultMap.put(newAttr, attr);
                    }

                    if (availableProperties.contains(attrName + "Size")) {
                        ObjectModelAttributeImpl newAttr = new ObjectModelAttributeImpl();
                        newAttr.setName(attrName + "Size");
                        newAttr.setType("int");
                        // only keep navigable attributes
                        properties.put(newAttr.getName(), newAttr);
                        resultMap.put(newAttr, attr);
                    }

                    // Get a ref

                    if (availableProperties.contains(attrName)) {
                        ObjectModelAttributeImpl newAttr = new ObjectModelAttributeImpl();
                        newAttr.setName(attrName);
                        if (!type.endsWith("Reference")) {
                            // not a ref, get a ref
                            type = ProjectPackagesDefinition.cleanType(type) + "Reference";
                        }
                        newAttr.setType(classMapping.apply(type));
                        properties.put(attrName, newAttr);
                        resultMap.put(newAttr, attr);
                    }
                    continue;
                }

                if (availableProperties.contains(attrName + "Size")) {
                    ObjectModelAttributeImpl newAttr = new ObjectModelAttributeImpl();
                    newAttr.setName(attrName + "Size");
                    newAttr.setType("int");
                    // only keep navigable attributes
                    properties.put(newAttr.getName(), newAttr);
                    resultMap.put(newAttr, attr);
                    continue;
                }

                // Simple type

                if (availableProperties.contains(attrName)) {
                    ObjectModelAttributeImpl newAttr = new ObjectModelAttributeImpl();
                    newAttr.setName(attrName);
                    newAttr.setType(type);
                    properties.put(attrName, newAttr);
                    resultMap.put(newAttr, attr);
                }

            }
        }
        if (availableProperties.contains("label") && properties.values().stream().noneMatch(p -> p.getName().equals("label"))) {

            ObjectModelAttributeImpl attr = new ObjectModelAttributeImpl();
            attr.setName("label");
            attr.setType("String");
            properties.put("label", attr);
        }
        Map<ObjectModelAttribute, ObjectModelAttribute> result = new LinkedHashMap<>();
        for (String availableProperty : availableProperties) {
            ObjectModelAttribute key = properties.get(availableProperty);
            ObjectModelAttribute value;
            if (key == null) {
                ObjectModelAttributeImpl ee = new ObjectModelAttributeImpl();
                ee.setName(availableProperty);
                ee.setType("String");
                key = ee;
                value = null;
            } else {
                value = resultMap.get(key);
            }
            Objects.requireNonNull(key, "Cant' find property " + availableProperty + " on " + input.getQualifiedName());
            result.put(key, value);
        }
        return result;
    }

    public static String getAttributeType(ObjectModel model, BeanTransformerContext context, String attrType) {
        if (!JavaGeneratorUtil.isPrimitiveType(attrType)) {
            boolean hasClass = model.hasClass(attrType);
            if (hasClass) {
                ObjectModelClass attributeClass = model.getClass(attrType);
                String attributeType = context.classesNameTranslation.get(attributeClass);
                if (attributeType != null) {
                    attrType = attributeClass.getPackageName() + "." + attributeType;
                }
            }
        }
        return attrType;
    }

    public DtoReferenceTransformer() {
        coreTagValues = new EugeneCoreTagValues();
        javaTemplatesTagValues = new EugeneJavaTagValues();
        beanTagValues = new BeanTransformerTagValues();
        observeTagValues = new ToolkitTagValues();
    }

    private void generateClass(ObjectModelClass input, String className, String abstractClassName, String dtoName, String generatedDtoName, Set<ObjectModelAttribute> properties, boolean referential) {

        ObjectModelClass output = createClass(className, input.getPackageName());
        addAnnotation(output, output, GenerateJavaBeanDefinition.class);
        setSuperClass(output, abstractClassName);

        getLog().debug("will generate " + output.getQualifiedName());

        String importReferentialLocale = importAndSimplify(output, ReferentialLocale.class.getName());

        ObjectModelOperation constructor = addConstructor(output, ObjectModelJavaModifier.PUBLIC);
        addParameter(constructor, importReferentialLocale, "referentialLocale");
        addParameter(constructor, dtoName, "dto");
        setOperationBody(constructor, ""+"\n"
+"        super(referentialLocale, dto);\n"
+"    ");
        ObjectModelOperation constructor2 = addConstructor(output, ObjectModelJavaModifier.PUBLIC);
        addParameter(constructor2, importReferentialLocale, "referentialLocale");
        addParameter(constructor2, generatedDtoName, "dto");
        setOperationBody(constructor2, ""+"\n"
+"        super(referentialLocale, dto);\n"
+"    ");
        ObjectModelOperation emptyConstructor = addConstructor(output, ObjectModelJavaModifier.PUBLIC);
        addParameter(emptyConstructor, importReferentialLocale, "referentialLocale");
        setOperationBody(emptyConstructor, ""+"\n"
+"        super(referentialLocale);\n"
+"    ");
    }

    private void generateGeneratedClass(ObjectModelClass input, String className, String abstractClassName, String dtoName, String generatedDtoName, String superClass, Set<ObjectModelAttribute> properties, Map<ObjectModelAttribute, ObjectModelAttribute> binderProperties, boolean referential, Set<String> extractInterfaces) {
        ObjectModelClass output = createAbstractClass(abstractClassName, input.getPackageName());
        ImportsManager importManager = getImportManager(output);
        importManager.addExcludedPattern(".+\\." + dtoName);
        importManager.addExcludedPattern(".+\\." + className);
        setSuperClass(output, superClass);
        for (String extractInterface : extractInterfaces) {
            addInterface(output, extractInterface);
        }

        @SuppressWarnings("rawtypes") Class builderMethodName = referential ? ReferentialDtoReferenceDefinition.class : DataDtoReferenceDefinition.class;
        addImport(output, builderMethodName);
        StringBuilder definition = new StringBuilder("" +"\n"
+"            "+builderMethodName.getSimpleName()+"\n"
+"                    .builder("+dtoName+".class, "+className+".class) "
        );
        for (ObjectModelAttribute attr : properties) {
            String type = importManager.importAndSimplify(attr.getType());
            String name = attr.getName();
            String getterName;
            boolean booleanProperty = JavaGeneratorUtil.isBooleanPrimitive(attr);

            definition.append("" +"\n"
+"                    .addProperty("+type+".class, \""+name+"\")"
            );
        }
        definition.append(""+"\n"
+"                    .build()"
        );
        String definitionType = String.format("%s<%s, %s>", builderMethodName.getName(), dtoName, className);
        addAttribute(output, "DEFINITION", definitionType, definition.toString(), ObjectModelJavaModifier.PUBLIC, ObjectModelJavaModifier.STATIC, ObjectModelJavaModifier.FINAL);

        getLog().debug("will generate " + output.getQualifiedName());
        String importReferentialLocale = importAndSimplify(output, ReferentialLocale.class.getName());

        ObjectModelOperation emptyConstructor = addConstructor(output, ObjectModelJavaModifier.PROTECTED);
        addParameter(emptyConstructor, importReferentialLocale, "referentialLocale");
        setOperationBody(emptyConstructor, ""+"\n"
+"        this(referentialLocale, new "+dtoName+"());\n"
+"    ");
        ObjectModelOperation constructorFromDto = addConstructor(output, ObjectModelJavaModifier.PROTECTED);
        addParameter(constructorFromDto, importReferentialLocale, "referentialLocale");
        addParameter(constructorFromDto, dtoName, "dto");
        setOperationBody(constructorFromDto, ""+"\n"
+"        this(referentialLocale, ("+generatedDtoName+") dto);\n"
+"    ");
        ObjectModelOperation constructorFromGeneratedDto = addConstructor(output, ObjectModelJavaModifier.PROTECTED);
        addParameter(constructorFromGeneratedDto, importReferentialLocale, "referentialLocale");
        addParameter(constructorFromGeneratedDto, generatedDtoName, "dto");

        StringBuilder body = new StringBuilder(""+"\n"
+"        super(dto);");
        for (Map.Entry<ObjectModelAttribute, ObjectModelAttribute> entry : binderProperties.entrySet()) {
            ObjectModelAttribute attr = entry.getKey();
            createProperty(output, attr);
            ObjectModelAttribute dtoAttr = entry.getValue();
            String name = attr.getName();
            String assignment = getReferencePropertyAssignment(output, dtoName, dtoAttr, attr);
            body.append(""+"\n"
+"        this."+name+" = "+assignment+";"
            );
        }
        body.append(""+"\n"
+"    ");
        setOperationBody(constructorFromGeneratedDto, body.toString());

        ObjectModelOperation getDefinition = addOperation(output, "getDefinition", definitionType, ObjectModelJavaModifier.PUBLIC);
        addAnnotation(output, getDefinition, Override.class);
        setOperationBody(getDefinition, ""+"\n"
+"        return DEFINITION;\n"
+"    "
        );
        ObjectModelOperation getDtoType = addOperation(output, "getDtoType", "Class<" + dtoName + ">", ObjectModelJavaModifier.PUBLIC);
        addAnnotation(output, getDtoType, Override.class);
        setOperationBody(getDtoType, ""+"\n"
+"        return "+dtoName+".class;\n"
+"    "
        );
        ObjectModelOperation getReferenceType = addOperation(output, "getReferenceType", "Class<" + className + ">", ObjectModelJavaModifier.PUBLIC);
        addAnnotation(output, getReferenceType, Override.class);
        setOperationBody(getReferenceType, ""+"\n"
+"        return "+className+".class;\n"
+"    "
        );
        addImport(output, ToolkitDtoIdBean.class);
        ObjectModelOperation toShortReference = addOperation(output, "toShortReference", ToolkitDtoIdBean.class.getSimpleName(), ObjectModelJavaModifier.PUBLIC);
        addAnnotation(output, toShortReference, Override.class);
        setOperationBody(toShortReference, ""+"\n"
+"        return ToolkitDtoIdBean.of("+className+".class, getId(), getLastUpdateDate());\n"
+"    "
        );
        ObjectModelOperation toShortDto = addOperation(output, "toShortDto", ToolkitDtoIdBean.class.getSimpleName(), ObjectModelJavaModifier.PUBLIC);
        addAnnotation(output, toShortDto, Override.class);
        setOperationBody(toShortDto, ""+"\n"
+"        return ToolkitDtoIdBean.of("+dtoName+".class, getId(), getLastUpdateDate());\n"
+"    "
        );
    }

    private String getReferencePropertyAssignment(ObjectModelClass output, String dtoName, ObjectModelAttribute dtoAttr, ObjectModelAttribute attr) {
        String content;
        if (dtoAttr != null) {
            String getterName;
            String dtoAttrName = dtoAttr.getName();
            String type = getAttributeType(dtoAttr);
            boolean isSizeMethod = attr.getName().equals(dtoAttrName + "Size");
            boolean booleanProperty = JavaGeneratorUtil.isBooleanPrimitive(dtoAttr);

            if (booleanProperty) {
                getterName = JavaGeneratorUtil.OPERATION_GETTER_BOOLEAN_PREFIX;
            } else {
                getterName = JavaGeneratorUtil.OPERATION_GETTER_DEFAULT_PREFIX;
            }
            getterName += JavaGeneratorUtil.capitalizeJavaBeanPropertyName(dtoAttrName);
            if (isSizeMethod) {
                getterName += "Size";
            }
            content = "dto." + getterName + "()";

            if (isSizeMethod) {
                return content;
            }
            if (context.selectedClassesFqn.contains(ProjectPackagesDefinition.cleanType(dtoAttr.getType()))) {

                // Dto

                if (!type.endsWith("Reference")) {

                    // not a ref, get a ref
                    type = ProjectPackagesDefinition.cleanType(type) + "Reference";
                }

                if (attr.getName().equals(dtoAttrName + "Id")) {
                    addImport(output, Optional.class);
                    addImport(output, IdDto.class);
                    type = importAndSimplify(output, type);
                    content = String.format("Optional.ofNullable(%s).map(%s::getId).orElse(null)", content, type);
                } else if (attr.getName().equals(dtoAttrName + "Label")) {
                    addImport(output, Optional.class);
                    addImport(output, IdDto.class);
                    String method = "getLabel";
                    if (type.equals("SpeciesReference")) {
                        method = "getScientificLabel";
                    }
                    type = importAndSimplify(output, type);
                    content = String.format("Optional.ofNullable(%s).map(%s::%s).orElse(null)", content, type, method);
                } else if (attr.getName().equals(dtoAttrName + "Code")) {
                    addImport(output, Optional.class);
                    addImport(output, IdDto.class);
                    type = importAndSimplify(output, type);
                    content = String.format("Optional.ofNullable(%s).map(%s::getCode).orElse(null)", content, type);
                }
            }
        } else {

            if (attr.getName().equals("label")) {
                content = "dto.getLabel(referentialLocale)";
            } else {
                throw new IllegalStateException(String.format("Can't come here (class: %s, property: %s)", output.getQualifiedName(), attr));
            }
        }
        return content;
    }

    private String getAttributeType(ObjectModelAttribute attr) {
        String attrType = attr.getType();
        return getAttributeType(attrType);
    }

    @Override
    public void transformFromModel(ObjectModel model) {
        super.transformFromModel(model);

        context = new BeanTransformerContext(model, coreTagValues, javaTemplatesTagValues, beanTagValues, false, false, input -> {

            ObjectModelPackage aPackage = model.getPackage(input.getPackageName());

            boolean referential = TemplateContract.isReferentialFromPackageName(aPackage.getName());

            String referencesTagValue = observeTagValues.getReferencesTagValue(input);
            return referencesTagValue != null || referential;
        }, getLog());

        context.report();

        I18nKeySet i18nGetterFile = getConfiguration().getI18nGetterFile();

        Map<String, String> dtoToReferenceMapping = new TreeMap<>();
        Map<String, String> referenceToDtoMapping = new TreeMap<>();
        for (ObjectModelClass input : context.selectedClasses) {

            ObjectModelPackage aPackage = getPackage(input);

            String packageName = aPackage.getName();

            boolean referential = TemplateContract.isReferentialFromPackageName(packageName);
            String referencesTagValue = observeTagValues.getReferencesTagValue(input);
            if (referencesTagValue == null && referential) {
                referencesTagValue = "code,label,uri";
            }

            Set<String> availableProperties = new LinkedHashSet<>(Arrays.asList(Objects.requireNonNull(referencesTagValue).split(",")));

            Map<ObjectModelAttribute, ObjectModelAttribute> binderProperties = DtoReferenceTransformer.getReferenceProperties(context.selectedClassesFqn, input, availableProperties, this::getAttributeType);

            Set<ObjectModelAttribute> properties = getProperties(model, context, input, availableProperties);

            String prefix = getConstantPrefix(input);
            setConstantPrefix(prefix);

            String dtoName = context.classesNameTranslation.get(input);
            String generatedDtoName = "Generated" + dtoName;
            i18nGetterFile.addKey("observe." + TemplateContract.cleanDtoType(referential, packageName + "." + dtoName) + ".type");
            String className = ProjectPackagesDefinition.cleanType(dtoName) + "Reference";
            String generatedClassName = "Generated" + className;

            dtoToReferenceMapping.put(packageName + "." + dtoName, packageName + "." + className + ".DEFINITION");
            referenceToDtoMapping.put(packageName + "." + className, packageName + "." + dtoName + ".class");

            Class<?> superClass = referential ? ReferentialDtoReference.class : DataDtoReference.class;

            Set<String> extractInterfaces = getInterfaces(input,
                                                          model.getInterface(DtoAndReferenceAware.class.getName()),
                                                          model.getInterface(ReferentialDtoReferenceWithNoCodeAware.class.getName()));

            generateGeneratedClass(input, className, generatedClassName, dtoName, generatedDtoName, superClass.getName(), properties, binderProperties, referential, extractInterfaces);

            boolean generateClass = notFoundInClassPath(input.getPackageName(), className);
            if (generateClass) {
                generateClass(input, className, generatedClassName, dtoName, generatedDtoName, properties, referential);
            }
        }

        generateClassMapping(true, DtoToReferenceDtoMapping.class, "Class<? extends " + IdDto.class.getName() + ">", DtoReferenceDefinition.class.getName() + "<?,?>", ImmutableMap.class, "build", dtoToReferenceMapping);
        generateClassMapping(true, ReferenceDtoToDtoClassMapping.class, DtoReference.class.getName(), IdDto.class.getName(), ImmutableClassMapping.class, "getMappingBuilder", referenceToDtoMapping);

    }

    private String getAttributeType(String attrType) {
        return getAttributeType(model, context, attrType);
    }

    private boolean notFoundInClassPath(String input, String className) {
        String fqn = input + "." + className;
        boolean inClassPath = getResourcesHelper().isJavaFileInClassPath(fqn);
        return !inClassPath;
    }

    private void createProperty(ObjectModelClass output, ObjectModelAttribute attr) {

        String attrName = attr.getName();
        String attrType = getAttributeTypeWithGeneric(attr);

        boolean booleanProperty = JavaGeneratorUtil.isBooleanPrimitive(attr);
        if (booleanProperty) {
            createGetMethod(output, attrName, attrType, JavaGeneratorUtil.OPERATION_GETTER_BOOLEAN_PREFIX);
        } else {
            createGetMethod(output, attrName, attrType, JavaGeneratorUtil.OPERATION_GETTER_DEFAULT_PREFIX);
        }
        addAttribute(output, attrName, attrType, "", ObjectModelJavaModifier.PRIVATE, ObjectModelJavaModifier.FINAL);

    }

    private Set<ObjectModelAttribute> getProperties(ObjectModel model, BeanTransformerContext context, ObjectModelClass input, Set<String> availableProperties) {
        Collection<ObjectModelAttribute> attributes = new LinkedList<>(input.getAttributes());
        attributes.addAll(input.getAllOtherAttributes());
        Map<String, ObjectModelAttribute> properties = new TreeMap<>();
        for (ObjectModelAttribute attr : attributes) {

            if (attr.isNavigable()) {

                String attrName = attr.getName();
                String type = attr.getType();

                if (context.selectedClassesFqn.contains(ProjectPackagesDefinition.cleanType(type))) {

                    // Dto

                    if (availableProperties.contains(attrName + "Label")) {
                        ObjectModelAttributeImpl newAttr = new ObjectModelAttributeImpl();
                        newAttr.setName(attrName + "Label");
                        newAttr.setType("String");
                        // only keep navigable attributes
                        properties.put(newAttr.getName(), newAttr);
                    }

                    if (availableProperties.contains(attrName + "Id")) {
                        ObjectModelAttributeImpl newAttr = new ObjectModelAttributeImpl();
                        newAttr.setName(attrName + "Id");
                        newAttr.setType("String");
                        // only keep navigable attributes
                        properties.put(newAttr.getName(), newAttr);
                    }
                    if (availableProperties.contains(attrName + "Code")) {
                        ObjectModelAttributeImpl newAttr = new ObjectModelAttributeImpl();
                        newAttr.setName(attrName + "Code");
                        newAttr.setType("String");
                        // only keep navigable attributes
                        properties.put(newAttr.getName(), newAttr);
                    }
                    if (availableProperties.contains(attrName + "Size")) {
                        ObjectModelAttributeImpl newAttr = new ObjectModelAttributeImpl();
                        newAttr.setName(attrName + "Size");
                        newAttr.setType("int");
                        // only keep navigable attributes
                        properties.put(newAttr.getName(), newAttr);
                        continue;
                    }

                    // Get a ref

                    if (availableProperties.contains(attrName)) {

                        ObjectModelAttributeImpl newAttr = new ObjectModelAttributeImpl();

                        newAttr.setName(attrName);

                        if (!type.endsWith("Reference")) {

                            // not a ref, get a ref
                            type = ProjectPackagesDefinition.cleanType(type) + "Reference";
                        }

                        newAttr.setType(getAttributeType(model, context, type));

                        properties.put(attrName, newAttr);
                    }

                    continue;
                }

                if (availableProperties.contains(attrName + "Size")) {
                    ObjectModelAttributeImpl newAttr = new ObjectModelAttributeImpl();
                    newAttr.setName(attrName + "Size");
                    newAttr.setType("int");
                    // only keep navigable attributes
                    properties.put(newAttr.getName(), newAttr);
                    continue;
                }

                // Simple type

                if (availableProperties.contains(attrName)) {
                    ObjectModelAttributeImpl newAttr = new ObjectModelAttributeImpl();
                    newAttr.setName(attrName);
                    newAttr.setType(type);
                    properties.put(attrName, newAttr);
                }

            }
        }
        if (availableProperties.contains("label") && properties.values().stream().noneMatch(p -> p.getName().equals("label"))) {

            ObjectModelAttributeImpl attr = new ObjectModelAttributeImpl();
            attr.setName("label");
            attr.setType("String");
            properties.put("label", attr);
        }
        Set<ObjectModelAttribute> result = new LinkedHashSet<>();
        for (String availableProperty : availableProperties) {
            ObjectModelAttribute e = properties.get(availableProperty);
            if (e == null) {
                ObjectModelAttributeImpl ee = new ObjectModelAttributeImpl();
                ee.setName(availableProperty);
                ee.setType("String");
                e = ee;
            }
            Objects.requireNonNull(e, "Cant' find property " + availableProperty + " on " + input.getQualifiedName());
            result.add(e);
        }
        return result;
    }

    private void createGetMethod(ObjectModelClass output, String attrName, String attrType, String methodPrefix) {
        ObjectModelOperation operation = addOperation(
                output,
                getJavaBeanMethodName(methodPrefix, attrName),
                attrType,
                ObjectModelJavaModifier.PUBLIC
        );
        setOperationBody(operation, ""
    +"\n"
+"        return "+attrName+";\n"
+"    "
        );
    }

    private String getAttributeTypeWithGeneric(ObjectModelAttribute attr) {
        String attrType = getAttributeType(attr);
        String generic = eugeneTagValues.getAttributeGenericTagValue(attr);
        if (generic != null) {
            attrType += "<" + getAttributeType(generic) + ">";
        }
        return attrType;
    }

    @Override
    public ObjectModelClass createClass(String className, String packageName) {
        return super.createClass(className, packageName);
    }

    @Override
    public void addImport(ObjectModelClass output, Class<?> type) {
        super.addImport(output, type);
    }

    @Override
    public void addImport(ObjectModelClass output, String type) {
        super.addImport(output, type);
    }

    @Override
    public void setSuperClass(ObjectModelClass output, Class<?> superClass) {
        super.setSuperClass(output, superClass);
    }

    @Override
    public ObjectModelAnnotation addAnnotation(ObjectModelClass output, ObjectModelElement output1, Class<?> annotationType) {
        return super.addAnnotation(output, output1, annotationType);
    }

    @Override
    public void addAnnotationParameter(ObjectModelClass output, ObjectModelAnnotation annotation, String name, String value) {
        super.addAnnotationParameter(output, annotation, name, value);
    }

    @Override
    public ObjectModelOperation addConstructor(ObjectModelClass output, ObjectModelJavaModifier modifiers) {
        return super.addConstructor(output, modifiers);
    }

    @Override
    public void setOperationBody(ObjectModelOperation constructor, String body) {
        super.setOperationBody(constructor, body);
    }

    @Override
    public ObjectModelParameter addParameter(ObjectModelOperation operation, Class<?> type, String name) {
        return super.addParameter(operation, type, name);
    }

    public ObjectModelOperation addOperation(ObjectModelClassifier classifier, String name, String type, ObjectModelModifier... modifiers) {
        return super.addOperation(classifier, name, type, modifiers);
    }

    @Override
    public ObjectModelAttribute addAttribute(ObjectModelClassifier classifier, String name, String type, String value, ObjectModelModifier... modifiers) {
        return super.addAttribute(classifier, name, type, value, modifiers);
    }
}
