package fr.ird.observe.toolkit.templates.dto;

/*-
 * #%L
 * ObServe Toolkit :: Templates
 * %%
 * Copyright (C) 2017 - 2021 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.ImmutableMap;
import fr.ird.observe.dto.DtoToReference;
import fr.ird.observe.dto.IdDto;
import fr.ird.observe.dto.ToolkitDtoIdBean;
import fr.ird.observe.dto.data.ContainerDto;
import fr.ird.observe.dto.form.FormDefinition;
import fr.ird.observe.dto.referential.ReferentialLocale;
import fr.ird.observe.spi.ProjectPackagesDefinition;
import fr.ird.observe.spi.mapping.DtoToFormDtoMapping;
import fr.ird.observe.spi.mapping.DtoToMainDtoClassMapping;
import fr.ird.observe.toolkit.templates.ToolkitTagValues;
import fr.ird.observe.toolkit.templates.TemplateContract;
import io.ultreia.java4all.classmapping.ImmutableClassMapping;
import org.nuiton.eugene.EugeneCoreTagValues;
import org.nuiton.eugene.GeneratorUtil;
import org.nuiton.eugene.java.BeanTransformer;
import org.nuiton.eugene.java.BeanTransformerContext;
import org.nuiton.eugene.java.BeanTransformerTagValues;
import org.nuiton.eugene.java.EugeneJavaTagValues;
import org.nuiton.eugene.java.JavaBuilder;
import org.nuiton.eugene.java.JavaGeneratorUtil;
import org.nuiton.eugene.java.extension.ObjectModelAnnotation;
import org.nuiton.eugene.models.object.ObjectModel;
import org.nuiton.eugene.models.object.ObjectModelAttribute;
import org.nuiton.eugene.models.object.ObjectModelClass;
import org.nuiton.eugene.models.object.ObjectModelClassifier;
import org.nuiton.eugene.models.object.ObjectModelElement;
import org.nuiton.eugene.models.object.ObjectModelJavaModifier;
import org.nuiton.eugene.models.object.ObjectModelModifier;
import org.nuiton.eugene.models.object.ObjectModelOperation;
import org.nuiton.eugene.models.object.ObjectModelPackage;
import org.nuiton.eugene.models.object.ObjectModelParameter;
import org.nuiton.eugene.models.object.xml.ObjectModelAttributeImpl;
import org.nuiton.eugene.models.object.xml.ObjectModelClassifierImpl;
import org.nuiton.eugene.models.object.xml.ObjectModelOperationImpl;
import org.nuiton.eugene.models.object.xml.ObjectModelParameterImpl;

import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.TreeMap;

import static fr.ird.observe.toolkit.templates.TemplateContract.isReferentialFromPackageName;





/**
 * @author Tony Chemit - dev@tchemit.fr
 * @since ?
 */
@SuppressWarnings({"unused", "StringOperationCanBeSimplified", "UnusedAssignment"})
public class DtoTransformer extends BeanTransformer implements TemplateContract {

    private final EugeneCoreTagValues coreTagValues;
    private final EugeneJavaTagValues javaTemplatesTagValues;
    private final BeanTransformerTagValues beanTagValues;
    private final ToolkitTagValues observeTagValues;
    private final Map<String, String> dtoFormMapping = new TreeMap<>();
    private final Map<String, String> dtoMainMapping = new TreeMap<>();
    private BeanTransformerContext context;

    public DtoTransformer() {
        coreTagValues = new EugeneCoreTagValues();
        javaTemplatesTagValues = new EugeneJavaTagValues();
        beanTagValues = new BeanTransformerTagValues();
        observeTagValues = new ToolkitTagValues();
    }

    @Override
    public void transformFromModel(ObjectModel model) {
        super.transformFromModel(model);

        context = new BeanTransformerContext(model, coreTagValues, javaTemplatesTagValues, beanTagValues, false, false, input -> {

            ObjectModelPackage aPackage = model.getPackage(input.getPackageName());

            boolean referential = isReferentialFromPackageName(aPackage.getName());

            String referencesTagValue = observeTagValues.getReferencesTagValue(input);
            return referencesTagValue != null || referential;
        }, getLog());

        context.report();

    }

    @Override
    protected void generateInitializers() {
        // do not generate global model initializer
    }

    @Override
    protected void debugOutputModel() {
        super.debugOutputModel();

        generateClassMapping(true, DtoToFormDtoMapping.class, "Class<? extends " + IdDto.class.getName() + ">", FormDefinition.class.getName() + "<?>", ImmutableMap.class, "build", dtoFormMapping);
        generateClassMapping(true, DtoToMainDtoClassMapping.class, IdDto.class.getName(), IdDto.class.getName(), ImmutableClassMapping.class, "getMappingBuilder", dtoMainMapping);

    }

    @Override
    protected boolean canGenerateAbstractClass(ObjectModelClass aClass, String abstractClassName) {
        boolean result = super.canGenerateAbstractClass(aClass, abstractClassName);
        if (result) {
            result = !EugeneCoreTagValues.isSkip(aClass, model.getPackage(aClass));
        }
        return result;
    }

    @Override
    protected ObjectModelClass generateGeneratedClass(ObjectModelPackage aPackage, ObjectModelClass input, String className, String mainClassName) {
        ObjectModelClass output = super.generateGeneratedClass(aPackage, input, className, mainClassName);
        ObjectModelClass firstSuperClass = output.getSuperclasses().iterator().next();
        if (firstSuperClass.getName().equals(ContainerDto.class.getSimpleName())) {
            ObjectModelAttribute mainAttribute = input.getAttributes().iterator().next();
            if (mainAttribute == null) {
                throw new IllegalStateException(String.format("Can't find main attribute on %s", input.getName()));
            }
            setSuperClass(output, String.format("%s<%sDto>", firstSuperClass.getQualifiedName(), mainAttribute.getType()));
            String propertyNameValue = getConstantName(mainAttribute.getName());
            if (propertyNameValue != null) {
                ObjectModelOperation constructor = addConstructor(output, ObjectModelJavaModifier.PUBLIC);
                setOperationBody(constructor, ""+"\n"
+"        super("+propertyNameValue+");\n"
+"    ");
            }
        }
        boolean referential = TemplateContract.isReferentialFromPackageName(aPackage.getName());
        boolean notAbstract = !input.isAbstract();
        if (notAbstract) {
            addStaticFactoryMethod(output, mainClassName);
        }
        if (context.selectedClasses.contains(input)) {

            String referencesTagValue = observeTagValues.getReferencesTagValue(input);
            if (referencesTagValue == null && referential) {
                referencesTagValue = "code,label,uri";
            }

            Objects.requireNonNull(referencesTagValue);

            Set<String> availableProperties = new LinkedHashSet<>(Arrays.asList(referencesTagValue.split(",")));

            Map<ObjectModelAttribute, ObjectModelAttribute> binderProperties = DtoReferenceTransformer.getReferenceProperties(context.selectedClassesFqn, input, availableProperties, this::getAttributeType);

            String dtoName = context.classesNameTranslation.get(input);
            String referenceName = ProjectPackagesDefinition.cleanType(dtoName) + "Reference";

            addInterface(output, String.format("%s<%s>", DtoToReference.class.getName(), referenceName));
            addImport(output, ToolkitDtoIdBean.class);
            ObjectModelOperation toShortDto = addOperation(output, "toShortDto", ToolkitDtoIdBean.class.getSimpleName(), ObjectModelJavaModifier.PUBLIC);
            addAnnotation(output, toShortDto, Override.class);
            setOperationBody(toShortDto, ""+"\n"
+"        return ToolkitDtoIdBean.of("+dtoName+".class, getId(), getLastUpdateDate());\n"
+"    "
            );
            addToReferenceMethod(output, binderProperties, referenceName);
        }

        if (notAbstract) {
            addFormDefinitionAttribute(input, output, referential);
            addMainDtoMapping(input, referential);
        }
        return output;
    }

    private void addStaticFactoryMethod(ObjectModelClass output, String className) {
        ObjectModelOperation operation = addOperation(output, "newDto", className, ObjectModelJavaModifier.PUBLIC, ObjectModelJavaModifier.STATIC);
        addParameter(operation, importAndSimplify(output, Date.class.getName()), "createDate");
        setOperationBody(operation, ""+"\n"
+"        return newDto("+className+".class, createDate);\n"
+"    ");
    }

    private void addMainDtoMapping(ObjectModelClass input, boolean referential) {
        String mainDtoType = observeTagValues.getMainDtoTagValue(input);
        if (observeTagValues.notSkip(mainDtoType) == null) {
            return;
        }
        String dtoType = input.getPackageName() + "." + context.classesNameTranslation.get(input);
        dtoMainMapping.put(dtoType, mainDtoType + ".class");
    }

    private void addFormDefinitionAttribute(ObjectModelClass input, ObjectModelClass output, boolean referential) {
        ObjectModelPackage thisPackage = getPackage(input);

        String formTagValue = observeTagValues.getFormTagValue(input, thisPackage);
        getLog().debug("FormTagValue: " + formTagValue);

        ObjectModelPackage defaultPackage = getModel().getPackage(getDefaultPackageName());

        String packageName = defaultPackage.getName() + ".";

        String formType;
        String dtoType = thisPackage.getName() + "." + context.classesNameTranslation.get(input);
        String dtoTypeSimpleName = GeneratorUtil.getSimpleName(thisPackage.getName() + "." + context.classesNameTranslation.get(input));

        boolean addForm;

        Map<String, String> properties;
        if (formTagValue.equals(dtoType)) {

            formType = dtoType;
            properties = getFormProperties(input, output);

            getLog().debug(String.format("form: %s, found %d properties.", formType, properties.size()));

            addForm = referential || !properties.isEmpty();
            if (addForm) {
                StringBuilder bodyBuilder = new StringBuilder();
                bodyBuilder.append(""+"\n"
+"        new FormDefinition.Builder<>("+dtoTypeSimpleName+".class)");
                if (referential) {
                    String referenceType = dtoTypeSimpleName.replace("Dto", "Reference");
                    bodyBuilder.append(""+"\n"
+"                .addProperty(FormDefinition.REFERENTIAL_LIST_HEADER, "+referenceType+".DEFINITION)");
                }

                for (Map.Entry<String, String> entry : properties.entrySet()) {
                    String propertyName = entry.getKey();
                    String referenceType = entry.getValue();
                    propertyName = JavaGeneratorUtil.convertVariableNameToConstantName("property" + JavaGeneratorUtil.capitalizeJavaBeanPropertyName(propertyName));
                    bodyBuilder.append(""+"\n"
+"                .addProperty("+propertyName+", "+referenceType+".DEFINITION)");
                }

                bodyBuilder.append(""+"\n"
+"                .build()");

                addImport(output, formType);
                addAttribute(output, "FORM_DEFINITION", "FormDefinition<" + GeneratorUtil.getSimpleName(formType) + ">", bodyBuilder.toString(), ObjectModelJavaModifier.STATIC, ObjectModelJavaModifier.FINAL, ObjectModelJavaModifier.PUBLIC);
            }
        } else {
            formType = formTagValue;
            addForm = true;
        }

        if (addForm) {
            addImport(output, FormDefinition.class);
            dtoFormMapping.put(dtoType, formType + ".FORM_DEFINITION");
        }
    }

    private void addToReferenceMethod(ObjectModelClass output, Map<ObjectModelAttribute, ObjectModelAttribute> properties, String referenceName) {

        boolean useRelativeName = context.useRelativeName;

        ObjectModelOperation operation = addOperation(output, "toReference", referenceName, ObjectModelJavaModifier.PUBLIC);
        addAnnotation(output, operation, Override.class);
        String importReferentialLocale = importAndSimplify(output, ReferentialLocale.class.getName());
        addParameter(operation, importReferentialLocale, "referentialLocale");

        setOperationBody(operation, ""+"\n"
+"        return new "+referenceName+"(referentialLocale, this);\n"
+"    ");
        ObjectModelOperation toShortReference = addOperation(output, "toShortReference", ToolkitDtoIdBean.class.getSimpleName(), ObjectModelJavaModifier.PUBLIC);
        addAnnotation(output, toShortReference, Override.class);
        setOperationBody(toShortReference, ""+"\n"
+"        return ToolkitDtoIdBean.of("+referenceName+".class, getId(), getLastUpdateDate());\n"
+"    "
        );
    }

    @Override
    protected ObjectModel initOutputModel() {

        //FIXME Override builder to avoid bad imports when using synonyms in model...
        builder = new JavaBuilder(getModel().getName()) {
            @Override
            public ObjectModelOperation addOperation(ObjectModelClassifier classifier, String name, String type, ObjectModelModifier... modifiers) {
                ObjectModelOperationImpl result = new ObjectModelOperationImpl();
                result.setName(name);

                if (type != null) {
                    ObjectModelParameterImpl returnParameter =
                            new ObjectModelParameterImpl();
                    returnParameter.setType(type);
                    result.setReturnParameter(returnParameter);
                }

                result.addModifier(modifiers);
                ((ObjectModelClassifierImpl) classifier).addOperation(result);
                return result;
            }

            @Override
            public ObjectModelParameter addParameter(ObjectModelOperation operation, String type, String name) {
                ObjectModelOperationImpl impl = (ObjectModelOperationImpl) operation;
                ObjectModelParameterImpl param = new ObjectModelParameterImpl();
                param.setType(type);
                param.setName(name);
                impl.addParameter(param);
                return param;
            }

            @Override
            public ObjectModelAttribute addAttribute(ObjectModelClassifier classifier, String name, String type, String value, ObjectModelModifier... modifiers) {
                ObjectModelAttributeImpl attribute = new ObjectModelAttributeImpl();
                attribute.setName(name);
                attribute.setType(type);
                attribute.setDefaultValue(value);

                attribute.addModifier(modifiers);
                ObjectModelClassifierImpl classifierImpl = (ObjectModelClassifierImpl) classifier;
                classifierImpl.addAttribute(attribute);
                return attribute;
            }
        };
        setConstantPrefix("");
        return builder.getModel();
    }

    private Map<String, String> getFormProperties(ObjectModelClass input, ObjectModelClass output) {
        Collection<ObjectModelAttribute> attributes = new LinkedList<>(input.getAttributes());
        attributes.addAll(input.getAllOtherAttributes());
        Map<String, String> properties = new LinkedHashMap<>();
        for (ObjectModelAttribute attr : attributes) {

            if (!attr.isNavigable()) {
                continue;
            }

            String type = attr.getType();

            if (!type.endsWith("Reference") || !type.contains("referential")) {
                continue;
            }
//            if (true) {
//                properties.put(attr.getName(), type);
//            } else {
            addImport(output, type);
            properties.put(attr.getName(), JavaGeneratorUtil.getSimpleName(type));
//            }

        }

        return properties;
    }

    @Override
    protected void createSizeMethod(ObjectModelClass output, String attrName) {
        ObjectModelOperation operation = addOperation(
                output,
                getJavaBeanMethodName("get", attrName + "Size"),
                int.class,
                ObjectModelJavaModifier.PUBLIC
        );
        setOperationBody(operation, ""+"\n"
+"                return "+attrName+" == null ? 0 : "+attrName+".size();\n"
+"    "
        );
    }

    @Override
    protected void createIsEmptyMethod(ObjectModelClass output, String attrName) {
        super.createIsEmptyMethod(output, attrName);
        ObjectModelOperation isNotEmptyOperationImpl = addOperation(
                output,
                getJavaBeanMethodName("isNot", attrName + "Empty"),
                boolean.class,
                ObjectModelJavaModifier.PUBLIC
        );
        String isEmptyMethodName = getJavaBeanMethodName("is", attrName + "Empty");
        setOperationBody(isNotEmptyOperationImpl, ""+"\n"
+"        boolean empty = "+isEmptyMethodName+"();\n"
+"        return !empty;\n"
+"    "
        );
    }

    @Override
    public ObjectModelClass createClass(String className, String packageName) {
        return super.createClass(className, packageName);
    }

    @Override
    public void addImport(ObjectModelClass output, Class<?> type) {
        super.addImport(output, type);
    }

    @Override
    public void addImport(ObjectModelClass output, String type) {
        super.addImport(output, type);
    }

    @Override
    public void setSuperClass(ObjectModelClass output, Class<?> superClass) {
        super.setSuperClass(output, superClass);
    }

    @Override
    public ObjectModelAnnotation addAnnotation(ObjectModelClass output, ObjectModelElement output1, Class<?> annotationType) {
        return super.addAnnotation(output, output1, annotationType);
    }

    @Override
    public void addAnnotationParameter(ObjectModelClass output, ObjectModelAnnotation annotation, String name, String value) {
        super.addAnnotationParameter(output, annotation, name, value);
    }

    @Override
    public ObjectModelOperation addConstructor(ObjectModelClass output, ObjectModelJavaModifier modifiers) {
        return super.addConstructor(output, modifiers);
    }

    @Override
    public void setOperationBody(ObjectModelOperation constructor, String body) {
        super.setOperationBody(constructor, body);
    }

    @Override
    public ObjectModelParameter addParameter(ObjectModelOperation operation, Class<?> type, String name) {
        return super.addParameter(operation, type, name);
    }

    public ObjectModelOperation addOperation(ObjectModelClassifier classifier, String name, String type, ObjectModelModifier... modifiers) {
        return super.addOperation(classifier, name, type, modifiers);
    }

    @Override
    public ObjectModelAttribute addAttribute(ObjectModelClassifier classifier, String name, String type, String value, ObjectModelModifier... modifiers) {
        return super.addAttribute(classifier, name, type, value, modifiers);
    }
}
