package fr.ird.observe.toolkit.templates.entity;

/*-
 * #%L
 * ObServe Toolkit :: Templates
 * %%
 * Copyright (C) 2017 - 2021 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.ImmutableList;
import fr.ird.observe.spi.context.DtoEntityContext;
import org.nuiton.eugene.models.object.ObjectModelClass;
import org.nuiton.eugene.models.object.ObjectModelClassifier;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.topia.persistence.metadata.TopiaMetadataEntity;
import org.nuiton.topia.persistence.metadata.TopiaMetadataEntityPath;
import org.nuiton.topia.persistence.metadata.TopiaMetadataLink;
import org.nuiton.topia.persistence.metadata.TopiaMetadataReverseAssociation;
import org.nuiton.topia.templates.TopiaMetadataModelGeneratorSupport;

import java.io.IOException;
import java.io.Writer;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;

/**
 * Created on 13/09/2020.
 *
 * @author Tony Chemit - dev@tchemit.fr
 * @since 1.27
 */
public class EntityLastUpdateDateFieldSqlScriptGenerator extends TopiaMetadataModelGeneratorSupport {

    private static final String UPDATE_LAST_UPDATE_DATE_FIELD_SQL_START = "UPDATE %s.%s SET lastUpdateDate = '%%1$s'::timestamp, topiaVersion = topiaVersion + 1";
    private static final String UPDATE_LAST_UPDATE_DATE_FIELD_SQL_WHERE_EQUALS = " WHERE %s = '%%2$s'";
    private static final String UPDATE_LAST_UPDATE_DATE_FIELD_SQL_WHERE_SELECT = " WHERE topiaId = ( %s )";
    private static final String UPDATE_LAST_UPDATE_DATA_FIELD_SQL_SELECT_START = "SELECT %3$s FROM %1$s.%2$s %2$s";
    private static final String UPDATE_LAST_UPDATE_DATA_FIELD_SQL_INNER_JOIN_REVERSE = " INNER JOIN %1$s.%2$s %2$s ON %2$s.%3$s = %4$s.%5$s";
    private static final String UPDATE_LAST_UPDATE_DATA_FIELD_SQL_INNER_JOIN_SIMPLE = " INNER JOIN %1$s.%2$s %2$s ON %2$s.%3$s = %4$s.topiaId";

    @Override
    public String getFilenameForClassifier(ObjectModelClassifier classifier) {
        return super.getFilenameForClassifier(classifier) + DtoEntityContext.LAST_UPDATE_DATE_SQL_SCRIPT_FIELD_CLASSIFIER;
    }

    @Override
    public void generateFromClass(Writer output, ObjectModelClass input) throws IOException {
        TopiaMetadataEntity entity = getEntityEnumName(input);
        if (entity != null && !entity.isAbstract()) {
            processEntity(entity, output);
        }
    }

    protected void processEntity(TopiaMetadataEntity entity, Writer output) throws IOException {

        ImmutableList<TopiaMetadataEntity> types;
        ImmutableList<TopiaMetadataLink> links;
        Optional<TopiaMetadataEntityPath> optionalPath = getAllPaths().getEntityPathsForEntryPoint(entity);
        if (optionalPath.isPresent()) {
            TopiaMetadataEntityPath path = optionalPath.get();
            links = path.getLinks();
            types = path.getTypes();
        } else {
            // limit case, only one type
            types = ImmutableList.of(entity);
            // and no link
            links = ImmutableList.of();
        }

        ImmutableList<String> requests = generateLastUpdateDateFieldRequests(types, links);
        generate(output, requests);
    }

    private ImmutableList<String> generateLastUpdateDateFieldRequests(ImmutableList<TopiaMetadataEntity> types, ImmutableList<TopiaMetadataLink> links) {
        ImmutableList.Builder<String> builder = ImmutableList.builder();

        // take types in reverse order
        types = types.reverse();
        // take links in reverse order
        links = links.reverse();
        List<TopiaMetadataLink> inProcess = null;
        Iterator<TopiaMetadataLink> linksIterator = links.iterator();
        for (TopiaMetadataEntity type : types) {
            String fieldSql = generateLastUpdateDateFieldRequest(type, inProcess);
            builder.add(fieldSql + ";");
            if (inProcess == null) {
                // first round
                inProcess = new LinkedList<>();
            }
            if (linksIterator.hasNext()) {
                inProcess.add(linksIterator.next());
            }
        }
        return builder.build();
    }

    private String generateLastUpdateDateFieldRequest(TopiaMetadataEntity type, List<TopiaMetadataLink> inProcess) {
        String sql = String.format(UPDATE_LAST_UPDATE_DATE_FIELD_SQL_START, type.getDbSchemaName(), type.getDbTableName());
        if (inProcess == null) {
            sql += finalizeRequest(TopiaEntity.PROPERTY_TOPIA_ID);
            return sql;
        }
        // Needs links from root to type
        Iterator<TopiaMetadataLink> inOderLinks = ImmutableList.copyOf(inProcess).reverse().iterator();

        TopiaMetadataLink firstLink = inOderLinks.next();
        if (!inOderLinks.hasNext() && firstLink instanceof TopiaMetadataReverseAssociation) {
            sql += finalizeRequest(firstLink.getTargetDbName());
            return sql;
        }
        StringBuilder selectSql = new StringBuilder();
        selectSql.append(prepareSubSelect(firstLink));
        TopiaMetadataLink lastLink = firstLink;
        while (inOderLinks.hasNext()) {
            TopiaMetadataLink nextLink = inOderLinks.next();
            if (!(nextLink instanceof TopiaMetadataReverseAssociation)) {
                selectSql.append(appendInnerJoin(lastLink, nextLink));
            }
            lastLink = nextLink;
        }
        selectSql.append(finalizeSubSelect(lastLink));

        sql += String.format(UPDATE_LAST_UPDATE_DATE_FIELD_SQL_WHERE_SELECT, selectSql);
        return sql;
    }

    private String finalizeRequest(String column) {
        return String.format(UPDATE_LAST_UPDATE_DATE_FIELD_SQL_WHERE_EQUALS, column);
    }

    private String prepareSubSelect(TopiaMetadataLink firstLink) {
        String selectColumn;
        TopiaMetadataEntity tableOwner;
        if (firstLink instanceof TopiaMetadataReverseAssociation) {
            tableOwner = firstLink.getOwner();
            selectColumn = TopiaEntity.PROPERTY_TOPIA_ID;
        } else {
            tableOwner = firstLink.getTarget();
            selectColumn = firstLink.getTarget().getDbColumnName(firstLink.getOwner().getDbTableName());
        }
        return String.format(UPDATE_LAST_UPDATE_DATA_FIELD_SQL_SELECT_START, tableOwner.getDbSchemaName(), tableOwner.getDbTableName(), tableOwner.getDbTableName() + "." + selectColumn);
    }

    private String appendInnerJoin(TopiaMetadataLink lastLink, TopiaMetadataLink nextLink) {
        TopiaMetadataEntity tableOwner = nextLink.getTarget();
        if (lastLink instanceof TopiaMetadataReverseAssociation) {
            String columnName = lastLink.getTargetDbName();
            return String.format(UPDATE_LAST_UPDATE_DATA_FIELD_SQL_INNER_JOIN_REVERSE, tableOwner.getDbSchemaName(), tableOwner.getDbTableName(), columnName, lastLink.getOwner().getDbTableName(), lastLink.getTargetDbName());
        }
        String columnName = tableOwner.getDbColumnName(nextLink.getOwner().getDbTableName());
        return String.format(UPDATE_LAST_UPDATE_DATA_FIELD_SQL_INNER_JOIN_SIMPLE, tableOwner.getDbSchemaName(), tableOwner.getDbTableName(), columnName, lastLink.getTarget().getDbTableName());
    }

    private String finalizeSubSelect(TopiaMetadataLink lastLink) {
        String targetColumn;
        if (lastLink instanceof TopiaMetadataReverseAssociation) {
            targetColumn = lastLink.getTableName() + "." + lastLink.getTargetDbName();
        } else {
            targetColumn = lastLink.getTarget().getDbTableName() + ".topiaId";
        }
        return finalizeRequest(targetColumn);
    }

}
