package fr.ird.observe.toolkit.templates.entity;

/*-
 * #%L
 * ObServe Toolkit :: Templates
 * %%
 * Copyright (C) 2017 - 2021 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.auto.service.AutoService;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import fr.ird.observe.dto.DtoParentAware;
import fr.ird.observe.dto.IdDto;
import fr.ird.observe.dto.data.ContainerChildDto;
import fr.ird.observe.dto.data.EditableDto;
import fr.ird.observe.dto.data.OpenableDto;
import fr.ird.observe.dto.data.SimpleDto;
import fr.ird.observe.dto.reference.DtoReference;
import fr.ird.observe.dto.referential.ReferentialLocale;
import fr.ird.observe.entities.Entity;
import fr.ird.observe.entities.EntityHelper;
import fr.ird.observe.entities.data.DataEntity;
import fr.ird.observe.entities.referential.I18nReferentialEntity;
import fr.ird.observe.entities.referential.ReferentialEntity;
import fr.ird.observe.spi.PersistenceBusinessProject;
import fr.ird.observe.spi.ProjectPackagesDefinition;
import fr.ird.observe.spi.context.ContainerDtoEntityContext;
import fr.ird.observe.spi.context.DataDtoEntityContext;
import fr.ird.observe.spi.context.DataDtoEntityContextWithParent;
import fr.ird.observe.spi.context.DtoEntityContext;
import fr.ird.observe.spi.context.EditableDtoEntityContext;
import fr.ird.observe.spi.context.OpenableDtoEntityContext;
import fr.ird.observe.spi.context.ReferentialDtoEntityContext;
import fr.ird.observe.spi.context.SimpleDtoEntityContext;
import fr.ird.observe.spi.filter.EntityFilterConsumer;
import fr.ird.observe.spi.filter.EntityFilterPropertyConsumer;
import fr.ird.observe.spi.mapping.DtoToEntityContextMapping;
import fr.ird.observe.spi.mapping.EntityToDtoClassMapping;
import fr.ird.observe.spi.module.BusinessModule;
import fr.ird.observe.toolkit.navigation.spi.NavigationNodeDescriptor;
import fr.ird.observe.toolkit.templates.TemplateContract;
import fr.ird.observe.toolkit.templates.navigation.NavigationNodesBuilder;
import fr.ird.observe.toolkit.templates.navigation.NodeModel;
import io.ultreia.java4all.bean.definition.JavaBeanDefinition;
import io.ultreia.java4all.bean.definition.JavaBeanDefinitionStore;
import io.ultreia.java4all.bean.spi.GenerateJavaBeanDefinition;
import io.ultreia.java4all.classmapping.ImmutableClassMapping;
import io.ultreia.java4all.lang.Strings;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.maven.plugin.logging.SystemStreamLog;
import org.nuiton.eugene.EugeneCoreTagValues;
import org.nuiton.eugene.GeneratorUtil;
import org.nuiton.eugene.java.extension.ImportsManager;
import org.nuiton.eugene.java.extension.ObjectModelAnnotation;
import org.nuiton.eugene.models.object.ObjectModel;
import org.nuiton.eugene.models.object.ObjectModelAttribute;
import org.nuiton.eugene.models.object.ObjectModelClass;
import org.nuiton.eugene.models.object.ObjectModelClassifier;
import org.nuiton.eugene.models.object.ObjectModelElement;
import org.nuiton.eugene.models.object.ObjectModelEnumeration;
import org.nuiton.eugene.models.object.ObjectModelInterface;
import org.nuiton.eugene.models.object.ObjectModelJavaModifier;
import org.nuiton.eugene.models.object.ObjectModelModifier;
import org.nuiton.eugene.models.object.ObjectModelOperation;
import org.nuiton.eugene.models.object.ObjectModelPackage;
import org.nuiton.eugene.models.object.ObjectModelParameter;
import org.nuiton.eugene.models.object.xml.ObjectModelAttributeImpl;
import org.nuiton.eugene.models.object.xml.ObjectModelImpl;
import org.nuiton.eugene.models.object.xml.ObjectModelInterfaceImpl;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.topia.persistence.TopiaEntityContextable;
import org.nuiton.topia.persistence.event.ListenableTopiaEntity;
import org.nuiton.topia.persistence.internal.AbstractTopiaEntity;
import org.nuiton.topia.templates.TopiaEntityTransformer;
import org.nuiton.topia.templates.TopiaTemplateHelperExtension;

import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.Parameter;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.lang.reflect.TypeVariable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.stream.Collectors;




/**
 * A template to generate all the {@link TopiaEntity} api for all classifier
 * with a {@code entity} stereotype.
 * <p>
 * For example, given a {@code House} entity, it will generates :
 * <ul>
 * <li>{@code House} : contract of entity</li>
 * <li>{@code AbstractHouse} : default abstract implementation of entity</li>
 * <li>{@code HouseImpl} : default impl of abstract entity</li>
 * </ul>
 *
 * <b>Note: </b> The impl will ony be generated in these cases :
 * <ul>
 * <li>There is no abstract method</li>
 * <li>There is no already defined such class in class-path</li>
 * </ul>
 *
 * @author Tony Chemit - dev@tchemit.fr
 * @since 2.3.4
 */
@SuppressWarnings({"StringOperationCanBeSimplified", "CommentedOutCode"})
public class EntityTransformer extends TopiaEntityTransformer implements TemplateContract {
    private static final Logger log = LogManager.getLogger(EntityTransformer.class);
    private final Set<String> excludedMethodNames;
    private final Map<String, String> dtoToEntityContextMapping = new TreeMap<>();
    private final Map<String, String> entityToDtoClassMapping = new TreeMap<>();
    protected boolean doReference;
    private Class<? extends DtoReference> referenceContractType;
    private Class<? extends IdDto> dtoContractType;
    private Class<? extends IdDto> dtoType;
    private Class<? extends DtoReference> referenceType;
    private String parentWithGeneric;
    private String[] relativeNameExcludes;
    private List<String> entityClasses;
    private List<String> entityClassesFully;
    private ProjectPackagesDefinition def;
    private List<NodeModel> navigationNodes;

    public EntityTransformer() {
        excludedMethodNames = new LinkedHashSet<>();
        excludedMethodNames.add("toEntityType");
        excludedMethodNames.add("wait");
        excludedMethodNames.add("equals");
        excludedMethodNames.add("toString");
        excludedMethodNames.add("getClass");
        excludedMethodNames.add("clone");
        excludedMethodNames.add("hashCode");
        excludedMethodNames.add("notify");
        excludedMethodNames.add("notifyAll");
    }

    @SuppressWarnings({"unchecked", "rawtypes"})
    @Override
    protected void setDoDto(ObjectModelClass input) {
        super.setDoDto(input);
        boolean isAbstract = input.isAbstract() && !input.isStatic();
        doReference = doDto && !isAbstract && entityToDtoMapping.acceptReference(input);
        if (doDto) {
            dtoContractType = ((ToolkitEntityToDtoMapping) entityToDtoMapping).getDtoContract2();
            referenceContractType = ((ToolkitEntityToDtoMapping) entityToDtoMapping).getReferenceType(dtoContractType);
            dtoType = (Class<? extends IdDto>) entityToDtoMapping.getDtoType();
            Collection<? extends ObjectModelClassifier> classifiersToTest = new LinkedList<>(input.getInterfaces());
            classifiersToTest.addAll((Collection) input.getSuperclasses());
            for (ObjectModelClassifier parent : classifiersToTest) {
                ObjectModelClass parent1 = model.getClass(parent.getQualifiedName());
                if (parent1 != null && templateHelper.isEntity(parent1)) {
                    parentWithGeneric = parent.getQualifiedName();
                    break;
                }
            }
        }
        if (doReference) {
            referenceType = ((ToolkitEntityToDtoMapping) entityToDtoMapping).getReferenceType(dtoType);
        }
    }

    @Override
    public void transformFromModel(ObjectModel model) {
        def = ProjectPackagesDefinition.of(getClassLoader());
        EugeneCoreTagValues coreTagValues = new EugeneCoreTagValues();
        Set<String> relativeNameExcludes = coreTagValues.getRelativeNameExcludes(model);
        this.relativeNameExcludes = relativeNameExcludes == null ? new String[0] : relativeNameExcludes.toArray(new String[0]);
        templateHelper = new TopiaTemplateHelperExtension(model);
        NavigationNodeDescriptor navigationNodeDescriptor = NavigationNodeDescriptor.getRootDescriptor();
        navigationNodes = new NavigationNodesBuilder(new SystemStreamLog()).build(navigationNodeDescriptor);
        entityClasses = templateHelper.getEntityClasses(model, false).stream().map(ObjectModelClass::getName).collect(Collectors.toList());
        entityClassesFully = templateHelper.getEntityClasses(model, false).stream().map(ObjectModelClass::getQualifiedName).collect(Collectors.toList());
        //FIXME Remove this ASAP
        entityClassesFully.add("fr.ird.observe.entities.data.ll.common.Program");
        entityClassesFully.add("fr.ird.observe.entities.data.ps.common.Program");
        super.transformFromModel(model);
    }

    @Override
    protected void debugOutputModel() {
        super.debugOutputModel();

        generateClassMapping(true, DtoToEntityContextMapping.class, "Class<? extends " + IdDto.class.getName() + ">", DtoEntityContext.class.getName() + "<?, ?, ?, ?>", ImmutableMap.class, "build", dtoToEntityContextMapping);
        generateClassMapping(true, EntityToDtoClassMapping.class, Entity.class.getName(), IdDto.class.getName(), ImmutableClassMapping.class, "getMappingBuilder", entityToDtoClassMapping);
        String businessProjectClassName = generateBusinessProject(model.getName());
        if (isVerbose()) {
            getLog().info("Generated: " + businessProjectClassName);
        }
    }

    @Override
    protected boolean isGenerateImpl(ObjectModelClass input) {
        if (ReferentialEntity.class.getName().equals(input.getQualifiedName())) {
            return true;
        }
        if (I18nReferentialEntity.class.getName().equals(input.getQualifiedName())) {
            return true;
        }
        if (DataEntity.class.getName().equals(input.getQualifiedName())) {
            return true;
        }
        return super.isGenerateImpl(input);
    }


    @SuppressWarnings("unused")
    private String generateBusinessProject(String name) {

        String className = String.format("%s%s", name, PersistenceBusinessProject.class.getSimpleName());

        ObjectModelClass aClass = createClass(className, PersistenceBusinessProject.class.getPackage().getName());
        ObjectModelAnnotation annotation = addAnnotation(aClass, aClass, AutoService.class);
        ImportsManager importManager = getImportManager(aClass);
        addAnnotationClassParameter(importManager, aClass, annotation, "value", PersistenceBusinessProject.class);

        setSuperClass(aClass, PersistenceBusinessProject.class);
        addImport(aClass, ImmutableList.class);
        addImport(aClass, BusinessModule.class);
        ObjectModelOperation constructor = addConstructor(aClass, ObjectModelJavaModifier.PUBLIC);
        String entityToDtoMapping = EntityToDtoClassMapping.class.getPackage().getName() + "." + name + EntityToDtoClassMapping.class.getSimpleName();
        String dtoToEntityContextMapping = DtoToEntityContextMapping.class.getPackage().getName() + "." + name + DtoToEntityContextMapping.class.getSimpleName();

        setOperationBody(constructor, ""+"\n"
+"        super("+entityToDtoMapping+".get(), "+dtoToEntityContextMapping+".get());\n"
+"    ");
        return aClass.getQualifiedName();
    }

    @Override
    protected void createEntityInterface(ObjectModelPackage aPackage, ObjectModelClass input) {

        boolean isAbstract = input.isAbstract() && !input.isStatic();
        if (isAbstract && doDto) {

            String name = String.format("%s<Dt extends %s, R extends %s>", input.getName(), dtoContractType.getName(), referenceContractType.getName());
            if (generateInterface) {
                outputInterface = createInterface(name, input.getPackageName());
                addImport(outputInterface, dtoContractType);
                addImport(outputInterface, referenceContractType);
            } else {
                outputInterface = new ObjectModelInterfaceImpl();
                ((ObjectModelInterfaceImpl) outputInterface).setName(name);
            }
        } else {
            if (generateInterface) {
                outputInterface = createInterface(input.getName(), input.getPackageName());
            } else {
                outputInterface = new ObjectModelInterfaceImpl();
                ((ObjectModelInterfaceImpl) outputInterface).setName(input.getName());
            }
        }
        addImport(outputInterface, outputInterface.getQualifiedName());
        if (!generateInterface) {
            return;
        }
        // Documentation
        if (GeneratorUtil.hasDocumentation(input)) {
            setDocumentation(outputInterface, input.getDocumentation());
        }
        addAnnotation(outputInterface, outputInterface, GenerateJavaBeanDefinition.class);
        boolean oneParentContainsEntity = false;
        List<String> interfaceAlreadyDone = new LinkedList<>();
        for (ObjectModelClassifier parent : input.getInterfaces()) {
            if (doDto) {
                Set<String> parents = collectAllInterfaces(parent);
                oneParentContainsEntity = parents.contains(Entity.class.getName());

                if (parent.getQualifiedName().equals(parentWithGeneric)) {
                    if (isAbstract) {
                        addImport(outputInterface, parentWithGeneric);
                        addInterface(interfaceAlreadyDone, outputInterface, String.format("%s<Dt, R>", parentWithGeneric));
                    } else {
                        addImport(outputInterface, dtoType);
                        addImport(outputInterface, referenceType);
                        addImport(outputInterface, parentWithGeneric);
                        addInterface(interfaceAlreadyDone, outputInterface, String.format("%s<%s, %s>", parentWithGeneric, dtoType.getName(), referenceType.getName()));
                    }
                    continue;
                }
                if (oneParentContainsEntity) {
                    if (isAbstract) {
                        addImport(outputInterface, parent.getQualifiedName());
                        addInterface(interfaceAlreadyDone, outputInterface, String.format("%s<Dt, R>", parent.getQualifiedName()));
                    } else {
                        addImport(outputInterface, dtoType);
                        addImport(outputInterface, referenceType);
                        addImport(outputInterface, parent.getQualifiedName());
                        addInterface(interfaceAlreadyDone, outputInterface, String.format("%s<%s, %s>", parent.getQualifiedName(), dtoType.getName(), referenceType.getName()));
                    }
                    continue;
                }
            }
            addInterface(interfaceAlreadyDone, outputInterface, parent);

        }

        // Extends from inheritance
        boolean needTopiaEntity = true;
        for (ObjectModelClassifier parent : input.getSuperclasses()) {
            if (templateHelper.isEntity(parent)) {
                if (doDto) {
                    if (isAbstract) {
                        addInterface(interfaceAlreadyDone, outputInterface, String.format("%s<Dt, R>", parent.getQualifiedName()));
                    } else {
                        if (oneParentContainsEntity) {
                            continue;
                        }
                        addImport(outputInterface, dtoType);
                        addImport(outputInterface, referenceType);
                        addInterface(interfaceAlreadyDone, outputInterface, String.format("%s<%s, %s>", parent.getQualifiedName(), dtoType.getName(), referenceType.getName()));
                    }
                } else {
                    addInterface(interfaceAlreadyDone, outputInterface, parent);
                }
                needTopiaEntity = false;
            }
        }

        // Extends TopiaEntity (only if hasn't parent entity)
        if (needTopiaEntity) {

            Class<?> interfaze = TopiaEntity.class;

            if (topiaCoreTagValues.getContextableTagValue(input, aPackage, model)) {
                interfaze = TopiaEntityContextable.class;
            }

            addInterface(interfaceAlreadyDone,
                         outputInterface,
                         interfaze);

        } else if (topiaCoreTagValues.getContextableTagValue(input, aPackage, model)) {
            // Even if there is no need to implement TopiaEntity, it might be
            // necessary to implement TopiaEntityContextable
            addInterface(interfaceAlreadyDone,
                         outputInterface,
                         TopiaEntityContextable.class);
        }

        if (generatePropertyChangeSupport) {
            addInterface(interfaceAlreadyDone,
                         outputInterface,
                         ListenableTopiaEntity.class);
        }

        if (generateInterface && doDto && !isAbstract) {
            String daoMethodName = Strings.getRelativeCamelCaseName(getDefaultPackageName(), def.getRelativeEntityPackage(input.getQualifiedName() + "Dao"), relativeNameExcludes);

            String entityName = input.getName();
            String entityPackageName = input.getPackageName();

            String daoName = entityName + "TopiaDao";
            String persistenceContext = getDefaultPackageName() + "." + templateHelper.getPersistenceContextConcreteName(model);
            addImport(outputInterface, persistenceContext);
//            persistenceContext = GeneratorUtil.getSimpleName(persistenceContext);
            for (Class<?> dtoDtoType : entityToDtoMapping.getDtoTypes(input)) {
                generateSpi(input, dtoDtoType, entityPackageName, entityName, daoName, persistenceContext, daoMethodName);
            }
        }
        if (!isAbstract) {
            generateJavaBeanMethods();
        }
        generateInterfaceUsageConstant(input);
    }

    protected Pair<String, String> getParent(Class<?> dtoDtoType) {
        for (NodeModel navigationNode : navigationNodes) {
            if (navigationNode.isSimple() || navigationNode.isOpenList() || navigationNode.isReferential()) {
                continue;
            }
            String dtoType = navigationNode.getDtoType();
            if (dtoDtoType.getName().equals(dtoType)) {
                NodeModel parent = navigationNode.getParent();
                if (parent != null) {
                    String propertyName;
                    if (parent.isOpenList()) {
                        NodeModel finalParent = parent;
                        if (parent.getParent().isRoot()) {
                            propertyName = "children";
                        } else {
                            propertyName = parent.getParent().getChildren().stream().filter(c -> c.getTargetModel().equals(finalParent)).findFirst().orElseThrow().getPropertyName();
                            parent = parent.getParent();
                        }
                    } else {
                        propertyName = parent.getChildren().stream().filter(c -> c.getTargetModel().equals(navigationNode)).findFirst().orElseThrow().getPropertyName();
                    }
                    return Pair.of(propertyName, parent.getDtoType().replace("Dto", "").replace("fr.ird.observe.dto", "fr.ird.observe.entities"));
                }
                return null;
            }
        }
        return null;
    }

    @SuppressWarnings({"UnusedAssignment", "unused"})
    protected void generateSpi(ObjectModelClass input, Class<?> dtoDtoType, String entityPackageName, String entityName, String daoName, @SuppressWarnings("unused") String persistenceContext, @SuppressWarnings("unused") String daoMethodName) {

        @SuppressWarnings("unchecked") Class<? extends DtoReference> referenceType = ((ToolkitEntityToDtoMapping) entityToDtoMapping).getReferenceType((Class<? extends IdDto>) dtoDtoType);
        String parentType = null;
        String parentPropertyName = null;
        boolean referential = TemplateContract.isReferentialFromPackageName(entityPackageName);
        boolean mainSpi = dtoDtoType.getSimpleName().equals(entityName + "Dto");
        if (!referential) {
            Pair<String, String> pair = getParent(dtoDtoType);
            parentType = pair == null ? null : pair.getValue();
            parentPropertyName = pair == null ? null : pair.getKey();
            if (parentType != null && !entityClassesFully.contains(parentType)) {
                parentType = input.getQualifiedName();
            }
        }

        boolean noReference = referenceType == null;
        if (!mainSpi && referenceType == null) {
            referenceType = ((ToolkitEntityToDtoMapping) entityToDtoMapping).getReferenceType(dtoType);
        }
        String dtoTypeName = dtoDtoType.getSimpleName();
        String referenceTypeName = Objects.requireNonNull(referenceType).getSimpleName();
        String spiClassName = dtoTypeName.replace("Dto", "Spi");
        String generatedSpiClassName = "Generated" + spiClassName;
        ObjectModelClass generatedSpi = createAbstractClass(generatedSpiClassName, entityPackageName);

        Class<?> superClass;
        String generatedSpiSuperClass = null;
        String containerTypeName = null;
        if (parentType == null) {
            if (referential) {
                superClass = ReferentialDtoEntityContext.class;
                generatedSpiSuperClass = String.format("%s<%s, %s, %s, %s>", superClass.getName(), dtoTypeName, referenceTypeName, entityName, daoName);
            } else {
                superClass = DataDtoEntityContext.class;
                if (SimpleDto.class.isAssignableFrom(dtoDtoType)) {
                    superClass = SimpleDtoEntityContext.class;
                }
                generatedSpiSuperClass = String.format("%s<%s, %s, %s, %s>", superClass.getName(), dtoTypeName, referenceTypeName, entityName, daoName);
            }
        } else {
            if (OpenableDto.class.isAssignableFrom(dtoDtoType)) {
                superClass = OpenableDtoEntityContext.class;
            } else if (EditableDto.class.isAssignableFrom(dtoDtoType)) {
                superClass = EditableDtoEntityContext.class;
            } else if (ContainerChildDto.class.isAssignableFrom(dtoDtoType)) {
                superClass = ContainerDtoEntityContext.class;
                containerTypeName = dtoDtoType.getPackageName() + "." + GeneratorUtil.getSimpleName(parentType) + dtoDtoType.getSimpleName();
                containerTypeName = importAndSimplify(generatedSpi, containerTypeName);
                generatedSpiSuperClass = String.format("%s<%s, %s, %s, %s, %s, %s>",
                                                       superClass.getName(),
                                                       parentType,
                                                       dtoTypeName,
                                                       containerTypeName,
                                                       referenceTypeName,
                                                       entityName,
                                                       daoName);
            } else {
                superClass = DataDtoEntityContextWithParent.class;
            }
            if (generatedSpiSuperClass == null) {
                generatedSpiSuperClass = String.format("%s<%s, %s, %s, %s, %s>", superClass.getName(), parentType, dtoTypeName, referenceTypeName, entityName, daoName);
            }
        }

        setSuperClass(generatedSpi, generatedSpiSuperClass);
        addImport(generatedSpi, dtoDtoType);
        addImport(generatedSpi, referenceType);
        if (mainSpi || referential) {
            // add filter methods on generated spi
            addFilterMethods(input, generatedSpi);
        }
        ObjectModelOperation constructor = addConstructor(generatedSpi, ObjectModelJavaModifier.PROTECTED);
        if (noReference) {
            referenceTypeName = "null";
        } else {
            referenceTypeName += ".class";
        }
        if (parentType == null) {
            setOperationBody(constructor, ""+"\n"
+"        super("+dtoTypeName+".class, "+referenceTypeName+", "+entityName+".class, "+entityName+"Impl.class, t->(("+persistenceContext+") t).get"+daoMethodName+"());\n"
+"    ");
        } else {
            String parentTypeName = importAndSimplify(generatedSpi, parentType);
            if (containerTypeName != null) {
                setOperationBody(constructor, ""+"\n"
+"        super("+parentTypeName+".class, \""+parentPropertyName+"\", "+dtoTypeName+".class, "+containerTypeName+".class, "+referenceTypeName+", "+entityName+".class,  "+entityName+"Impl.class, t->(("+persistenceContext+") t).get"+daoMethodName+"());\n"
+"    ");
            } else {
                setOperationBody(constructor, ""+"\n"
+"        super("+parentTypeName+".class, \""+parentPropertyName+"\", "+dtoTypeName+".class, "+referenceTypeName+", "+entityName+".class,  "+entityName+"Impl.class, t->(("+persistenceContext+") t).get"+daoMethodName+"());\n"
+"    ");
            }
        }
        String fqn;
        if (mainSpi) {
            // generate main spi (with delegate method)
            fqn = generateMainSpi(dtoDtoType, entityPackageName, entityName, spiClassName);
            // generate default methods
            generateSpiDelegateMethods(referential, daoName);
        } else {
            // generate not main psi
            fqn = generateSubSpi(dtoDtoType, entityPackageName, entityName, spiClassName);
        }
        dtoToEntityContextMapping.put(dtoDtoType.getName(), fqn);
        if (getResourcesHelper().isJavaFileInClassPath(entityPackageName + "." + spiClassName)) {
            return;
        }
        ObjectModelClass spi = createClass(spiClassName, entityPackageName);
        setSuperClass(spi, generatedSpi.getQualifiedName());

    }

    @SuppressWarnings("unused")
    private void addFilterMethods(ObjectModelClass input, ObjectModelClass generatedSpi) {

        ObjectModelPackage aPackage = model.getPackage(input);
        String inputSimpleName = input.getName();
        StringBuilder filterContent = new StringBuilder(""+"\n"
+"            ImmutableList.<EntityFilterPropertyConsumer>builder()\n"
+"                    .add(EntityFilterPropertyConsumer.newIdProperty())");
        Collection<ObjectModelAttribute> attributes = new LinkedList<>(input.getAttributes());
        attributes.addAll(input.getAllOtherAttributes());
        attributes.removeIf(a->!a.isNavigable());
        attributes.removeIf(GeneratorUtil::isNMultiplicity);
        Map<String, ObjectModelAttribute> orderedAttributes = new TreeMap<>();
        for (ObjectModelAttribute attribute : attributes) {
            String attrName = GeneratorUtil.getSimpleName(getPropertyName(attribute));
            orderedAttributes.put(attrName, attribute);
        }
        for (Map.Entry<String, ObjectModelAttribute> entry : orderedAttributes.entrySet()) {
            String attrName = entry.getKey();
            ObjectModelAttribute attribute = entry.getValue();
            String attrType = getPropertyType(attribute, input, aPackage);
            attrType = GeneratorUtil.removeAnyGenericDefinition(attrType);

            String methodName = null;
            String extraCode="";
            if (entityClassesFully.contains(attrType)) {
                if (attrType.contains(".referential.")) {
                    methodName = "newReferentialProperty";
                } else {
                    methodName = "newDataProperty";
                }
            } else {
                ObjectModelEnumeration enumeration = model.getEnumeration(attrType);
                attrType = GeneratorUtil.getSimpleName(attrType);
                if (enumeration != null) {
                    methodName = "newEnumProperty";
                    extraCode=", "+enumeration.getQualifiedName()+".class";
                } else if ("String".equals(attrType)) {
                    methodName = "newStringProperty";
                } else if ("Boolean".equals(attrType) || "boolean".equals(attrType)) {
                    methodName = "newBooleanProperty";
                } else if ("Integer".equals(attrType) || "int".equals(attrType)) {
                    methodName = "newIntegerProperty";
                } else if ("Long".equals(attrType) || "long".equals(attrType)) {
                    methodName = "newLongProperty";
                } else if ("Float".equals(attrType) || "float".equals(attrType)) {
                    methodName = "newFloatProperty";
                } else if ("Date".equals(attrType)) {
                    String type = topiaHibernateTagValues.getHibernateAttributeType(attribute, input, aPackage, model);
                    if (type != null) {
                        if ("date".equals(type)) {
                            methodName = "newDateProperty";
                        } else if ("time".equals(type)) {
                            methodName = "newTimeProperty";
                        }
                    }
                    if (methodName == null) {
                        methodName = "newTimestampProperty";
                    }
                }
            }
            if (methodName == null) {
                getLog().warn(String.format("No filter on %s.%s", inputSimpleName, attrName));
                continue;
            }
            String constantName = getConstantName(attrName);
            filterContent.append(""+"\n"
+"                    .add(EntityFilterPropertyConsumer."+methodName+"("+inputSimpleName+"."+constantName+""+extraCode+"))");
        }
        filterContent.append(""+"\n"
+"                    .build()");
        addImport(generatedSpi, ImmutableList.class);
        addImport(generatedSpi, ImmutableMap.class);
        addImport(generatedSpi, ImmutableSet.class);
        addImport(generatedSpi, EntityFilterPropertyConsumer.class);
        addImport(generatedSpi, EntityFilterConsumer.class);
        addAttribute(generatedSpi, "FILTER_PROPERTIES", "ImmutableList<EntityFilterPropertyConsumer>", filterContent.toString(),
                     ObjectModelJavaModifier.FINAL,
                     ObjectModelJavaModifier.PRIVATE,
                     ObjectModelJavaModifier.STATIC);

        addAttribute(generatedSpi, "FILTER_AUTHORIZED_PROPERTIES", "ImmutableSet<String>", "EntityFilterConsumer.getAuthorizedPropertyNames(FILTER_PROPERTIES)",
                     ObjectModelJavaModifier.FINAL,
                     ObjectModelJavaModifier.PUBLIC,
                     ObjectModelJavaModifier.STATIC);
        addAttribute(generatedSpi, "FILTER_AUTHORIZED_ORDERS", "ImmutableSet<String>", "EntityFilterConsumer.getAuthorizedOrders(FILTER_PROPERTIES)",
                     ObjectModelJavaModifier.FINAL,
                     ObjectModelJavaModifier.PUBLIC,
                     ObjectModelJavaModifier.STATIC);
        addAttribute(generatedSpi, "FILTER_PROPERTIES_CONSUMERS", "ImmutableMap<String, EntityFilterPropertyConsumer>", "EntityFilterConsumer.getPropertyConsumers(FILTER_PROPERTIES)",
                     ObjectModelJavaModifier.FINAL,
                     ObjectModelJavaModifier.PRIVATE,
                     ObjectModelJavaModifier.STATIC);

        ObjectModelOperation newFilterConsumer = addOperation(generatedSpi, "newFilterConsumer", "EntityFilterConsumer<" + inputSimpleName + ">",
                                                              ObjectModelJavaModifier.PUBLIC,
                                                              ObjectModelJavaModifier.FINAL);
        addParameter(newFilterConsumer, ReferentialLocale.class, "referentialLocale");
        addAnnotation(generatedSpi, newFilterConsumer, Override.class);
        setOperationBody(newFilterConsumer, ""+"\n"
+"        return new EntityFilterConsumer<>(referentialLocale, FILTER_PROPERTIES, FILTER_AUTHORIZED_PROPERTIES, FILTER_AUTHORIZED_ORDERS, FILTER_PROPERTIES_CONSUMERS);\n"
+"    ");
    }

    protected String generateMainSpi(Class<?> dtoDtoType, String entityPackageName, String entityName, String constantType) {
        String constantName = "SPI";
        addAttribute(outputInterface, constantName, constantType, "new " + constantType + "()");
        entityToDtoClassMapping.put(entityPackageName + "." + entityName, dtoDtoType.getName() + ".class");
        entityToDtoClassMapping.put(entityPackageName + "." + entityName + "Impl", dtoDtoType.getName() + ".class");
        return entityPackageName + "." + entityName + "." + constantName;
    }

    protected String generateSubSpi(Class<?> dtoDtoType, String entityPackageName, String entityName, String constantType) {
        String constantName = builder.getConstantName(ProjectPackagesDefinition.cleanType(dtoDtoType.getSimpleName()).replaceAll("\\.", "_") + "_spi");
        addAttribute(outputInterface, constantName, constantType, "new " + constantType + "()");
        return entityPackageName + "." + entityName + "." + constantName;
    }

    @SuppressWarnings({"unused"})
    @Override
    protected void createEntityAbstractClass(ObjectModelPackage aPackage, ObjectModelClass input) {
        boolean isAbstract = input.isAbstract() && !input.isStatic();
        if (doDto && isAbstract) {
            outputAbstract = createAbstractClass(String.format("%s<Dt extends %s, R extends %s>", input.getName() + "Abstract", dtoContractType.getName(), referenceContractType.getName()), input.getPackageName());
            addImport(outputAbstract, dtoContractType);
            addImport(outputAbstract, referenceContractType);
        } else {
            outputAbstract = createAbstractClass(input.getName() + "Abstract", input.getPackageName());
        }
        addImport(outputAbstract, input.getQualifiedName());

        // Documentation
        StringBuilder doc = new StringBuilder();
        doc.append("Implantation POJO pour l'entité {@link ");
        doc.append(StringUtils.capitalize(input.getName()));
        doc.append("}\n");

        String dbName = templateHelper.getDbName(input);
        if (dbName != null) {
            doc.append("<p>Nom de l'entité en BD : ");
            doc.append(dbName);
            doc.append(".</p>");
        }

        setDocumentation(outputAbstract, doc.toString());

        boolean isDataEntity = DataEntity.class.getSimpleName().equals(input.getName()) || ReferentialEntity.class.getSimpleName().equals(input.getName());

        if (doDto) {

            if (isAbstract) {
                addImport(outputAbstract, parentWithGeneric + "Impl");
                addImport(outputAbstract, parentWithGeneric);
                if (isDataEntity) {
                    setSuperClass(outputAbstract, parentWithGeneric + "Impl");
                } else {
                    setSuperClass(outputAbstract, String.format("%s<Dt, R>", parentWithGeneric + "Impl"));
                }
                addInterface(outputAbstract, String.format("%s<Dt, R>", input.getQualifiedName()));
            } else {
                addImport(outputAbstract, dtoType);
                addImport(outputAbstract, referenceType);
                addImport(outputAbstract, parentWithGeneric);
                setSuperClass(outputAbstract, String.format("%s<%s, %s>", parentWithGeneric + "Impl", dtoType.getName(), referenceType.getName()));
                addInterface(outputAbstract, String.format("%s<%s, %s>", parentWithGeneric, dtoType.getName(), referenceType.getName()));
                addInterface(outputAbstract, input.getQualifiedName());
            }
        } else {

            // Implements
            addInterface(outputAbstract, input.getQualifiedName());

            // Extends
            for (ObjectModelClass parent : input.getSuperclasses()) {
                //tchemit-2011-09-12 What ever abstract or not, we always use an Impl, moreover use the util method instead
                String extendClass = templateHelper.getDOType(parent, model);
//            String extendClass = parent.getQualifiedName();
//            //Si une des classes parentes définies des méthodes abstraites, son
//            // impl ne sera pas créé
//            boolean abstractParent = templateHelper.shouldBeAbstract(parent);
//            if (templateHelper.isEntity(parent)) {
//                if (abstractParent) {
//                    extendClass += "Abstract";
//                } else {
//                    extendClass += "Impl";
//                }
//            }
                setSuperClass(outputAbstract, extendClass);
            }

            // Extends AbstractTopiaEntity (only if hasn't parent entity)
            if (outputAbstract.getSuperclasses().isEmpty()) {

                String superClassName = topiaCoreTagValues.getEntitySuperClassTagValue(input, aPackage, model);
                if (superClassName == null) {
                    superClassName = AbstractTopiaEntity.class.getName();
                }
                setSuperClass(outputAbstract, superClassName);
            }

        }

        if (topiaCoreTagValues.getContextableTagValue(input, aPackage, model)) {
            addContextableMethods(input, outputAbstract);
        }


        if (doReference) {
            String mainClassName = dtoType.getName();
            String referenceClassName = referenceType.getName();

            // add toDto method
            ObjectModelOperation toDto = addOperation(outputAbstract, "toDto", mainClassName, ObjectModelJavaModifier.PUBLIC, ObjectModelJavaModifier.FINAL);
            addAnnotation(outputAbstract, toDto, Override.class);
            addParameter(toDto, ReferentialLocale.class, "referentialLocale");
            setOperationBody(toDto, ""+"\n"
+"        return "+input.getName()+".toDto(referentialLocale, this);\n"
+"    "
            );

            // add toReference method
            ObjectModelOperation toReference = addOperation(outputAbstract, "toReference", referenceClassName, ObjectModelJavaModifier.PUBLIC);
            addAnnotation(outputAbstract, toReference, Override.class);
            addParameter(toReference, ReferentialLocale.class, "referentialLocale");
            setOperationBody(toReference, ""+"\n"
+"        return toDto(referentialLocale).toReference(referentialLocale);\n"
+"    ");
        }
    }

    @Override
    protected void generateImpl(ObjectModelClass input) {
        String implName = input.getName() + "Impl";
        String packageName = input.getPackageName();
        if (isVerbose()) {
            log.info("Will generate [" + implName + "]");
        }
        boolean isAbstract = isAbstract(input) && !input.isStatic();
        if (isAbstract) {
            if (doDto) {
                outputImpl = createAbstractClass(String.format("%s<Dt extends %s, R extends %s>", implName, dtoContractType.getName(), referenceContractType.getName()), packageName);
                setSuperClass(outputImpl, input.getQualifiedName() + "Abstract<Dt, R>");
//                if (DataEntity.class.getSimpleName().equals(input.getName())) {
//                    setSuperClass(outputImpl, input.getQualifiedName() + "Abstract");
//                }
            } else {
                outputImpl = createAbstractClass(implName, packageName);
                setSuperClass(outputImpl, input.getQualifiedName() + "Abstract");
            }
        } else {
            outputImpl = createClass(implName, packageName);
            setSuperClass(outputImpl, input.getQualifiedName() + "Abstract");
        }

        setDocumentation(outputImpl, String.format("Implantation des operations pour l'entité %s.", input.getName()));
    }

    @Override
    protected void generateProperties(Collection<ObjectModelAttribute> attributes, ObjectModelClassifier aClass, ObjectModelPackage aPackage) {
        boolean withParentId = aClass.getInterfaces().stream().anyMatch(i -> i.getQualifiedName().equals(DtoParentAware.class.getName()));
        ObjectModelAttributeImpl e = null;
        if (withParentId) {
            e = new ObjectModelAttributeImpl();
            e.setName("parentId");
            e.setType(String.class.getName());
            e.setObjectModelImpl((ObjectModelImpl) model);
            e.setReverseAttributeName("nope");
            e.postInit();
            attributes.add(e);
        }
        super.generateProperties(attributes, aClass, aPackage);
        if (withParentId) {
            attributes.remove(e);
        }
    }

    @Override
    protected void closeAcceptInternalOperation() {
        if (outputAbstract == null) {
            return;
        }
        super.closeAcceptInternalOperation();
        if (DataEntity.class.getSimpleName().equals(outputInterface.getName())) {
            addImport(outputAbstract, EntityHelper.class);
            ObjectModelOperation toDto = outputAbstract.getOperations("toDto").iterator().next();
            setOperationBody(toDto, "" +"\n"
+"        EntityHelper.toDataDto(this, dto);\n"
+"    "
            );
            ObjectModelOperation fromDto = outputAbstract.getOperations("fromDto").iterator().next();
            setOperationBody(fromDto, "" +"\n"
+"        EntityHelper.fromDataDto(this, dto);\n"
+"    "
            );
        } else if (ReferentialEntity.class.getSimpleName().equals(outputInterface.getName())) {
            addImport(outputAbstract, EntityHelper.class);
            ObjectModelOperation toDto = outputAbstract.getOperations("toDto").iterator().next();
            setOperationBody(toDto, "" +"\n"
+"        EntityHelper.toReferentialDto(this, dto);\n"
+"    "
            );
            ObjectModelOperation fromDto = outputAbstract.getOperations("fromDto").iterator().next();
            setOperationBody(fromDto, "" +"\n"
+"        EntityHelper.fromReferentialDto(this, dto);\n"
+"    "
            );
        } else {
            for (ObjectModelOperation toDto : outputAbstract.getOperations("toDto")) {
                if (toDto.getParameters().size() == 2) {
                    setOperationBody(toDto, toDto.getBodyCode() + "" +"    dto.postInit();\n"
+"    "
                    );
                    break;
                }
            }
        }
    }

    @Override
    public ObjectModelOperation addOperation(ObjectModelClassifier classifier, String name, String type, ObjectModelModifier... modifiers) {
        if (entityClasses != null && !type.startsWith(getDefaultPackageName())) {
            String type1 = GeneratorUtil.removeAnyGenericDefinition(GeneratorUtil.getSimpleName(type, true));
            if (!entityClasses.contains(type1)) {
                Set<String> types = GeneratorUtil.getTypesList(type);
                boolean doIt = true;
                for (String s : types) {
                    String s1 = GeneratorUtil.getSimpleName(s);
                    if (entityClasses.contains(s1)) {
                        doIt = false;
                        break;
                    }
                }
                if (doIt) {
                    addImport(classifier, type);
                    type = GeneratorUtil.getSimpleName(type);
                }
            }
        }
        return super.addOperation(classifier, name, type, modifiers);
    }

    @Override
    protected ObjectModelParameter addParameter(ObjectModelOperation operation, String type, String name) {
        if (!type.startsWith(getDefaultPackageName())) {
            String type1 = GeneratorUtil.removeAnyGenericDefinition(GeneratorUtil.getSimpleName(type, true));
            ObjectModelClassifier declaringElement = (ObjectModelClassifier) operation.getDeclaringElement();
            if (!entityClasses.contains(type1)) {
                Set<String> types = GeneratorUtil.getTypesList(type);
                boolean doIt = true;
                for (String s : types) {
                    String s1 = GeneratorUtil.getSimpleName(s);
                    if (entityClasses.contains(s1)) {
                        doIt = false;
                        break;
                    }
                }
                if (doIt) {
                    addImport(declaringElement, type);
                    type = GeneratorUtil.getSimpleName(type);
                }
            }
        }
        return super.addParameter(operation, type, name);
    }

    @Override
    protected void clean() {
        super.clean();
        dtoContractType = dtoType = null;
        referenceContractType = referenceType = null;
        parentWithGeneric = null;
        doDto = doReference = false;
    }


    private String getTypeGenericBounds(Type type, String daoName) {
        Set<String> typeVariables = null;
        if (type instanceof TypeVariable) {
            TypeVariable<?> type1 = (TypeVariable<?>) type;
            typeVariables = Arrays.stream(type1.getGenericDeclaration().getTypeParameters()).map(TypeVariable::getName).collect(Collectors.toCollection(TreeSet::new));
        } else if (type instanceof ParameterizedType) {
            ParameterizedType type1 = (ParameterizedType) type;
            typeVariables = Arrays.stream(type1.getActualTypeArguments()).map(Type::getTypeName).collect(Collectors.toCollection(TreeSet::new));
        }
        String returnType = type.getTypeName();
        if (typeVariables != null) {
            List<String> typesList = new ArrayList<>(GeneratorUtil.getTypesList(returnType));
            for (String typeVariable : typeVariables) {
                if (Objects.equals(returnType, typeVariable.trim())) {
                    switch (typeVariable) {
                        case "D":
                            return dtoType.getName();
                        case "R":
                            return referenceType.getName();
                        case "T":
                            return daoName;
                        case "E":
                            return outputInterface.getQualifiedName();

                    }
                }
                if (typesList.contains(typeVariable)) {
                    switch (typeVariable) {
                        case "D":
                            returnType = replaceType(returnType, typesList, typeVariable, dtoType.getName());
                            break;
                        case "R":
                            returnType = replaceType(returnType, typesList, typeVariable, (doReference ? referenceType.getName() : "?"));
                            break;
                        case "T":
                            returnType = replaceType(returnType, typesList, typeVariable, daoName);
                            break;
                        case "E":
                            returnType = replaceType(returnType, typesList, typeVariable, outputInterface.getQualifiedName());
                            break;
                    }
                }
            }
        }
        return returnType;
    }

    private String replaceType(String returnType, List<String> typesList, String typeVariable, String name) {
        StringBuilder result = new StringBuilder(returnType.substring(0, returnType.indexOf('<') + 1));
        Iterator<String> restingParts = Arrays.asList(returnType.trim().substring(result.length(), returnType.length() - 1).split("\\s*,\\s*")).iterator();
        while (restingParts.hasNext()) {
            String restingPart = restingParts.next();
            if (restingPart.trim().equals(typeVariable)) {
                result.append(name);
                typesList.remove(typeVariable);
            } else {
                result.append(restingPart);
            }
            if (restingParts.hasNext()) {
                result.append(" ,");
            }
        }
        result.append(">");
        return result.toString();
    }

    private void generateJavaBeanMethods() {

        // add JavaBeanDefinition constant
        addImport(outputInterface, JavaBeanDefinition.class);
        addImport(outputInterface, JavaBeanDefinitionStore.class);
        addImport(outputInterface, IllegalStateException.class);
        addConstant(outputInterface, "JAVA_BEAN_DEFINITION", JavaBeanDefinition.class.getSimpleName(), String.format("JavaBeanDefinitionStore.getDefinition(%s.class).orElseThrow(IllegalStateException::new)", outputInterface.getName()), ObjectModelJavaModifier.PUBLIC);

        // add JavaBean method
        ObjectModelOperation operation = addOperation(outputInterface, "javaBeanDefinition", JavaBeanDefinition.class.getSimpleName(), ObjectModelJavaModifier.DEFAULT);
        setOperationBody(operation, ""+"\n"
+"            return JAVA_BEAN_DEFINITION;\n"
+"        ");
        addAnnotation(outputInterface, operation, Override.class);
    }

    private void generateSpiDelegateMethods(boolean referential, String daoName) {

        List<Method> methods = Arrays.stream((referential ? ReferentialDtoEntityContext.class : DataDtoEntityContext.class).getMethods()).filter(method -> {
            int modifiers = method.getModifiers();
            if (Modifier.isStatic(modifiers) || method.isSynthetic()) {
                return false;
            }
            String methodName = method.getName();
            return !excludedMethodNames.contains(methodName);
        }).collect(Collectors.toList());

        for (Method method : methods) {
            String methodName = method.getName();
            String returnType = getTypeGenericBounds(method.getGenericReturnType(), daoName);
            addImport(outputInterface, returnType);
            ObjectModelOperation operation = addOperation(outputInterface, methodName, returnType, ObjectModelJavaModifier.STATIC);
            for (Parameter parameter : method.getParameters()) {
                String parameterName = parameter.getName();
                String parameterType = getTypeGenericBounds(parameter.getParameterizedType(), daoName);
                addImport(outputInterface, parameterType);
                addParameter(operation, parameterType, parameterName);
            }

            boolean toReference = methodName.equals("toReference") && method.getParameters().length == 2 && method.getParameters()[1].getName().equals("entity");
            if (toReference) {
                @SuppressWarnings("unused") String parameterName1 = method.getParameters()[0].getName();
                @SuppressWarnings("unused") String parameterName2 = method.getParameters()[1].getName();
                setOperationBody(operation, ""+"\n"
+"        return "+parameterName2+".toReference("+parameterName1+");\n"
+"    "
                );
                continue;
            }
            StringBuilder content = new StringBuilder();
            if (!void.class.equals(method.getReturnType())) {
                content.append(""+"\n"
+"         return "
                );
            }

            content.append(""+"SPI."+methodName+"("
            );

            Iterator<Parameter> iterator = Arrays.asList(method.getParameters()).iterator();
            while (iterator.hasNext()) {
                Parameter parameter = iterator.next();
                @SuppressWarnings("unused") String parameterName = parameter.getName();
                content.append(""+""+parameterName+""
                );
                if (iterator.hasNext()) {
                    content.append(""+", "
                    );
                }
            }
            content.append(""+");\n"
+"    "
            );
            setOperationBody(operation, content.toString());

        }

    }

    private Set<String> collectAllInterfaces(ObjectModelClassifier classifier) {
        Set<String> result = new LinkedHashSet<>();
        collectAllInterfaces(classifier, false, result);
        return result;
    }

    private void collectAllInterfaces(ObjectModelClassifier classifier, boolean collectSuperClasses, Set<String> result) {
        if (collectSuperClasses && classifier instanceof ObjectModelClass) {
            ObjectModelClass classifier1 = (ObjectModelClass) classifier;
            for (ObjectModelClass superclass : classifier1.getSuperclasses()) {
                result.add(superclass.getQualifiedName());
                collectAllInterfaces(superclass, true, result);
            }

        }
        for (ObjectModelInterface anInterface : classifier.getInterfaces()) {
            result.add(anInterface.getQualifiedName());
            ObjectModelClassifier classifier1 = model.getClassifier(anInterface.getQualifiedName());
            if (classifier1 != null) {
                collectAllInterfaces(classifier1, true, result);
            }
        }

    }

    protected void addInterface(List<String> interfaceAlreadyDone,
                                ObjectModelClassifier output,
                                String qualifiedName) {
        if (!interfaceAlreadyDone.contains(qualifiedName)) {

            // add it to output
            addInterface(output, qualifiedName);

        }
    }

    @Override
    public ObjectModelClass createClass(String className, String packageName) {
        return super.createClass(className, packageName);
    }

    @Override
    public void addImport(ObjectModelClass output, Class<?> type) {
        super.addImport(output, type);
    }

    @Override
    public void addImport(ObjectModelClass output, String type) {
        super.addImport(output, type);
    }

    @Override
    public void setSuperClass(ObjectModelClass output, Class<?> superClass) {
        super.setSuperClass(output, superClass);
    }

    @Override
    public ObjectModelAnnotation addAnnotation(ObjectModelClass output, ObjectModelElement output1, Class<?> annotationType) {
        return super.addAnnotation(output, output1, annotationType);
    }

    @Override
    public void addAnnotationParameter(ObjectModelClass output, ObjectModelAnnotation annotation, String name, String value) {
        super.addAnnotationParameter(output, annotation, name, value);
    }

    @Override
    public ObjectModelOperation addConstructor(ObjectModelClass output, ObjectModelJavaModifier modifiers) {
        return super.addConstructor(output, modifiers);
    }

    @Override
    public void setOperationBody(ObjectModelOperation constructor, String body) {
        super.setOperationBody(constructor, body);
    }

    @Override
    public ObjectModelParameter addParameter(ObjectModelOperation operation, Class<?> type, String name) {
        return super.addParameter(operation, type, name);
    }

    public ObjectModelOperation addOperation(ObjectModelClassifier classifier, String name, Class<?> type, ObjectModelModifier... modifiers) {
        return super.addOperation(classifier, name, type, modifiers);
    }

    @Override
    public ObjectModelAttribute addAttribute(ObjectModelClassifier classifier, String name, String type, String value, ObjectModelModifier... modifiers) {
        return super.addAttribute(classifier, name, type, value, modifiers);
    }
}
