package fr.ird.observe.toolkit.templates.entity;

/*-
 * #%L
 * ObServe Toolkit :: Templates
 * %%
 * Copyright (C) 2017 - 2021 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import fr.ird.observe.toolkit.templates.ToolkitTagValues;
import org.codehaus.plexus.component.annotations.Component;
import org.nuiton.eugene.GeneratorUtil;
import org.nuiton.eugene.Template;
import org.nuiton.eugene.models.object.ObjectModel;
import org.nuiton.eugene.models.object.ObjectModelAttribute;
import org.nuiton.eugene.models.object.ObjectModelClass;
import org.nuiton.eugene.models.object.ObjectModelClassifier;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.topia.persistence.metadata.TopiaMetadataAssociation;
import org.nuiton.topia.persistence.metadata.TopiaMetadataEntity;
import org.nuiton.topia.persistence.metadata.TopiaMetadataEntityPath;
import org.nuiton.topia.persistence.metadata.TopiaMetadataLink;
import org.nuiton.topia.persistence.metadata.TopiaMetadataModel;
import org.nuiton.topia.persistence.metadata.TopiaMetadataModelPaths;
import org.nuiton.topia.persistence.metadata.TopiaMetadataReverseAssociation;
import org.nuiton.topia.templates.EntityHibernateMappingTransformer;
import org.nuiton.topia.templates.HibernateClassContext;
import org.nuiton.topia.templates.TopiaMetadataModelBuilder;

import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;




/**
 * Created on 27/04/2021.
 *
 * @author Tony Chemit - dev@tchemit.fr
 * @since 5.0.19
 */
@Component(role = Template.class, hint = "fr.ird.observe.toolkit.templates.entity.ObserveEntityHibernateMappingTransformer")
public class ObserveEntityHibernateMappingTransformer extends EntityHibernateMappingTransformer {

    private TopiaMetadataModel metadataModel;
    private TopiaMetadataModelPaths allPaths;
    private ToolkitTagValues observeTagValues;

    @Override
    public void generateFromModel(Writer output, ObjectModel input) throws IOException {
        metadataModel = TopiaMetadataModelBuilder.build(isVerbose(), input, getTemplateHelper());
        observeTagValues = new ToolkitTagValues();
        metadataModel.setPaths(allPaths = TopiaMetadataModelPaths.create(metadataModel));
        super.generateFromModel(output, input);
    }

    @Override
    protected void generateSqlQueries(Writer output, HibernateClassContext classContext) throws IOException {

        ObjectModelClass input = classContext.getInput();
        addEntityToIdsProjection(output, input);
        addEntityToIdEqualsProjection(output, input);
        addEntityToIdInProjection(output, input);
        addEntityToIdBeforeProjection(output, input);
        addEntityToIdAfterProjection(output, input);
        addEntityToMapProjection(output, classContext);

        TopiaMetadataEntity entity = metadataModel.getEntity(getTemplateHelper().getEntityEnumLiteralName(input));

        String navigationParentTagValue = observeTagValues.getNavigationParentTagValue(input);
        if (navigationParentTagValue != null) {
            getLog().info(String.format("Add entry point parent query on %s", entity.getFullyQualifiedName()));
            TopiaMetadataEntity owner = metadataModel.getEntity("common_Program");
            addGetMultipleParentEntity(output, input.getQualifiedName(), entity, owner, navigationParentTagValue, navigationParentTagValue, true);
            return;
        }
        if (!allPaths.containsKey(entity)) {
            return;
        }

        List<TopiaMetadataEntityPath> paths = new ArrayList<>(allPaths.getEntityPath(entity));
        if (paths.size() == 2) {
            int match = 0;
            TopiaMetadataEntityPath mainPath = null;
            for (TopiaMetadataEntityPath path : paths) {
                TopiaMetadataLink lastLink = path.getLastLink();
                if (lastLink instanceof TopiaMetadataAssociation) {
                    mainPath = path;
                    match++;
                }
            }
            if (match == 2) {
                // no need two paths
                paths.remove(1);
            } else {
                for (TopiaMetadataEntityPath path : paths) {
                    TopiaMetadataLink lastLink = path.getLastLink();
                    String property = lastLink.getTargetPropertyName();
                    TopiaMetadataEntity owner = lastLink.getOwner();
                    if (lastLink instanceof TopiaMetadataReverseAssociation) {
                        getLog().info(String.format("Add single parent query on %s", lastLink));
                        addGetSingleParentEntity(output, input.getQualifiedName(), lastLink.getOwner(), lastLink.getTargetPropertyName(), property, mainPath == path);
                    } else {
                        getLog().info(String.format("Add multiple parent query on %s", lastLink));
                        addGetMultipleParentEntity(output, input.getQualifiedName(), entity, owner, lastLink.getTargetPropertyName(), owner.getDbTableName(), mainPath == path);
                    }
                }
                return;
            }
        }

        if (paths.size() == 1) {
            for (TopiaMetadataEntityPath path : paths) {
                int linksSize = path.getLinksSize();
                if (linksSize == 0) {
                    continue;
                }
                TopiaMetadataLink lastLink = path.getLastLink();
                String property = lastLink.getTargetPropertyName();
                TopiaMetadataEntity owner = lastLink.getOwner();
                if (lastLink instanceof TopiaMetadataReverseAssociation) {
                    getLog().info(String.format("Add single parent query on %s", lastLink));
                    addGetSingleParentEntity(output, input.getQualifiedName(), lastLink.getOwner(), lastLink.getTargetPropertyName(), property, true);
                } else {
                    getLog().info(String.format("Add multiple parent query on %s", lastLink));
                    addGetMultipleParentEntity(output, input.getQualifiedName(), entity, owner, lastLink.getTargetPropertyName(), owner.getDbTableName(), true);
                }
            }
        } else {
            throw new IllegalStateException("Can't manage three paths to data for " + input.getQualifiedName());
        }
    }

    private void addGetMultipleParentEntity(Writer output, String input, TopiaMetadataEntity table, TopiaMetadataEntity parentTable, String entityProperty, String property, boolean main) throws IOException {
        String query = String.format("select '%s', p.topiaId, p.lastUpdateDate from %s.%s p inner join %s.%s e on e.%s = p.topiaId where e.topiaId = ?", entityProperty, parentTable.getDbSchemaName(), parentTable.getDbTableName(), table.getDbSchemaName(), table.getDbTableName(), property);
//        String query = String.format("select '%s', e.%s, e.lastUpdateDate from %s.%s as e where e.topiaId = ?", entityProperty, property, table.getDbSchemaName(), table.getDbTableName());
        String queryName = input + "::getParentId" + (main ? "" : "2");
        boolean added = addSqlQuery(output, queryName, query, " read-only=\"true\"");
        if (!added) {
            throw new IllegalStateException(String.format("Can't add query: %s", queryName));
        }
    }

    private void addGetSingleParentEntity(Writer output, String input, TopiaMetadataEntity table, String entityProperty, String property, boolean main) throws IOException {
        String query = String.format("select '%s', p.topiaId, p.lastUpdateDate from %s.%s as p where p.%s = ?", entityProperty, table.getDbSchemaName(), table.getDbTableName(), property);
        String queryName = input + "::getParentId" + (main ? "" : "2");
        boolean added = addSqlQuery(output, queryName, query, " read-only=\"true\"");
        if (!added) {
            throw new IllegalStateException(String.format("Can't add query: %s", queryName));
        }
    }

    private void addEntityToIdsProjection(Writer output, ObjectModelClass input) throws IOException {
        String query = String.format("select new fr.ird.observe.dto.ToolkitIdBean(id, lastUpdateDate)\n" +
                                             "    from %s as e", input.getQualifiedName() + "Impl");
        addQuery(output, input.getQualifiedName() + "::id::all", query, " read-only=\"true\"");
    }

    private void addEntityToIdEqualsProjection(Writer output, ObjectModelClass input) throws IOException {
        String query = String.format("select new fr.ird.observe.dto.ToolkitIdBean(id, lastUpdateDate)\n" +
                                             "    from %s as e\n" +
                                             "    where e.id = ?1", input.getQualifiedName() + "Impl");
        addQuery(output, input.getQualifiedName() + "::id::equals", query, " read-only=\"true\"");
    }

    private void addEntityToIdInProjection(Writer output, ObjectModelClass input) throws IOException {
        String query = String.format("select new fr.ird.observe.dto.ToolkitIdBean(id, lastUpdateDate)\n" +
                                             "    from %s as e\n" +
                                             "    where e.id in ( ?1 )", input.getQualifiedName() + "Impl");
        addQuery(output, input.getQualifiedName() + "::id::in", query, " read-only=\"true\"");
    }

    private void addEntityToIdBeforeProjection(Writer output, ObjectModelClass input) throws IOException {
        String query = String.format("select new fr.ird.observe.dto.ToolkitIdBean(id, lastUpdateDate)\n" +
                                             "    from %s as e\n" +
                                             "    where e.lastUpdateDate <= ?1", input.getQualifiedName() + "Impl");
        addQuery(output, input.getQualifiedName() + "::id::before", query, " read-only=\"true\"");
    }

    private void addEntityToIdAfterProjection(Writer output, ObjectModelClass input) throws IOException {
        String query = String.format("select new fr.ird.observe.dto.ToolkitIdBean(id, lastUpdateDate)\n" +
                                             "    from %s as e\n" +
                                             "    where e.lastUpdateDate > ?1", input.getQualifiedName() + "Impl");
        addQuery(output, input.getQualifiedName() + "::id::after", query, " read-only=\"true\"");
    }

    private void addEntityToMapProjection(Writer output, HibernateClassContext classContext) throws IOException {

        ObjectModelClass input = classContext.getInput();
        Set<String> allDbColumnNames = new LinkedHashSet<>();
        allDbColumnNames.add("id as _" + TopiaEntity.PROPERTY_TOPIA_ID);
        allDbColumnNames.add(TopiaEntity.PROPERTY_TOPIA_CREATE_DATE + " as _" + TopiaEntity.PROPERTY_TOPIA_CREATE_DATE);
        allDbColumnNames.add(TopiaEntity.PROPERTY_TOPIA_VERSION + " as _" + TopiaEntity.PROPERTY_TOPIA_VERSION);
        List<ObjectModelAttribute> naturalAttributes = classContext.getNaturalAttributes();
        naturalAttributes.forEach(a -> addAttribute(a, allDbColumnNames));
        List<ObjectModelAttribute> nonNaturalAttributes = classContext.getNoneNaturalAttributes();
        nonNaturalAttributes.forEach(a -> addAttribute(a, allDbColumnNames));
        String constructorParameters = allDbColumnNames.stream().map(t -> "e." + t).collect(Collectors.joining(",\n     "));
        String fromEntityName = input.getQualifiedName() + "Impl";
        String queryNamePrefix = input.getQualifiedName() + "::map::";
        String extraParameters = " read-only=\"true\"";

        String queryAll = String.format("select new map(\n     %s)\n    from %s as e", constructorParameters, fromEntityName);
        addQuery(output, queryNamePrefix + "all", queryAll, extraParameters);

        String queryOne = String.format("select new map(\n     %s)\n    from %s as e\n    where e.id = ?1", constructorParameters, fromEntityName);
        addQuery(output, queryNamePrefix + "one", queryOne, extraParameters);

        String querySome = String.format("select new map(\n     %s)\n    from %s as e\n    where e.id in ( ?1 )", constructorParameters, fromEntityName);
        addQuery(output, queryNamePrefix + "some", querySome, extraParameters);
    }

    protected void addAttribute(ObjectModelAttribute attribute, Set<String> allDbColumnNames) {
        ObjectModelClassifier classifier = attribute.getClassifier();
        if (GeneratorUtil.isNMultiplicity(attribute)) {
            return;
        }
        if (!attribute.isNavigable()) {
            return;
        }
        String name = attribute.getName();
        if (classifier != null && templateHelper.isEntity(classifier)) {
            allDbColumnNames.add(name + ".id as _" + name);
            return;
        }
        allDbColumnNames.add(name + " as _" + name);
    }
}
