package fr.ird.observe.toolkit.templates.io;

/*-
 * #%L
 * ObServe Toolkit :: Templates
 * %%
 * Copyright (C) 2017 - 2021 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import fr.ird.observe.toolkit.templates.ToolkitTagValues;
import io.ultreia.java4all.util.SortedProperties;
import org.nuiton.eugene.EugeneCoreTagValues;
import org.nuiton.eugene.LogProxy;
import org.nuiton.eugene.models.extension.io.ModelExtensionReader;
import org.nuiton.eugene.models.extension.tagvalue.InvalidStereotypeSyntaxException;
import org.nuiton.eugene.models.extension.tagvalue.InvalidTagValueSyntaxException;
import org.nuiton.eugene.models.object.ObjectModel;
import org.nuiton.eugene.models.object.ObjectModelAttribute;
import org.nuiton.eugene.models.object.ObjectModelClassifier;
import org.nuiton.eugene.models.object.xml.ObjectModelAttributeImpl;
import org.nuiton.eugene.models.object.xml.ObjectModelClassifierImpl;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Objects;
import java.util.Properties;
import java.util.function.Function;

/**
 * Created on 28/01/2021.
 *
 * @author Tony Chemit - dev@tchemit.fr
 * @since 8.0.5
 */
public class TagValues {

    private final ObjectModel model;
    private final String classifier;
    private final LogProxy log;

    public static String getResourceFilePath(ObjectModel model, String classifier) {
        return "models/" + model.getName() + "/" + classifier + ".properties";
    }

    public static String getResourcePath(ObjectModel model, String classifier) {
        return "models/" + model.getName() + "/" + classifier + "/";
    }

    public static void loadTagValuesFile(SortedProperties toLoad, URL resource, Function<String, String> keyExtractor) throws IOException {
        Properties incoming = new Properties();
        try (InputStream inputStream = resource.openStream()) {
            incoming.load(inputStream);
        }
        for (String key : incoming.stringPropertyNames()) {
            String fullKey = keyExtractor.apply(key);
            String value = incoming.getProperty(key);
            toLoad.setProperty(fullKey, value);
        }
    }

    public TagValues(ObjectModel model, String classifier, LogProxy log) {
        this.model = model;
        this.classifier = classifier;
        this.log = log;
    }

    public void load(ClassLoader classLoader, Path outputDirectory, boolean verbose) throws InvalidTagValueSyntaxException, IOException, InvalidStereotypeSyntaxException {
        Path tmpFile = outputDirectory.resolve(String.format("%s-%s-tagValues.properties", classifier, System.nanoTime()));

        String resourcesPath = TagValues.getResourceFilePath(model, Objects.requireNonNull(classifier));
        URL resource = classLoader.getResource(resourcesPath);
        Properties loadedTagValues = new Properties();
        if (resource != null) {
            try (InputStreamReader reader = new InputStreamReader(resource.openStream())) {
                loadedTagValues.load(reader);
            }
        }
        log.info(String.format("[%s] Load %d tag-value(s).", classifier, loadedTagValues.size()));
        try (BufferedWriter writer = Files.newBufferedWriter(tmpFile)) {
            loadedTagValues.store(writer, String.format("Generated by %s", getClass().getName()));
        }
        log.info(String.format("[%s] Inject in model.", classifier));
        ModelExtensionReader reader = new ModelExtensionReader(verbose, true, model) {
            //FIXME Move that in eugene the default-value should be loaded from tagValue
            @Override
            public boolean onAttributeTagValueFound(String className, String attributeName, String tag, String value) {
                boolean success = super.onAttributeTagValueFound(className, attributeName, tag, value);
                if (success && tag.equals(EugeneCoreTagValues.Store.defaultValue.name())) {
                    ObjectModelClassifier omc = getClassifier(className);
                    ObjectModelAttribute attribute = getAttribute(omc, attributeName);
                    ((ObjectModelAttributeImpl) attribute).setDefaultValue(value);
                }
                if (success && tag.equals(EugeneCoreTagValues.Store.unique.name())) {
                    ObjectModelClassifier omc = getClassifier(className);
                    ObjectModelAttribute attribute = getAttribute(omc, attributeName);
                    ((ObjectModelAttributeImpl) attribute).setUnique(Boolean.parseBoolean(value));
                }
                if (!success) {
                    ObjectModelAttribute attribute = getInheritedAttribute(getClassifier(className), attributeName);
                    if (attribute != null) {
                        // just add for the moment in nice tagValues store, will use it asap
                        log.debug(String.format("Add extra tag-value: %s.%s → %s", className, attributeName, value));
                        tagValuesStore.setAttributeTagValue(className, attributeName, tag, value);
                        success = true;
                    }
                }
                return success;
            }

            @Override
            public boolean onClassTagValueFound(String className, String tag, String value) {
                boolean success = super.onClassTagValueFound(className, tag, value);
                if (success && tag.equals(ToolkitTagValues.Store.Static.name())) {
                    ObjectModelClassifier omc = getClassifier(className);
                    ((ObjectModelClassifierImpl) omc).setStatic(Boolean.parseBoolean(value));
                }
                return success;
            }

            protected ObjectModelAttribute getInheritedAttribute(ObjectModelClassifier clazz, String name) {
                // we can ask for a attribute some else on classifier, since tag value are no more attached to an object model
                // element, we can do this :)
                return clazz.getAllOtherAttributes().stream().filter(a -> name.equals(a.getName())).findAny().orElse(null);
            }
        };
        reader.read(tmpFile.toFile());
    }
}
