package fr.ird.observe.toolkit.templates.navigation;

/*-
 * #%L
 * ObServe Toolkit :: Templates
 * %%
 * Copyright (C) 2017 - 2021 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.auto.service.AutoService;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ImmutableList;
import fr.ird.observe.dto.referential.ReferentialDto;
import fr.ird.observe.spi.ProjectPackagesDefinition;
import fr.ird.observe.spi.module.BusinessModule;
import fr.ird.observe.spi.module.BusinessProject;
import fr.ird.observe.spi.module.BusinessReferentialPackage;
import fr.ird.observe.spi.module.BusinessSubModule;
import fr.ird.observe.toolkit.navigation.spi.NavigationNodeCapability;
import fr.ird.observe.toolkit.navigation.spi.NavigationNodeDescriptor;
import fr.ird.observe.toolkit.navigation.spi.NavigationNodeDescriptorVisitor;
import fr.ird.observe.toolkit.navigation.spi.NavigationNodeType;
import fr.ird.observe.toolkit.templates.ToolkitTagValues;
import fr.ird.observe.toolkit.templates.TemplateContract;
import io.ultreia.java4all.lang.Objects2;
import io.ultreia.java4all.util.ServiceLoaders;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.nuiton.eugene.GeneratorUtil;
import org.nuiton.eugene.java.ObjectModelTransformerToJava;
import org.nuiton.eugene.java.extension.ImportsManager;
import org.nuiton.eugene.java.extension.ObjectModelAnnotation;
import org.nuiton.eugene.models.object.ObjectModel;
import org.nuiton.eugene.models.object.ObjectModelAttribute;
import org.nuiton.eugene.models.object.ObjectModelClass;
import org.nuiton.eugene.models.object.ObjectModelClassifier;
import org.nuiton.eugene.models.object.ObjectModelElement;
import org.nuiton.eugene.models.object.ObjectModelJavaModifier;
import org.nuiton.eugene.models.object.ObjectModelModifier;
import org.nuiton.eugene.models.object.ObjectModelOperation;
import org.nuiton.eugene.models.object.ObjectModelPackage;
import org.nuiton.eugene.models.object.ObjectModelParameter;
import org.nuiton.eugene.models.object.xml.ObjectModelAttributeImpl;
import org.nuiton.topia.templates.TopiaExtensionTagValues;
import org.nuiton.topia.templates.TopiaTemplateHelperExtension;

import java.beans.Introspector;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;




/**
 * Created on 29/03/2021.
 *
 * @author Tony Chemit - dev@tchemit.fr
 * @since 5.0.17
 */
public class GenerateNavigationNodeDescriptors extends ObjectModelTransformerToJava implements TemplateContract {

    private final ToolkitTagValues observeTagValues = new ToolkitTagValues();
    private final TopiaExtensionTagValues topiaExtensionTagValues = new TopiaExtensionTagValues();
    private final ArrayListMultimap<Pair<String, String>, ObjectModelAttribute> mapping = ArrayListMultimap.create();
    private final List<Pair<String, String>> mappingEmpty = new LinkedList<>();
    private ProjectPackagesDefinition def;
    private List<String> entityClassesWithNode;
    private LinkedList<String> entityClassesOpen;

    @Override
    public void transformFromModel(ObjectModel model) {
        BusinessProject businessProject = ServiceLoaders.loadUniqueService(BusinessProject.class);
        super.transformFromModel(model);
        def = ProjectPackagesDefinition.of(getClassLoader());
        TopiaTemplateHelperExtension templateHelper = new TopiaTemplateHelperExtension(model);
        List<ObjectModelClass> entityClasses = templateHelper.getEntityClasses(model, false);
        entityClassesWithNode = entityClasses.stream().filter(entityClass -> observeTagValues.getNavigationNodeTypeTagValue(entityClass, null) != null).map(ObjectModelClass::getQualifiedName).collect(Collectors.toCollection(LinkedList::new));
        entityClassesOpen = entityClasses.stream().filter(entityClass -> Objects.equals(observeTagValues.getNavigationNodeTypeTagValue(entityClass, null), "Open")).map(ObjectModelClass::getQualifiedName).collect(Collectors.toCollection(LinkedList::new));
        LinkedList<String> entityClassesEntryPoints = entityClasses.stream().filter(topiaExtensionTagValues::isEntryPoint).map(ObjectModelClass::getQualifiedName).collect(Collectors.toCollection(LinkedList::new));

        String rootPackageName = toNavigationPackage("");
        String rootClazzName = toNavigationNodeName("Root");

        for (String inputClazzName : entityClassesEntryPoints) {
            ObjectModelClass input = model.getClass(inputClazzName);
            String navigationParentTagValue = Objects.requireNonNull(observeTagValues.getNavigationParentTagValue(input));
            ObjectModelAttribute parentAttribute = Objects.requireNonNull(input.getAttribute(navigationParentTagValue));
            ObjectModelPackage clazzPackage = model.getPackage(input);

            String parentPackageName = clazzPackage.getName().substring(0, clazzPackage.getName().lastIndexOf('.'));
            String moduleName = parentPackageName.substring(parentPackageName.lastIndexOf('.') + 1);
            ObjectModelAttribute entryPointAttribute = createAttribute(parentAttribute.getType(), moduleName, -1, NavigationNodeType.OpenList);
            entryPointAttribute.addTagValue("childrenPackage", input.getPackageName());
            entryPointAttribute.addTagValue("childrenName", parentAttribute.getName());
            registerMapping(rootPackageName, rootClazzName, entryPointAttribute);
            ObjectModelAttribute subEntryPointAttribute = createAttribute(input.getQualifiedName(), "children", -1, NavigationNodeType.OpenList);
            registerMapping(toNavigationPackage(input.getPackageName()), toNavigationNodeName(parentAttribute.getName()), subEntryPointAttribute);
            subEntryPointAttribute.addTagValue("childrenPackage", input.getPackageName());
            subEntryPointAttribute.addTagValue("childrenName", input.getName());

            scan(input);
        }

        for (BusinessModule module : businessProject.getModules()) {
            for (BusinessSubModule subModule : module.getSubModules()) {
                BusinessReferentialPackage referentialPackage = subModule.getReferentialPackage().orElse(null);
                if (referentialPackage != null) {
                    ObjectModelAttribute attribute = createAttribute(referentialPackage.getPackageName(), module.getName() + StringUtils.capitalize(subModule.getName()), 0, NavigationNodeType.ReferentialPackage);
                    registerMapping(rootPackageName, rootClazzName, attribute);
                    String homePackage = referentialPackage.getPackageName().replace(".dto", ".navigation.descriptor");
                    List<ObjectModelAttribute> homeAttributes = new LinkedList<>();
                    for (Class<? extends ReferentialDto> type : referentialPackage.getTypes()) {
                        ObjectModelAttribute attribute2 = createAttribute(type.getName(), type.getSimpleName().replace("Dto", ""), 0, NavigationNodeType.ReferentialType);
                        homeAttributes.add(attribute2);
                        registerMapping(homePackage, toNavigationNodeName(attribute2.getName()), null);
                    }
                    registerMappings(homePackage, toNavigationNodeName("ReferentialPackage"), homeAttributes);
                }
            }
        }

        getLog().info(String.format("Found %d container node(s) to generate.", mapping.asMap().size()));
        for (Map.Entry<Pair<String, String>, Collection<ObjectModelAttribute>> entry : mapping.asMap().entrySet()) {
            Pair<String, String> coordinate = entry.getKey();
            Collection<ObjectModelAttribute> attributes = entry.getValue();
            doGenerate(coordinate.getKey(), coordinate.getValue(), attributes);
        }
        getLog().info(String.format("Found %d leaf node(s) to generate.", mappingEmpty.size()));
        for (Pair<String, String> coordinate : mappingEmpty) {
            doGenerate(coordinate.getKey(), coordinate.getValue(), Collections.emptyList());
        }
    }

    private void scan(ObjectModelClass input) {
        ObjectModelPackage clazzPackage = model.getPackage(input);
        List<ObjectModelAttribute> attributes = new LinkedList<>();
        String navigationNodeExtraTypeTagValue = observeTagValues.getNavigationNodeExtraTypeTagValue(input);
        if (navigationNodeExtraTypeTagValue != null) {
            String[] types = navigationNodeExtraTypeTagValue.split("\\s*,\\s*");
            for (String typeStr : types) {
                getLog().info(String.format("[%-70s] Add extra type %s.", input.getQualifiedName(), typeStr));
                Class<?> type = Objects2.forName(typeStr);
                String dtoSimpleName = type.getSimpleName().replace("Dto", "");
                ObjectModelAttribute attribute2 = createAttribute(type.getName().replace("Dto", ""), dtoSimpleName, 0, NavigationNodeType.Simple);
                attributes.add(attribute2);
                registerMapping(toNavigationPackage(input.getPackageName()), toNavigationNodeName(dtoSimpleName), null);
            }
        }
        attributes.addAll(input.getAttributes().stream().filter(
                a -> a.isNavigable()
                        && entityClassesWithNode.contains(a.getType())
                        && !topiaExtensionTagValues.isSkipModelNavigation(clazzPackage, input, a)
        ).collect(Collectors.toList()));
        String packageName = toNavigationPackage(input.getPackageName());
        String clazzName = toNavigationNodeName(input.getName());
        registerMappings(packageName, clazzName, attributes);
        for (ObjectModelAttribute attribute : attributes) {
            String navigationNodeTypeTagValue = Objects.requireNonNull(observeTagValues.getNavigationNodeTypeTagValue(null, attribute));
            if (NavigationNodeType.Simple.name().equals(navigationNodeTypeTagValue)) {
                continue;
            }
            ObjectModelClass dependency = model.getClass(attribute.getType());

            navigationNodeTypeTagValue = Objects.requireNonNull(observeTagValues.getNavigationNodeTypeTagValue(dependency, attribute));
            if (entityClassesOpen.contains(dependency.getQualifiedName()) && NavigationNodeType.Edit.name().equals(navigationNodeTypeTagValue)) {
                // special case: use a open data as a edit one
                // must declare a new type
                getLog().info(String.format("Found a special Open → Edit node on %s.%s", input.getName(), attribute.getName()));
                registerMapping(toNavigationPackage(dependency.getPackageName()), toNavigationNodeName(input.getName() + StringUtils.capitalize(dependency.getName())), null);
            }
            scan(dependency);
        }
    }

    private void registerMapping(String packageName, String clazzName, ObjectModelAttribute attribute) {
        registerMappings(packageName, clazzName, attribute == null ? Collections.emptyList() : Collections.singletonList(attribute));
    }

    private void registerMappings(String packageName, String clazzName, List<ObjectModelAttribute> attributes) {
        if (Objects.requireNonNull(attributes).isEmpty()) {
            mappingEmpty.add(Pair.of(packageName, clazzName));
        } else {
            mapping.putAll(Pair.of(packageName, clazzName), attributes);
        }
    }

    private void doGenerate(String packageName, String clazzName, Collection<ObjectModelAttribute> attributes) {
        generateGenerated(packageName, clazzName, attributes);
        if (!getResourcesHelper().isJavaFileInClassPath(packageName + "." + clazzName)) {
            generateConcrete(packageName, clazzName);
        }
    }

    private ObjectModelAttribute createAttribute(String type, String name, int multiplicity, NavigationNodeType capabilityType) {
        ObjectModelAttributeImpl attribute = new ObjectModelAttributeImpl();
        attribute.setNavigable(true);
        attribute.setType(type);
        attribute.setName(Introspector.decapitalize(name));
        attribute.setMaxMultiplicity(multiplicity);
        if (capabilityType != null) {
            attribute.addTagValue("navigationNodeType", capabilityType.name());
        }
        return attribute;
    }

    @SuppressWarnings({"unused", "StringOperationCanBeSimplified", "MismatchedQueryAndUpdateOfStringBuilder", "UnusedAssignment"})
    private void generateGenerated(String packageName, String clazzName, Collection<ObjectModelAttribute> attributes) {
        String generatedClazzName = "Generated" + clazzName;
        ObjectModelClass output = createAbstractClass(generatedClazzName, packageName);
        setSuperClass(output, NavigationNodeDescriptor.class);
        addStaticFactory(output, clazzName);
        if (clazzName.startsWith("Root")) {
            ObjectModelOperation getRootDescriptor = addOperation(output, "getRootDescriptor", clazzName, ObjectModelJavaModifier.PUBLIC, ObjectModelJavaModifier.STATIC);
            setOperationBody(getRootDescriptor, ""+"\n"
+"        return NavigationNodeDescriptor.getRootDescriptor("+clazzName+".class);\n"
+"    ");
        }
        ObjectModelOperation constructor = addConstructor(output, ObjectModelJavaModifier.PROTECTED);
        ObjectModelOperation loadCapabilitiesOperation = addOperation(output, "loadCapabilities", "ImmutableList<NavigationNodeCapability<?>>", ObjectModelJavaModifier.PROTECTED, ObjectModelJavaModifier.FINAL);
        addAnnotation(output, loadCapabilitiesOperation, Override.class);

        ObjectModelOperation acceptOperation = addOperation(output, "accept", "void", ObjectModelJavaModifier.PUBLIC, ObjectModelJavaModifier.FINAL);
        addParameter(acceptOperation, NavigationNodeDescriptorVisitor.class, "visitor");
        addAnnotation(output, acceptOperation, Override.class);

        ImportsManager importManager = getImportManager(output);
        importManager.addImport(NavigationNodeCapability.class);
        importManager.addImport(ImmutableList.class);
        importManager.addExcludedPattern(".+NavigationNodeDescriptor.*$");
        if (isVerbose()) {
            getLog().info(String.format("will generate %s", output.getQualifiedName()));
        }
        StringBuilder loadCapabilitiesContent = new StringBuilder();
        StringBuilder constructorContent = new StringBuilder();
        StringBuilder visitorContent = new StringBuilder();
        int order = -1;
        for (ObjectModelAttribute attribute : attributes) {
            order++;
            String attributeType = attribute.getType();
            String navigationNodeTypeTagValue = observeTagValues.getNavigationNodeTypeTagValue(null, attribute);
            String attrName = attribute.getName();
            boolean multiple = false;
            String type;
            if (NavigationNodeType.ReferentialPackage.name().equals(navigationNodeTypeTagValue)) {
                type = attributeType.replace(".dto", ".navigation.descriptor") + "." + toNavigationNodeName("ReferentialPackage");
            } else if (NavigationNodeType.ReferentialType.name().equals(navigationNodeTypeTagValue)) {
                type = attributeType.replace(".dto", ".navigation.descriptor").replace("Dto", "") + toNavigationNodeName("");
            } else if (NavigationNodeType.Simple.name().equals(navigationNodeTypeTagValue)) {
                type = attributeType.replace(".dto", ".navigation.descriptor").replace("Dto", "") + toNavigationNodeName("");
            } else if (NavigationNodeType.OpenList.name().equals(navigationNodeTypeTagValue)) {
                ObjectModelClass attributeClazz = Objects.requireNonNull(model.getClass(attributeType));
                String childrenPackage = attribute.getTagValue("childrenPackage");
                String childrenName = attribute.getTagValue("childrenName");
                if (childrenName != null) {
                    type = toNavigationPackage(childrenPackage) + "." + toNavigationNodeName(childrenName);
                    multiple = true;
                } else {
                    type = toNavigationNode(attributeClazz);
                    navigationNodeTypeTagValue = Objects.requireNonNull(observeTagValues.getNavigationNodeTypeTagValue(attributeClazz, attribute));
                    multiple = GeneratorUtil.isNMultiplicity(attribute) && !"Table".equals(navigationNodeTypeTagValue);
                }
            } else {

                ObjectModelClass attributeClazz = Objects.requireNonNull(model.getClass(attributeType));
                type = toNavigationNode(attributeClazz);
                navigationNodeTypeTagValue = Objects.requireNonNull(observeTagValues.getNavigationNodeTypeTagValue(attributeClazz, attribute));
                multiple = GeneratorUtil.isNMultiplicity(attribute) && !"Table".equals(navigationNodeTypeTagValue);

                if (entityClassesOpen.contains(attributeType) && NavigationNodeType.Edit.name().equals(navigationNodeTypeTagValue)) {
                    // special case: use a open data as a edit one
                    // must declare a new type
                    type = toNavigationPackage(attributeClazz.getPackageName()) + "." + toNavigationNodeName(clazzName.replace("NavigationNodeDescriptor", "") + StringUtils.capitalize(attributeClazz.getName()));
                }
            }
            constructorContent.append(""+"\n"
+"        this."+attrName+" = NavigationNodeCapability.new"+navigationNodeTypeTagValue+"(this, \""+attrName+"\", "+order+", "+multiple+", "+type+".get());");
            visitorContent.append(""+"\n"
+"        visitor.visit(this."+attrName+"); ");
            loadCapabilitiesContent.append(""+"\n"
+"                .add(this."+attrName+")");
            type = importManager.importAndSimplify(type);
            type = NavigationNodeCapability.class.getSimpleName() + "<" + type + ">";
            addAttribute(output, attrName, type, null, ObjectModelJavaModifier.PRIVATE, ObjectModelJavaModifier.FINAL);
            ObjectModelOperation getOperation =
                    addOperation(output, getJavaBeanMethodName("get", attrName),
                                 type, ObjectModelJavaModifier.PUBLIC, ObjectModelJavaModifier.FINAL);

            setOperationBody(getOperation, ""+"\n"
+"        return this."+attrName+";\n"
+"    ");
        }

        setOperationBody(constructor, ""+""+constructorContent.toString()+"\n"
+"    ");
        setOperationBody(loadCapabilitiesOperation, ""+"\n"
+"        return ImmutableList.<NavigationNodeCapability<?>>builder()"+loadCapabilitiesContent.toString()+"\n"
+"                .build();\n"
+"    ");
        setOperationBody(acceptOperation, ""+"\n"
+"        visitor.start(this);"+visitorContent.toString()+"\n"
+"        visitor.end(this);\n"
+"    ");
    }

    private void generateConcrete(String packageName, String clazzName) {
        ObjectModelClass output = createClass(clazzName, packageName);
        setSuperClass(output, packageName + ".Generated" + clazzName);
        if (clazzName.startsWith("Root")) {
            addConstructor(output, ObjectModelJavaModifier.PUBLIC);
            ObjectModelAnnotation annotation = addAnnotation(output, output, AutoService.class);
            ImportsManager importManager = getImportManager(output);
            addAnnotationClassParameter(importManager, output, annotation, "value", NavigationNodeDescriptor.class);
        } else {
            addConstructor(output, ObjectModelJavaModifier.PACKAGE);
        }
        if (isVerbose()) {
            getLog().info(String.format("will generate %s", output.getQualifiedName()));
        }
    }

    private String toNavigationPackage(String packageName) {
        String relativeEntityPackage = def.getRelativeEntityPackage(packageName);
        return def.getDtoRootPackage().replace(".dto", ".navigation.descriptor") + relativeEntityPackage;
    }

    private String toNavigationNode(ObjectModelClass clazz) {
        return toNavigationPackage(clazz.getPackageName()) + "." + toNavigationNodeName(clazz.getName());
    }

    private String toNavigationNodeName(String prefix) {
        return StringUtils.capitalize(prefix) + "NavigationNodeDescriptor";
    }

    @Override
    public void addImport(ObjectModelClass output, Class<?> type) {
        super.addImport(output, type);
    }

    @Override
    public void addImport(ObjectModelClass output, String type) {
        super.addImport(output, type);
    }

    @Override
    public void setSuperClass(ObjectModelClass output, Class<?> superClass) {
        super.setSuperClass(output, superClass);
    }

    @Override
    public ObjectModelAnnotation addAnnotation(ObjectModelClass output, ObjectModelElement output1, Class<?> annotationType) {
        return super.addAnnotation(output, output1, annotationType);
    }

    @Override
    public void addAnnotationParameter(ObjectModelClass output, ObjectModelAnnotation annotation, String name, String value) {
        super.addAnnotationParameter(output, annotation, name, value);
    }

    @Override
    public ObjectModelOperation addConstructor(ObjectModelClass output, ObjectModelJavaModifier modifiers) {
        return super.addConstructor(output, modifiers);
    }

    @Override
    public ObjectModelOperation addOperation(ObjectModelClassifier classifier, String name, String type, ObjectModelModifier... modifiers) {
        return super.addOperation(classifier, name, type, modifiers);
    }

    @Override
    public ObjectModelClass createClass(String name, String packageName) {
        return super.createClass(name, packageName);
    }

    @Override
    public void setOperationBody(ObjectModelOperation constructor, String body) {
        super.setOperationBody(constructor, body);
    }

    @Override
    public ObjectModelParameter addParameter(ObjectModelOperation operation, Class<?> type, String name) {
        return super.addParameter(operation, type, name);
    }

    public ObjectModelOperation addOperation(ObjectModelClassifier classifier, String name, Class<?> type, ObjectModelModifier... modifiers) {
        return super.addOperation(classifier, name, type, modifiers);
    }

    @Override
    public ObjectModelAttribute addAttribute(ObjectModelClassifier classifier, String name, String type, String value, ObjectModelModifier... modifiers) {
        return super.addAttribute(classifier, name, type, value, modifiers);
    }

}
