package fr.ird.observe.toolkit.templates.navigation;

/*-
 * #%L
 * ObServe Toolkit :: Templates
 * %%
 * Copyright (C) 2017 - 2021 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import fr.ird.observe.dto.VoidDto;
import fr.ird.observe.toolkit.navigation.spi.NavigationNodeCapability;
import fr.ird.observe.toolkit.navigation.spi.NavigationNodeDescriptor;
import fr.ird.observe.toolkit.navigation.spi.NavigationNodeType;
import io.ultreia.java4all.i18n.spi.builder.I18nKeySet;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

import java.beans.Introspector;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;

/**
 * Created on 01/04/2021.
 *
 * @author Tony Chemit - dev@tchemit.fr
 * @since 5.0.17
 */
public class NodeModel implements NavigationNodeType.WithNavigationNodeType {

    final NodeModel parent;
    final NavigationNodeDescriptor descriptor;
    final int level;
    final NavigationNodeType type;
    final String dtoType;
    final String iconPath;
    final String path;
    final String text;
    final List<NodeLinkModel> children = new LinkedList<>();
    final Pair<String, String> gav;

    static String getPackageName(NavigationNodeDescriptor descriptor, String prefix) {
        String result = descriptor.getClass().getPackageName().replace("fr.ird.observe.navigation.descriptor", "");
        if (result.startsWith(".")) {
            result = result.substring(1);
        }
        return prefix + (prefix.isEmpty() || result.isEmpty() ? "" : ".") + result;
    }

    static String getSimpleName(NavigationNodeDescriptor descriptor, String suffix) {
        return descriptor.getClass().getSimpleName().replace("NavigationNodeDescriptor", suffix);
    }

    public static String getPropertyNameFromTarget(NavigationNodeCapability<?> capability) {
        return Introspector.decapitalize(getSimpleName(capability.getTarget(), "SelectNode").replace("SelectNode", ""));
    }

    public static Pair<String, String> referentialGav(NodeModel model) {
        String[] gav = model.getPackageName("").replace("referential.", "").split("\\.");
        String moduleName = gav[0];
        String subModuleName = gav.length == 2 ? gav[1] : gav[0];
        return Pair.of(moduleName, subModuleName);
    }

    public static Pair<String, String> dataGav(NodeModel model) {
        String[] gav = model.getPackageName("").replace("data.", "").split("\\.");
        String moduleName = gav[0];
        String subModuleName = gav[1];
        return Pair.of(moduleName, subModuleName);
    }

    public static String referentialI18n(Pair<String, String> gav) {
        String moduleName = gav.getLeft();
        String subModuleName = gav.getRight();
        String i18n;
        if (!moduleName.equals(subModuleName)) {
            i18n = moduleName + "." + subModuleName;
        } else {
            i18n = moduleName;
        }
        return i18n;
    }

    NodeModel(NodeModel parent, NavigationNodeDescriptor descriptor, int level, NavigationNodeType type) {
        this.parent = parent;
        this.type = Objects.requireNonNull(type);
        this.descriptor = Objects.requireNonNull(descriptor);
        this.level = level;
        if (level == -1) {
            dtoType = VoidDto.class.getName();
        } else {
            this.dtoType = descriptor.getClass().getName()
                    .replace("NavigationNodeDescriptor", "Dto")
                    .replace("fr.ird.observe.navigation.descriptor", "fr.ird.observe.dto");
        }
        if (isRoot()) {
            gav = null;
            iconPath = null;
            path = null;
            text = null;
        } else if (isReferentialType() || isReferentialPackage()) {
            gav = referentialGav(this);
            String simpleName = getSimpleName("");
            iconPath = String.format("%1$s.referential.%2$s.Home", gav.getLeft(), gav.getRight());
            if (isReferentialPackage()) {
                path = String.format("referential-%1$s%2$s", gav.getLeft(), StringUtils.capitalize(gav.getRight()));
                text = "project().getReferentialPackageTitle(module(), subModule())";
            } else {
                path = Introspector.decapitalize(simpleName);
                String i18n = referentialI18n(gav);
                text = String.format("observe.referential.%1$s.%2$s.type", i18n, simpleName);
            }

        } else {
            gav = dataGav(this);
            String simpleName = getSimpleName("");
            if (isOpenList()) {
                text = String.format("observe.data.%1$s.%2$s.%3$s.list.title", gav.getLeft(), gav.getRight(), simpleName.replace("List", ""));
            } else {
                text = String.format("observe.data.%1$s.%2$s.%3$s.type", gav.getLeft(), gav.getRight(), simpleName);
            }
            path = null;
            iconPath = String.format("%1$s.data.%2$s.%3$s", gav.getLeft(), gav.getRight(), simpleName);
        }
    }

    public String getPath() {
        if (isOpenList() || isTable() || isSimple() || isEdit() || isOpen()) {
            NodeLinkModel nodeLinkModel = getParent().getChildren().stream().filter(l -> l.getTargetModel().equals(this)).findFirst().orElseThrow();
            return "\"" + nodeLinkModel.getPropertyName() + "\"";
        }
        return "\"" + path + "\"";
    }

    public String getText(I18nKeySet getterFile) {
        if (isReferentialPackage() || isRoot()) {
            return text;
        }
        String i18nKey;
        if (isOpenList() && getLevel() == 0) {
            NodeLinkModel nodeLinkModel = getChildren().get(0);
            i18nKey = String.format("observe.data.%1$s.%2$s.%3$s.list.title", gav.getLeft(), gav.getRight(), nodeLinkModel.getSimpleName(""));
        } else {
            i18nKey = text;
        }
        getterFile.addKey(i18nKey);
        return String.format("io.ultreia.java4all.i18n.I18n.t(\"%s\")", i18nKey);
    }

    public String getIconPath() {
        if (isOpenList() && getLevel() == 0) {
            NodeLinkModel nodeLinkModel = getChildren().get(0);
            return String.format("%1$s.data.%2$s.%3$s", gav.getLeft(), gav.getRight(), nodeLinkModel.getSimpleName("List"));
        }
        if (isOpen() && getLevel() == 0) {
            NodeLinkModel nodeLinkModel = getChildren().get(0);
            return String.format("%1$s.data.%2$s.%3$s", gav.getLeft(), gav.getRight(), nodeLinkModel.getSimpleName("List"));
        }
        return iconPath;
    }

    @Override
    public NavigationNodeType getType() {
        return type;
    }

    public NodeModel getParent() {
        return parent;
    }

    public int getLevel() {
        return level;
    }

    public String getDtoType() {
        return dtoType;
    }

    public List<NodeLinkModel> getChildren() {
        return children;
    }

    public String getName(String prefix, String suffix) {
        String packageName = getPackageName(prefix);
        return packageName + (packageName.isEmpty() ? "" : ".") + getSimpleName(suffix);
    }

    public String getPackageName(String prefix) {
        return getPackageName(descriptor, prefix);
    }

    public String getSimpleName(String suffix) {
        return getSimpleName(descriptor, (isOpenList() && level > 1 ? "List" : "") + suffix);
    }

    public void addCapability(NavigationNodeCapability<?> capability, NodeModel targetModel) {
        children.add(new NodeLinkModel(capability, targetModel, children.size()));
    }

    public String getParentName(String prefix, String suffix) {
        return parent == null ? null : parent.getName(prefix, suffix);
    }

    public List<NodeModel> collectNodes() {
        LinkedList<NodeModel> result = new LinkedList<>();
        collectNodes(result);
        return result;
    }

    protected void collectNodes(LinkedList<NodeModel> result) {
        result.add(this);
        for (NodeLinkModel child : children) {
            child.getTargetModel().collectNodes(result);
        }
    }

    public void accept(NodeModelVisitor visitor) {
        visitor.start(this);
        for (NodeLinkModel child : children) {
            visitor.linkStart(this, child);
            child.getTargetModel().accept(visitor);
            visitor.linkEnd(this, child);
        }
        visitor.end(this);
    }
}
