package fr.ird.observe.toolkit.templates.navigation;

/*-
 * #%L
 * ObServe Toolkit :: Templates
 * %%
 * Copyright (C) 2017 - 2021 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import fr.ird.observe.toolkit.navigation.spi.DepthNavigationNodeDescriptorVisitor;
import fr.ird.observe.toolkit.navigation.spi.NavigationNodeCapability;
import fr.ird.observe.toolkit.navigation.spi.NavigationNodeDescriptor;
import fr.ird.observe.toolkit.navigation.spi.NavigationNodeType;
import org.apache.commons.lang3.mutable.MutableInt;

import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.function.Predicate;

/**
 * Created on 01/04/2021.
 *
 * @author Tony Chemit - dev@tchemit.fr
 * @since 5.0.17
 */
public class NodeModelsBuilder {

    private final Predicate<NavigationNodeCapability<?>> predicate;
    private final Map<NavigationNodeDescriptor, NodeModel> mapping = new LinkedHashMap<>();
    private NodeModel root;
    private final DepthNavigationNodeDescriptorVisitor.DelegateVisitor visitor = new DepthNavigationNodeDescriptorVisitor.DelegateVisitor() {

        @Override
        public void visit(NavigationNodeCapability<?> capability) {
            onCapacity(stack, mapping, capability);
        }
    };

    private static class PrintModel extends NodeModelVisitor {
        protected final MutableInt count = new MutableInt();
        protected final StringBuilder builder = new StringBuilder();
        protected String prefix = "";

        @Override
        public void start(NodeModel model) {
            super.start(model);
            prefix += "  ";
        }

        @Override
        public void end(NodeModel model) {
            super.end(model);
            if (stack.isEmpty()) {
                prefix = "";
            } else {
                prefix = prefix.substring(0, prefix.length() - 2);
            }
        }

        @Override
        public void linkStart(NodeModel model, NodeLinkModel link) {
            super.linkStart(model, link);
            add(String.format("%3s%s%s:%s[%s] %s → %s %s%n",
                              count.incrementAndGet(),
                              prefix,
                              model.getName("", ""),
                              link.getPropertyName(),
                              link.typeLabel(),
                              link.getCapability().isMultiple() ? "*" : " ",
                              link.getName("", ""),
                              "(" + getPrefix() + ")"));
        }

        @Override
        public void linkEnd(NodeModel model, NodeLinkModel link) {
            super.linkEnd(model, link);
        }

        public String getContent() {
            return builder.toString();
        }

        protected String getPrefix() {
            Iterator<NodeLinkModel> iterator = links.iterator();
            String result = "";
            for (NodeModel nodeModel : stack) {
                if (iterator.hasNext()) {
                    int order = 1 + nodeModel.children.indexOf(iterator.next());
                    result += "." + order;
                } else {
                    break;
                }
            }
            return result.isEmpty() ? result : result.substring(1);
        }

        protected void add(String content) {
            builder.append(content);
        }

    }

    public NodeModelsBuilder(Predicate<NavigationNodeCapability<?>> predicate) {
        this.predicate = predicate;
    }

    public List<NodeModel> build(NavigationNodeDescriptor descriptor) {
        mapping.clear();
        root = null;
        descriptor.accept(new DepthNavigationNodeDescriptorVisitor(visitor));
        return new LinkedList<>(mapping.values());
    }

    public String getDebugTree() {
        PrintModel print = new PrintModel();
        if (root != null) {
            root.accept(print);
        }
        return print.getContent();
    }

    protected void onCapacity(LinkedList<NavigationNodeCapability<?>> stack, Map<NavigationNodeDescriptor, NodeModel> mapping, NavigationNodeCapability<?> capability) {
        if (predicate.test(capability)) {
            int level = stack.size() - 2;
            NavigationNodeDescriptor parent = capability.getParent();
            registerNode(null, parent, level, NavigationNodeType.Root);
            NodeModel nodeModel = mapping.get(parent);
            NavigationNodeDescriptor target = capability.getTarget();
            registerNode(nodeModel, target, level + 1, capability.getType());
            NodeModel targetModel = mapping.get(target);

            nodeModel.addCapability(capability, targetModel);
        }
    }

    protected void registerNode(NodeModel parent, NavigationNodeDescriptor descriptor, int level, NavigationNodeType type) {
        registerNode(mapping, parent, descriptor, level, type);
        if (parent == null && root == null) {
            root = mapping.get(descriptor);
        }
    }

    protected void registerNode(Map<NavigationNodeDescriptor, NodeModel> mapping, NodeModel parent, NavigationNodeDescriptor descriptor, int level, NavigationNodeType type) {
        if (!mapping.containsKey(descriptor)) {
            mapping.computeIfAbsent(descriptor, descriptor1 -> new NodeModel(parent, descriptor1, level, type));
        }
    }
}
