package fr.ird.observe.toolkit.templates.service.api;

/*-
 * #%L
 * ObServe Toolkit :: Templates
 * %%
 * Copyright (C) 2017 - 2021 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */





import com.google.common.collect.ImmutableMap;
import fr.ird.observe.dto.data.OpenableDto;
import fr.ird.observe.services.service.data.OpenableDataService;
import fr.ird.observe.spi.ProjectPackagesDefinition;
import fr.ird.observe.spi.context.OpenableDtoServiceContext;
import fr.ird.observe.spi.mapping.OpenableDtoToServiceContextMapping;
import fr.ird.observe.toolkit.templates.ToolkitTagValues;
import fr.ird.observe.toolkit.templates.TemplateContract;
import io.ultreia.java4all.http.spi.Internal;
import org.nuiton.eugene.EugeneCoreTagValues;
import org.nuiton.eugene.java.BeanTransformerContext;
import org.nuiton.eugene.java.BeanTransformerTagValues;
import org.nuiton.eugene.java.EugeneJavaTagValues;
import org.nuiton.eugene.java.ObjectModelTransformerToJava;
import org.nuiton.eugene.java.extension.ImportsManager;
import org.nuiton.eugene.java.extension.ObjectModelAnnotation;
import org.nuiton.eugene.models.object.ObjectModel;
import org.nuiton.eugene.models.object.ObjectModelAttribute;
import org.nuiton.eugene.models.object.ObjectModelClass;
import org.nuiton.eugene.models.object.ObjectModelClassifier;
import org.nuiton.eugene.models.object.ObjectModelElement;
import org.nuiton.eugene.models.object.ObjectModelInterface;
import org.nuiton.eugene.models.object.ObjectModelJavaModifier;
import org.nuiton.eugene.models.object.ObjectModelModifier;
import org.nuiton.eugene.models.object.ObjectModelOperation;
import org.nuiton.eugene.models.object.ObjectModelPackage;
import org.nuiton.eugene.models.object.ObjectModelParameter;
import org.nuiton.topia.persistence.TopiaEntity;

import java.util.Comparator;
import java.util.Map;
import java.util.TreeMap;

/**
 * Created on 17/10/2020.
 *
 * @author Tony Chemit - dev@tchemit.fr
 * @since 8.0.1
 */
public class OpenableServiceTransformer extends ObjectModelTransformerToJava implements TemplateContract {

    private final EugeneCoreTagValues coreTagValues;
    private final EugeneJavaTagValues javaTemplatesTagValues;
    private final BeanTransformerTagValues beanTagValues;
    private final ToolkitTagValues observeTagValues;

    public OpenableServiceTransformer() {
        observeTagValues = new ToolkitTagValues();
        coreTagValues = new EugeneCoreTagValues();
        javaTemplatesTagValues = new EugeneJavaTagValues();
        beanTagValues = new BeanTransformerTagValues();
    }

    @Override
    public void transformFromModel(ObjectModel model) {
        super.transformFromModel(model);

        ProjectPackagesDefinition def = ProjectPackagesDefinition.of(getClassLoader());
        BeanTransformerContext context = SimpleServiceTransformer.create(def, model, coreTagValues, javaTemplatesTagValues, beanTagValues, "Openable", getLog());
        context.report();

        Map<String, String> mapping = new TreeMap<>();
        String serviceRootPackage = def.getServiceRootPackage();

        for (ObjectModelClass input : context.selectedClasses) {

            ObjectModelPackage aPackage = getPackage(input);

            String packageName = aPackage.getName();

            String prefix = getConstantPrefix(input);
            setConstantPrefix(prefix);

            String dtoName = context.classesNameTranslation.get(input);

            String referenceName = ProjectPackagesDefinition.cleanType(dtoName) + "Reference";
            String serviceName = (ProjectPackagesDefinition.cleanType(dtoName) + "Service");
            String generatedClassName = "Generated" + serviceName;

            String servicePackageName = serviceRootPackage + def.getRelativeDtoPackage(packageName);

            mapping.put(packageName + "." + dtoName, servicePackageName + "." + serviceName + ".SPI");

            String comparatorTagValue = observeTagValues.getComparatorTagValue(input);
            if (comparatorTagValue == null) {
                comparatorTagValue = TopiaEntity.PROPERTY_TOPIA_CREATE_DATE;
            }
            generateGeneratedClass(packageName, servicePackageName, serviceName, generatedClassName, dtoName, referenceName, comparatorTagValue);

            boolean generateClass = !getResourcesHelper().isJavaFileInClassPath(servicePackageName + "." + serviceName);
            if (generateClass) {
                generateClass(servicePackageName, serviceName, generatedClassName);
            }
        }

        generateClassMapping(true, OpenableDtoToServiceContextMapping.class, "Class<? extends " + OpenableDto.class.getName() + ">", OpenableDtoServiceContext.class.getName() + "<?, ?, ?>", ImmutableMap.class, "build", mapping);

    }

    @Override
    public ObjectModelClass createClass(String className, String packageName) {
        return super.createClass(className, packageName);
    }

    @Override
    public void addImport(ObjectModelClass output, Class<?> type) {
        super.addImport(output, type);
    }

    @Override
    public void addImport(ObjectModelClass output, String type) {
        super.addImport(output, type);
    }

    @Override
    public void setSuperClass(ObjectModelClass output, Class<?> superClass) {
        super.setSuperClass(output, superClass);
    }

    @Override
    public ObjectModelAnnotation addAnnotation(ObjectModelClass output, ObjectModelElement output1, Class<?> annotationType) {
        return super.addAnnotation(output, output1, annotationType);
    }

    @Override
    public void addAnnotationParameter(ObjectModelClass output, ObjectModelAnnotation annotation, String name, String value) {
        super.addAnnotationParameter(output, annotation, name, value);
    }

    @Override
    public ObjectModelOperation addConstructor(ObjectModelClass output, ObjectModelJavaModifier modifiers) {
        return super.addConstructor(output, modifiers);
    }

    @Override
    public void setOperationBody(ObjectModelOperation constructor, String body) {
        super.setOperationBody(constructor, body);
    }

    @Override
    public ObjectModelParameter addParameter(ObjectModelOperation operation, Class<?> type, String name) {
        return super.addParameter(operation, type, name);
    }

    public ObjectModelOperation addOperation(ObjectModelClassifier classifier, String name, String type, ObjectModelModifier... modifiers) {
        return super.addOperation(classifier, name, type, modifiers);
    }

    @Override
    public ObjectModelAttribute addAttribute(ObjectModelClassifier classifier, String name, String type, String value, ObjectModelModifier... modifiers) {
        return super.addAttribute(classifier, name, type, value, modifiers);
    }

    private void generateClass(String servicePackageName, String className, String abstractClassName) {

        ObjectModelInterface output = createInterface(className, servicePackageName);
        addInterface(output, abstractClassName);

        getLog().debug("will generate " + output.getQualifiedName());
    }

    private void generateGeneratedClass(String packageName, String servicePackageName, String className, String abstractClassName, String dtoName, String referenceName, String comparator) {
        ObjectModelInterface output = createInterface(abstractClassName, servicePackageName);
        ImportsManager importManager = builder.getImportManager(output);
        importManager.addExcludedPattern(".+\\." + className);

        addInterface(output, String.format("%s<%s, %s>", OpenableDataService.class.getName(), dtoName, referenceName));
        addAnnotation(output, output, Internal.class);
        if (referenceName.endsWith("TripReference")) {
            // Add TripAwareService contract
            addInterface(output, String.format("fr.ird.observe.services.service.data.TripAwareService<%s>", referenceName));
        }
        String spiClass = OpenableDtoServiceContext.class.getSimpleName();
        addImport(output, OpenableDtoServiceContext.class);
        addImport(output, packageName + "." + dtoName);
        addImport(output, packageName + "." + referenceName);
        addImport(output, Comparator.class);
        @SuppressWarnings("unused") String comparatorGetter = getJavaBeanMethodName("get", comparator);
        String definitionType = String.format("%s<%s, %s, %s>", spiClass, dtoName, referenceName, className);
        addAttribute(output, "SPI", definitionType, ""+"\n"
+"            "+spiClass+".of(\n"
+"            "+dtoName+".class,\n"
+"            "+referenceName+".class,\n"
+"            "+className+".class,\n"
+"            Comparator.comparing("+referenceName+"::"+comparatorGetter+"),\n"
+"            "+referenceName+"::new);\n"
+"     ", ObjectModelJavaModifier.PUBLIC, ObjectModelJavaModifier.STATIC, ObjectModelJavaModifier.FINAL);

        getLog().debug("will generate " + output.getQualifiedName());

    }

}

