package fr.ird.observe.toolkit.templates.service.api;

/*-
 * #%L
 * ObServe Toolkit :: Templates
 * %%
 * Copyright (C) 2017 - 2021 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */





import com.google.auto.service.AutoService;
import com.google.common.collect.ImmutableList;
import fr.ird.observe.spi.ProjectPackagesDefinition;
import fr.ird.observe.spi.ServiceBusinessProject;
import fr.ird.observe.spi.mapping.ContainerDtoToServiceContextMapping;
import fr.ird.observe.spi.mapping.EditableDtoToServiceContextMapping;
import fr.ird.observe.spi.mapping.OpenableDtoToServiceContextMapping;
import fr.ird.observe.spi.mapping.SimpleDtoToServiceContextMapping;
import fr.ird.observe.toolkit.templates.TemplateContract;
import org.nuiton.eugene.EugeneCoreTagValues;
import org.nuiton.eugene.java.BeanTransformerContext;
import org.nuiton.eugene.java.BeanTransformerTagValues;
import org.nuiton.eugene.java.EugeneJavaTagValues;
import org.nuiton.eugene.java.ObjectModelTransformerToJava;
import org.nuiton.eugene.java.extension.ImportsManager;
import org.nuiton.eugene.java.extension.ObjectModelAnnotation;
import org.nuiton.eugene.models.object.ObjectModel;
import org.nuiton.eugene.models.object.ObjectModelAttribute;
import org.nuiton.eugene.models.object.ObjectModelClass;
import org.nuiton.eugene.models.object.ObjectModelClassifier;
import org.nuiton.eugene.models.object.ObjectModelElement;
import org.nuiton.eugene.models.object.ObjectModelJavaModifier;
import org.nuiton.eugene.models.object.ObjectModelModifier;
import org.nuiton.eugene.models.object.ObjectModelOperation;
import org.nuiton.eugene.models.object.ObjectModelPackage;
import org.nuiton.eugene.models.object.ObjectModelParameter;

/**
 * Created on 17/10/2020.
 *
 * @author Tony Chemit - dev@tchemit.fr
 * @since 8.0.1
 */
public class ServiceBusinessProjectTransformer extends ObjectModelTransformerToJava implements TemplateContract {

    private final EugeneCoreTagValues coreTagValues;
    private final EugeneJavaTagValues javaTemplatesTagValues;
    private final BeanTransformerTagValues beanTagValues;
    private ProjectPackagesDefinition def;
    private String generatedPackageName;

    public ServiceBusinessProjectTransformer() {
        coreTagValues = new EugeneCoreTagValues();
        javaTemplatesTagValues = new EugeneJavaTagValues();
        beanTagValues = new BeanTransformerTagValues();
    }

    @Override
    public void transformFromModel(ObjectModel model) {
        super.transformFromModel(model);

        def = ProjectPackagesDefinition.of(getClassLoader());

        // keep only (for the moment) OpenableDto classes
        BeanTransformerContext context = new BeanTransformerContext(model, coreTagValues, javaTemplatesTagValues, beanTagValues, false, false, input -> {

            // keep only (for the moment) OpenableDto classes
            ObjectModelPackage aPackage = model.getPackage(input.getPackageName());
            String packageName = aPackage.getName();
            if (def.isReferentialFromPackageName(packageName)) {
                return false;
            }
            return input.getSuperclasses().stream().anyMatch(t -> t.getName().equals("Openable"));
        }, getLog());

        context.report();

        generatedPackageName = ServiceBusinessProject.class.getPackage().getName();

        String modelName = model.getName();

        String generatedServiceBusinessProject = generateGeneratedServiceBusinessProject(modelName);
        if (isVerbose()) {
            getLog().info("Generated: " + generatedServiceBusinessProject);
        }

        String projectClassName = generateServiceBusinessProject(modelName, generatedServiceBusinessProject);
        if (isVerbose() && projectClassName != null) {
            getLog().info("Generated: " + projectClassName);
        }
    }

    private String generateServiceBusinessProject(String modelName, String generatedServiceBusinessProject) {

        String className = String.format("%s%s", modelName, ServiceBusinessProject.class.getSimpleName());
        boolean skip = getResourcesHelper().isJavaFileInClassPath(generatedPackageName + "." + className);
        if (skip) {
            return null;
        }
        ObjectModelClass aClass = createClass(className, generatedPackageName);
        ObjectModelAnnotation annotation = addAnnotation(aClass, aClass, AutoService.class);
        ImportsManager importManager = getImportManager(aClass);
        addAnnotationClassParameter(importManager, aClass, annotation, "value", ServiceBusinessProject.class);

        setSuperClass(aClass, generatedServiceBusinessProject);
        addImport(aClass, ImmutableList.class);
        addConstructor(aClass, ObjectModelJavaModifier.PUBLIC);
        return aClass.getQualifiedName();
    }

    @SuppressWarnings("unused")
    private String generateGeneratedServiceBusinessProject(String name) {
        String concreteClassName = String.format("%s%s", model.getName(), ServiceBusinessProject.class.getSimpleName());
        String className = String.format("Generated%s%s", model.getName(), ServiceBusinessProject.class.getSimpleName());
        String openableMappingClassName = String.format("%s%s", model.getName(), OpenableDtoToServiceContextMapping.class.getSimpleName());
        String editableMappingClassName = String.format("%s%s", model.getName(), EditableDtoToServiceContextMapping.class.getSimpleName());
        String simpleMappingClassName = String.format("%s%s", model.getName(), SimpleDtoToServiceContextMapping.class.getSimpleName());
        String containerMappingClassName = String.format("%s%s", model.getName(), ContainerDtoToServiceContextMapping.class.getSimpleName());
        ObjectModelClass aClass = createClass(className, generatedPackageName);
        setSuperClass(aClass, ServiceBusinessProject.class);
        addImport(aClass, OpenableDtoToServiceContextMapping.class.getPackage().getName() + "." + openableMappingClassName);
        addImport(aClass, EditableDtoToServiceContextMapping.class.getPackage().getName() + "." + editableMappingClassName);
        addImport(aClass, SimpleDtoToServiceContextMapping.class.getPackage().getName() + "." + simpleMappingClassName);
        addImport(aClass, ContainerDtoToServiceContextMapping.class.getPackage().getName() + "." + containerMappingClassName);
        ObjectModelOperation constructor = addConstructor(aClass, ObjectModelJavaModifier.PUBLIC);
        addStaticFactory(aClass, concreteClassName);
        setOperationBody(constructor, ""+"\n"
+"        super("+openableMappingClassName+".get(), "+editableMappingClassName+".get(), "+simpleMappingClassName+".get(), "+containerMappingClassName+".get());\n"
+"    ");
        return aClass.getQualifiedName();
    }

    @Override
    public ObjectModelClass createClass(String className, String packageName) {
        return super.createClass(className, packageName);
    }

    @Override
    public void addImport(ObjectModelClass output, Class<?> type) {
        super.addImport(output, type);
    }

    @Override
    public void addImport(ObjectModelClass output, String type) {
        super.addImport(output, type);
    }

    @Override
    public void setSuperClass(ObjectModelClass output, Class<?> superClass) {
        super.setSuperClass(output, superClass);
    }

    @Override
    public ObjectModelAnnotation addAnnotation(ObjectModelClass output, ObjectModelElement output1, Class<?> annotationType) {
        return super.addAnnotation(output, output1, annotationType);
    }

    @Override
    public void addAnnotationParameter(ObjectModelClass output, ObjectModelAnnotation annotation, String name, String value) {
        super.addAnnotationParameter(output, annotation, name, value);
    }

    @Override
    public ObjectModelOperation addConstructor(ObjectModelClass output, ObjectModelJavaModifier modifiers) {
        return super.addConstructor(output, modifiers);
    }

    @Override
    public void setOperationBody(ObjectModelOperation constructor, String body) {
        super.setOperationBody(constructor, body);
    }

    @Override
    public ObjectModelParameter addParameter(ObjectModelOperation operation, Class<?> type, String name) {
        return super.addParameter(operation, type, name);
    }

    public ObjectModelOperation addOperation(ObjectModelClassifier classifier, String name, String type, ObjectModelModifier... modifiers) {
        return super.addOperation(classifier, name, type, modifiers);
    }

    @Override
    public ObjectModelAttribute addAttribute(ObjectModelClassifier classifier, String name, String type, String value, ObjectModelModifier... modifiers) {
        return super.addAttribute(classifier, name, type, value, modifiers);
    }
}

