package fr.ird.observe.toolkit.templates.service.api;

/*-
 * #%L
 * ObServe Toolkit :: Templates
 * %%
 * Copyright (C) 2017 - 2021 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */





import com.google.common.collect.ImmutableMap;
import fr.ird.observe.dto.data.SimpleDto;
import fr.ird.observe.services.service.data.SimpleDataService;
import fr.ird.observe.spi.ProjectPackagesDefinition;
import fr.ird.observe.spi.context.SimpleDtoServiceContext;
import fr.ird.observe.spi.mapping.SimpleDtoToServiceContextMapping;
import fr.ird.observe.toolkit.templates.TemplateContract;
import io.ultreia.java4all.http.spi.Internal;
import org.nuiton.eugene.EugeneCoreTagValues;
import org.nuiton.eugene.LogProxy;
import org.nuiton.eugene.java.BeanTransformerContext;
import org.nuiton.eugene.java.BeanTransformerTagValues;
import org.nuiton.eugene.java.EugeneJavaTagValues;
import org.nuiton.eugene.java.ObjectModelTransformerToJava;
import org.nuiton.eugene.java.extension.ImportsManager;
import org.nuiton.eugene.java.extension.ObjectModelAnnotation;
import org.nuiton.eugene.models.object.ObjectModel;
import org.nuiton.eugene.models.object.ObjectModelAttribute;
import org.nuiton.eugene.models.object.ObjectModelClass;
import org.nuiton.eugene.models.object.ObjectModelClassifier;
import org.nuiton.eugene.models.object.ObjectModelElement;
import org.nuiton.eugene.models.object.ObjectModelInterface;
import org.nuiton.eugene.models.object.ObjectModelJavaModifier;
import org.nuiton.eugene.models.object.ObjectModelModifier;
import org.nuiton.eugene.models.object.ObjectModelOperation;
import org.nuiton.eugene.models.object.ObjectModelPackage;
import org.nuiton.eugene.models.object.ObjectModelParameter;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

/**
 * Created on 17/10/2020.
 *
 * @author Tony Chemit - dev@tchemit.fr
 * @since 8.0.1
 */
public class SimpleServiceTransformer extends ObjectModelTransformerToJava implements TemplateContract {

    private final EugeneCoreTagValues coreTagValues;
    private final EugeneJavaTagValues javaTemplatesTagValues;
    private final BeanTransformerTagValues beanTagValues;

    static BeanTransformerContext create(ProjectPackagesDefinition def, ObjectModel model, EugeneCoreTagValues coreTagValues, EugeneJavaTagValues javaTemplatesTagValues, BeanTransformerTagValues beanTagValues, String contractMatcher, LogProxy log) {
        return new BeanTransformerContext(model, coreTagValues, javaTemplatesTagValues, beanTagValues, false, false, input -> {

            // keep only (for the moment) SimpleDto classes
            ObjectModelPackage aPackage = model.getPackage(input.getPackageName());
            String packageName = aPackage.getName();
            if (def.isReferentialFromPackageName(packageName)) {
                return false;
            }
            List<ObjectModelClass> allSuperClasses = new ArrayList<>(input.getSuperclasses());
            for (ObjectModelClass superclass : input.getSuperclasses()) {
                Collection<ObjectModelClass> superclasses1 = superclass.getSuperclasses();
                if (superclasses1 != null) {
                    allSuperClasses.addAll(superclasses1);
                }
            }
            return allSuperClasses.stream().anyMatch(t -> t.getName().equals(contractMatcher));
        }, log);

    }

    public SimpleServiceTransformer() {
        coreTagValues = new EugeneCoreTagValues();
        javaTemplatesTagValues = new EugeneJavaTagValues();
        beanTagValues = new BeanTransformerTagValues();
    }

    @Override
    public void transformFromModel(ObjectModel model) {
        super.transformFromModel(model);

        ProjectPackagesDefinition def = ProjectPackagesDefinition.of(getClassLoader());

        BeanTransformerContext context = create(def, model, coreTagValues, javaTemplatesTagValues, beanTagValues, "Simple", getLog());
        context.report();

        Map<String, String> mapping = new TreeMap<>();
        String serviceRootPackage = def.getServiceRootPackage();

        for (ObjectModelClass input : context.selectedClasses) {

            ObjectModelPackage aPackage = getPackage(input);

            String packageName = aPackage.getName();

            String prefix = getConstantPrefix(input);
            setConstantPrefix(prefix);

            String dtoName = context.classesNameTranslation.get(input);

            String serviceName = (ProjectPackagesDefinition.cleanType(dtoName) + "Service");
            String generatedClassName = "Generated" + serviceName;

            String servicePackageName = serviceRootPackage + def.getRelativeDtoPackage(packageName);

            mapping.put(packageName + "." + dtoName, servicePackageName + "." + serviceName + ".SPI");
            generateGeneratedClass(packageName, servicePackageName, serviceName, generatedClassName, dtoName);

            boolean generateClass = !getResourcesHelper().isJavaFileInClassPath(servicePackageName + "." + serviceName);
            if (generateClass) {
                generateClass(servicePackageName, serviceName, generatedClassName);
            }
        }

        generateClassMapping(true, SimpleDtoToServiceContextMapping.class, "Class<? extends " + SimpleDto.class.getName() + ">", SimpleDtoServiceContext.class.getName() + "<?, ?>", ImmutableMap.class, "build", mapping);

    }

    @Override
    public ObjectModelClass createClass(String className, String packageName) {
        return super.createClass(className, packageName);
    }

    @Override
    public void addImport(ObjectModelClass output, Class<?> type) {
        super.addImport(output, type);
    }

    @Override
    public void addImport(ObjectModelClass output, String type) {
        super.addImport(output, type);
    }

    @Override
    public void setSuperClass(ObjectModelClass output, Class<?> superClass) {
        super.setSuperClass(output, superClass);
    }

    @Override
    public ObjectModelAnnotation addAnnotation(ObjectModelClass output, ObjectModelElement output1, Class<?> annotationType) {
        return super.addAnnotation(output, output1, annotationType);
    }

    @Override
    public void addAnnotationParameter(ObjectModelClass output, ObjectModelAnnotation annotation, String name, String value) {
        super.addAnnotationParameter(output, annotation, name, value);
    }

    @Override
    public ObjectModelOperation addConstructor(ObjectModelClass output, ObjectModelJavaModifier modifiers) {
        return super.addConstructor(output, modifiers);
    }

    @Override
    public void setOperationBody(ObjectModelOperation constructor, String body) {
        super.setOperationBody(constructor, body);
    }

    @Override
    public ObjectModelParameter addParameter(ObjectModelOperation operation, Class<?> type, String name) {
        return super.addParameter(operation, type, name);
    }

    public ObjectModelOperation addOperation(ObjectModelClassifier classifier, String name, String type, ObjectModelModifier... modifiers) {
        return super.addOperation(classifier, name, type, modifiers);
    }

    @Override
    public ObjectModelAttribute addAttribute(ObjectModelClassifier classifier, String name, String type, String value, ObjectModelModifier... modifiers) {
        return super.addAttribute(classifier, name, type, value, modifiers);
    }

    private void generateClass(String servicePackageName, String className, String abstractClassName) {

        ObjectModelInterface output = createInterface(className, servicePackageName);
        addInterface(output, abstractClassName);

        getLog().debug("will generate " + output.getQualifiedName());
    }

    private void generateGeneratedClass(String packageName, String servicePackageName, String className, String abstractClassName, String dtoName) {
        ObjectModelInterface output = createInterface(abstractClassName, servicePackageName);
        ImportsManager importManager = builder.getImportManager(output);
        importManager.addExcludedPattern(".+\\." + className);

        addInterface(output, String.format("%s<%s>", SimpleDataService.class.getName(), dtoName));
        addAnnotation(output, output, Internal.class);
        String spiClass = SimpleDtoServiceContext.class.getSimpleName();
        addImport(output, SimpleDtoServiceContext.class);
        addImport(output, packageName + "." + dtoName);
        String definitionType = String.format("%s<%s, %s>", spiClass, dtoName, className);
        addAttribute(output, "SPI", definitionType, ""+"\n"
+"            "+spiClass+".of(\n"
+"            "+dtoName+".class,\n"
+"            "+className+".class);\n"
+"     ", ObjectModelJavaModifier.PUBLIC, ObjectModelJavaModifier.STATIC, ObjectModelJavaModifier.FINAL);

        getLog().debug("will generate " + output.getQualifiedName());

    }

}

