package fr.ird.observe.toolkit.templates.service.local;

/*-
 * #%L
 * ObServe Toolkit :: Templates
 * %%
 * Copyright (C) 2017 - 2021 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import fr.ird.observe.dto.data.SimpleDto;
import fr.ird.observe.security.Permission;
import fr.ird.observe.services.service.ObserveService;
import fr.ird.observe.spi.PersistenceBusinessProject;
import fr.ird.observe.spi.ServiceBusinessProject;
import fr.ird.observe.spi.context.ContainerDtoServiceContext;
import fr.ird.observe.spi.context.DataDtoEntityContext;
import fr.ird.observe.spi.context.EditableDtoEntityContext;
import fr.ird.observe.spi.context.EditableDtoServiceContext;
import fr.ird.observe.spi.context.OpenableDtoEntityContext;
import fr.ird.observe.spi.context.OpenableDtoServiceContext;
import fr.ird.observe.spi.context.SimpleDtoEntityContext;
import fr.ird.observe.spi.context.SimpleDtoServiceContext;
import fr.ird.observe.toolkit.templates.TemplateContract;
import io.ultreia.java4all.http.maven.plugin.HttpMojoSupport;
import io.ultreia.java4all.http.maven.plugin.model.ImportManager;
import io.ultreia.java4all.lang.Objects2;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.maven.monitor.logging.DefaultLog;
import org.nuiton.eugene.GeneratorUtil;
import org.nuiton.eugene.java.ObjectModelTransformerToJava;
import org.nuiton.eugene.java.extension.ImportsManager;
import org.nuiton.eugene.java.extension.ObjectModelAnnotation;
import org.nuiton.eugene.models.object.ObjectModel;
import org.nuiton.eugene.models.object.ObjectModelAttribute;
import org.nuiton.eugene.models.object.ObjectModelClass;
import org.nuiton.eugene.models.object.ObjectModelClassifier;
import org.nuiton.eugene.models.object.ObjectModelElement;
import org.nuiton.eugene.models.object.ObjectModelJavaModifier;
import org.nuiton.eugene.models.object.ObjectModelModifier;
import org.nuiton.eugene.models.object.ObjectModelOperation;
import org.nuiton.eugene.models.object.ObjectModelParameter;
import io.ultreia.java4all.util.TimeLog;

import java.lang.reflect.Method;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Created on 26/12/2020.
 *
 * @author Tony Chemit - dev@tchemit.fr
 * @since 8.0.3
 */
public class GenerateServiceLocalServices extends ObjectModelTransformerToJava implements TemplateContract {

    private DefaultLog mavenLog;
    private boolean withErrors = false;

    private static <S> Set<ServiceLocalMethodDescriptionImpl> createModel(Class<S> serviceType, ImportManager importManager, Map<String, String> genericMapping) {
        List<Method> declaredMethods = HttpMojoSupport.getDeclaredMethods(ObserveService.class, serviceType);
        Set<ServiceLocalMethodDescriptionImpl> methods = new LinkedHashSet<>();
        for (Method declaredMethod : declaredMethods) {
            if (declaredMethod.isDefault()) {
                continue;
            }
            methods.add(new ServiceLocalMethodDescriptionImpl(importManager, serviceType, declaredMethod, genericMapping));
        }
        return methods;
    }

    @Override
    public void transformFromModel(ObjectModel model) {
        super.transformFromModel(model);
        mavenLog = new DefaultLog(new MyAbstractLogger(getLog()));
        List<Class<?>> allServices = HttpMojoSupport.getAllServices(ObserveService.class);
        for (Class<?> service : allServices) {
            processInput(service);
        }
        if (withErrors) {
            throw new IllegalStateException("Some errors occurs, fix them to continue.");
        }
    }

    public void processInput(Class<?> serviceType) {

        String servicePackageName = serviceType.getPackage().getName();
        String serviceSimpleName = serviceType.getSimpleName();

        String serviceLocalPackageName = servicePackageName.replace(".services.", ".services.local.");

        String superClassName = serviceSimpleName + "LocalSupport";
        Map<String, String> genericMapping = HttpMojoSupport.genericMapping(mavenLog, ObserveService.class, serviceType);

        String superClassFullyQualifiedName = serviceLocalPackageName + "." + superClassName;
        if (!getResourcesHelper().isJavaFileInClassPath(superClassFullyQualifiedName)) {
            // try to generate LocalSupport class
            boolean safe = false;
            if ("TripService".equals(serviceSimpleName)) {
                generateTripDataLocalSupport(serviceLocalPackageName, superClassName, genericMapping, serviceType);
                safe = true;
            } else {
                switch (serviceType.getInterfaces()[0].getInterfaces()[0].getSimpleName()) {
                    case "SimpleDataService":
                        generateSimpleDataLocalSupport(serviceLocalPackageName, superClassName, genericMapping, serviceType);
                        safe = true;
                        break;
                    case "ContainerDataService":
                        generateContainerDataLocalSupport(serviceLocalPackageName, superClassName, genericMapping, serviceType);
                        safe = true;
                        break;
                    case "OpenableDataService":
                        generateOpenableDataLocalSupport(serviceLocalPackageName, superClassName, genericMapping, serviceType);
                        safe = true;
                        break;
                    case "EditableDataService":
                        generateEditableDataLocalSupport(serviceLocalPackageName, superClassName, genericMapping, serviceType);
                        safe = true;
                        break;
                }
            }
            if (!safe) {
                getLog().error(String.format("could not find class: %s", superClassFullyQualifiedName));
                withErrors = true;
                return;
            }
        }

        String generatedClassName = serviceSimpleName + "Local";

        ObjectModelClass output = createClass(generatedClassName, serviceLocalPackageName);
        setSuperClass(output, superClassName);
        ImportManager importManager = new ImportManager();
        Set<ServiceLocalMethodDescriptionImpl> methodDescriptions = createModel(serviceType, importManager, genericMapping);
        ImportsManager realImportManager = builder.getImportManager(output);
        realImportManager.addImport(TimeLog.class);
        realImportManager.addImport(LogManager.class);

        addAttribute(output, "log", importAndSimplify(output, Logger.class.getName()), "LogManager.getLogger(" + output.getName() + ".class)", ObjectModelJavaModifier.STATIC, ObjectModelJavaModifier.FINAL, ObjectModelJavaModifier.PRIVATE);
        realImportManager.addExcludedPattern(".+\\." + output.getName());
        getLog().info(String.format("will generate %d method(s) for %s", methodDescriptions.size(), output.getQualifiedName()));
        for (ServiceLocalMethodDescriptionImpl methodDescription : methodDescriptions) {
            addMethod(output, methodDescription);
        }
        importManager.toDescription().forEach(t -> realImportManager.addImport(t.getName()));
    }

    @SuppressWarnings({"unused", "UnusedAssignment"})
    private void generateContainerDataLocalSupport(String servicePackageName, String superClassName, Map<String, String> genericMapping, Class<?> serviceType) {

        ObjectModelClass output = createAbstractClass(superClassName, servicePackageName);
        addInterface(output, serviceType);
        String maintDtoType = genericMapping.get("D");
        ContainerDtoServiceContext<?, ?, ?> serviceSpi = ServiceBusinessProject.fromContainerDto(Objects2.forName(maintDtoType));
        DataDtoEntityContext<?, ?, ?, ?> parentSpi = PersistenceBusinessProject.fromDataDto(serviceSpi.getDtoType());
        DataDtoEntityContext<?, ?, ?, ?> spi = PersistenceBusinessProject.fromDataDto(serviceSpi.getChildType());
        String parentEntityType = importAndSimplify(output, parentSpi.toEntityType().getName());
        String childDtoType = importAndSimplify(output, serviceSpi.getChildType().getName());
        String childEntityType = importAndSimplify(output, spi.toEntityType().getName());
        String childEntitySpi = importAndSimplify(output, spi.toEntityType().getName() + "Spi");
        maintDtoType = importAndSimplify(output, maintDtoType);
        setSuperClass(output, String.format("%s<%s, %s, %s, %s, %s>"
                , "fr.ird.observe.services.local.service.data.ContainerDataServiceLocalSupport"
                , parentEntityType
                , childDtoType
                , maintDtoType
                , childEntityType
                , childEntitySpi
        ));
        String mainSpiName = GeneratorUtil.convertVariableNameToConstantName(GeneratorUtil.getSimpleName(maintDtoType).replace("Dto", "") + "Spi");
        String childSpiName;
        if (childDtoType.equals(childEntityType + "Dto")) {
            childSpiName = "SPI";
        } else {
            childSpiName = GeneratorUtil.convertVariableNameToConstantName(GeneratorUtil.getSimpleName(childDtoType).replace("Dto", "") + "Spi");
        }
        ObjectModelOperation constructor = addConstructor(output, ObjectModelJavaModifier.PUBLIC);
        setOperationBody(constructor, ""+"\n"
+"        super("+parentEntityType+".SPI, "+parentEntityType+"."+mainSpiName+", "+childEntityType+"."+childSpiName+");\n"
+"    ");
    }

    private void generateOpenableDataLocalSupport(String servicePackageName, String superClassName, Map<String, String> genericMapping, Class<?> serviceType) {

        ObjectModelClass output = createAbstractClass(superClassName, servicePackageName);
        addInterface(output, serviceType);
        String maintDtoType = genericMapping.get("D");
        String maintReferenceType = genericMapping.get("R");
        OpenableDtoServiceContext<?, ?, ?> serviceSpi = ServiceBusinessProject.fromOpenableDto(Objects2.forName(maintDtoType));
        OpenableDtoEntityContext<?, ?, ?, ?, ?> mainSpi = (OpenableDtoEntityContext<?, ?, ?, ?, ?>) PersistenceBusinessProject.fromDataDto(serviceSpi.getDtoType());
        String mainEntityType = importAndSimplify(output, mainSpi.toEntityType().getName());
        String parentEntityType = importAndSimplify(output, mainSpi.toParentEntityType().getName());
        String mainEntityDao = importAndSimplify(output, mainSpi.toEntityType().getName() + "TopiaDao");
        String mainEntitySpi = importAndSimplify(output, mainSpi.toEntityType().getName() + "Spi");

        maintDtoType = importAndSimplify(output, maintDtoType);
        setSuperClass(output, String.format("%s<%s, %s, %s, %s, %s, %s>"
                , "fr.ird.observe.services.local.service.data.OpenableDataServiceLocalSupport"
                , parentEntityType
                , maintDtoType
                , maintReferenceType
                , mainEntityType
                , mainEntityDao
                , mainEntitySpi
        ));
        ObjectModelOperation constructor = addConstructor(output, ObjectModelJavaModifier.PUBLIC);
        setOperationBody(constructor, ""+"\n"
+"        super("+parentEntityType+".SPI, "+mainEntityType+".SPI);\n"
+"    ");
    }

    private void generateSimpleDataLocalSupport(String servicePackageName, String superClassName, Map<String, String> genericMapping, Class<?> serviceType) {

        ObjectModelClass output = createAbstractClass(superClassName, servicePackageName);
        addInterface(output, serviceType);
        String maintDtoType = genericMapping.get("D");
        SimpleDtoServiceContext<SimpleDto, ?> serviceSpi = ServiceBusinessProject.fromSimpleDto(Objects2.forName(maintDtoType));
        SimpleDtoEntityContext<?, ?, ?, ?> mainSpi = (SimpleDtoEntityContext<?, ?, ?, ?>) PersistenceBusinessProject.fromDataDto(serviceSpi.getDtoType());
        String mainEntityType = importAndSimplify(output, mainSpi.toEntityType().getName());
        String mainEntitySpi = importAndSimplify(output, mainSpi.toEntityType().getPackageName()+"."+mainSpi.toDtoType().getSimpleName().replace("Dto","") + "Spi");
        maintDtoType = importAndSimplify(output, maintDtoType);
        setSuperClass(output, String.format("%s<%s, %s, %s>"
                , "fr.ird.observe.services.local.service.data.SimpleDataServiceLocalSupport"
                , maintDtoType
                , mainEntityType
                , mainEntitySpi
        ));
        String childSpiName = GeneratorUtil.convertVariableNameToConstantName(GeneratorUtil.getSimpleName(maintDtoType).replace("Dto", "") + "Spi");
        ObjectModelOperation constructor = addConstructor(output, ObjectModelJavaModifier.PUBLIC);
        setOperationBody(constructor, ""+"\n"
+"        super("+mainEntityType+".SPI, "+mainEntityType+"."+childSpiName+");\n"
+"    ");
    }

    private void generateTripDataLocalSupport(String servicePackageName, String superClassName, Map<String, String> genericMapping, Class<?> serviceType) {

        ObjectModelClass output = createAbstractClass(superClassName, servicePackageName);
        ImportsManager importManager = getImportManager(output);
        importManager.addExcludedPattern(".+\\." + superClassName);
        addInterface(output, serviceType);
        String maintDtoType = genericMapping.get("D");
        String maintReferenceType = genericMapping.get("R");
        OpenableDtoServiceContext<?, ?, ?> serviceSpi = ServiceBusinessProject.fromOpenableDto(Objects2.forName(maintDtoType));
        OpenableDtoEntityContext<?, ?, ?, ?, ?> mainSpi = (OpenableDtoEntityContext<?, ?, ?, ?, ?>) PersistenceBusinessProject.fromDataDto(serviceSpi.getDtoType());
        String mainEntityType = importAndSimplify(output, mainSpi.toEntityType().getName());
        String parentEntityType = importAndSimplify(output, mainSpi.toParentEntityType().getName());
        String mainEntityDao = importAndSimplify(output, mainSpi.toEntityType().getName() + "TopiaDao");
        String mainEntitySpi = importAndSimplify(output, mainSpi.toEntityType().getName() + "Spi");

        maintDtoType = importAndSimplify(output, maintDtoType);
        setSuperClass(output, String.format("%s<%s, %s, %s, %s, %s, %s>"
                , "fr.ird.observe.services.local.service.data.TripServiceLocalSupport"
                , parentEntityType
                , maintDtoType
                , maintReferenceType
                , mainEntityType
                , mainEntityDao
                , mainEntitySpi
        ));
        ObjectModelOperation constructor = addConstructor(output, ObjectModelJavaModifier.PUBLIC);
        setOperationBody(constructor, ""+"\n"
+"        super("+parentEntityType+".SPI, "+mainEntityType+".SPI);\n"
+"    ");
    }

    private void generateEditableDataLocalSupport(String servicePackageName, String superClassName, Map<String, String> genericMapping, Class<?> serviceType) {

        ObjectModelClass output = createAbstractClass(superClassName, servicePackageName);
        addInterface(output, serviceType);
        String maintDtoType = genericMapping.get("D");
        String maintReferenceType = genericMapping.get("R");
        EditableDtoServiceContext<?, ?, ?> serviceSpi = ServiceBusinessProject.fromEditableDto(Objects2.forName(maintDtoType));
        EditableDtoEntityContext<?, ?, ?, ?, ?> mainSpi = (EditableDtoEntityContext<?, ?, ?, ?, ?>) PersistenceBusinessProject.fromDataDto(serviceSpi.getDtoType());
        String mainEntityType = importAndSimplify(output, mainSpi.toEntityType().getName());
        String parentEntityType = importAndSimplify(output, mainSpi.toParentEntityType().getName());
        String mainEntityDao = importAndSimplify(output, mainSpi.toEntityType().getName() + "TopiaDao");
        String mainEntitySpi = importAndSimplify(output, mainSpi.toEntityType().getName() + "Spi");

        maintDtoType = importAndSimplify(output, maintDtoType);
        setSuperClass(output, String.format("%s<%s, %s, %s, %s, %s, %s>"
                , "fr.ird.observe.services.local.service.data.EditableDataServiceLocalSupport"
                , parentEntityType
                , maintDtoType
                , maintReferenceType
                , mainEntityType
                , mainEntityDao
                , mainEntitySpi
        ));
        ObjectModelOperation constructor = addConstructor(output, ObjectModelJavaModifier.PUBLIC);
        setOperationBody(constructor, ""+"\n"
+"        super("+parentEntityType+".SPI, "+mainEntityType+".SPI);\n"
+"    ");
    }

    @SuppressWarnings({"unused", "StringOperationCanBeSimplified", "UnusedAssignment"})
    private void addMethod(ObjectModelClass output, ServiceLocalMethodDescriptionImpl methodDescription) {

        Permission methodeCredentials = methodDescription.getMethodeCredentials();
        boolean write = methodDescription.isWrite();
        boolean noTransaction = methodDescription.isNoTransaction();

        String methodName = methodDescription.getName();
        String returnType = importAndSimplify(output, methodDescription.getReturnType());
        ObjectModelOperation operation = addOperation(output, methodName, returnType);

        returnType = GeneratorUtil.removeGenericDefinition(returnType);
        addAnnotation(output, operation, Override.class);
        for (Class<?> exception : methodDescription.getExceptions()) {
            addException(operation, importAndSimplify(output, exception.getName()));
        }

        StringBuilder parametersBuilder = new StringBuilder();
        int index = 0;
        for (String parameterName : methodDescription.getParameterNames()) {
            String parameterType = methodDescription.getParameterTypes().get(index++);
            if (!parameterType.endsWith("[]")) {
                parameterType = importAndSimplify(output, parameterType);
            }
            addParameter(operation, parameterType, parameterName);
            parametersBuilder.append(", ").append(parameterName);
        }
        String parameters = index == 0 ? "" : parametersBuilder.substring(2);
        String returnInvocation = methodDescription.getReturnInvocation();
        StringBuilder body = new StringBuilder();
        if (noTransaction) {
            // no transaction in this call
            body.append(""+"\n"
+"            "+returnInvocation+"super."+methodName+"("+parameters+");");
            boxWithTimeLog(output.getQualifiedName(), operation, body.toString());
            return;
        }

        boolean noCredential = methodeCredentials == null;
        if (noCredential) {
            methodeCredentials = Permission.NONE;
        }

        addImport(output, Permission.class);
        addImport(output, "fr.ird.observe.entities.ObserveTopiaPersistenceContext");
        body.append(""+"\n"
+"            ObserveTopiaPersistenceContext persistenceContext = initTransaction(\""+methodName+"\", Permission."+methodeCredentials+");");

        if (noCredential && methodName.equals("close")) {
            //FIXME This is a hack find a way to specify this
            // do not open transaction
            body.append(""+"\n"
+"            "+returnInvocation+"super."+methodName+"("+parameters+");");
            boxWithTimeLog(output.getQualifiedName(), operation, body.toString());
            return;
        }

        if (!write) {
            // no commit in this call
            body.append(""+"\n"
+"            if (persistenceContext == null) {\n"
+"                try (ObserveTopiaPersistenceContext topiaPersistenceContext = newPersistenceContext()) {\n"
+"                    "+returnInvocation+"super."+methodName+"("+parameters+");\n"
+"                }\n"
+"            } else {\n"
+"                "+returnInvocation+"super."+methodName+"("+parameters+");\n"
+"            }");
            boxWithTimeLog(output.getQualifiedName(), operation, body.toString());
            return;
        }

        boolean doReturn = !returnType.trim().endsWith("void");
        if (doReturn) {
            body.append(""+"\n"
+"            if (persistenceContext == null) {\n"
+"                try (ObserveTopiaPersistenceContext topiaPersistenceContext = newPersistenceContext()) {\n"
+"                    "+returnType+" invoke = super."+methodName+"("+parameters+");\n"
+"                    topiaPersistenceContext.commit();\n"
+"                    return invoke;\n"
+"                }\n"
+"            } else {\n"
+"                return super."+methodName+"("+parameters+");\n"
+"            }");
            boxWithTimeLog(output.getQualifiedName(), operation, body.toString());
            return;
        }

        body.append(""+"\n"
+"            if (persistenceContext == null) {\n"
+"                try (ObserveTopiaPersistenceContext topiaPersistenceContext = newPersistenceContext()) {\n"
+"                    super."+methodName+"("+parameters+");\n"
+"                    topiaPersistenceContext.commit();\n"
+"                }\n"
+"            } else {\n"
+"                super."+methodName+"("+parameters+");\n"
+"            }");
        boxWithTimeLog(output.getQualifiedName(), operation, body.toString());
    }

    @SuppressWarnings("unused")
    protected void boxWithTimeLog(String className, ObjectModelOperation operation, String bodyContent) {
        String methodName = operation.getName();
        setOperationBody(operation, ""+"\n"
+"        long t0 = TimeLog.getTime();\n"
+"        try {"+bodyContent+"\n"
+"        } catch (Exception e) {\n"
+"            log.error(\"Could not invoke "+className+"."+methodName+"\", e);\n"
+"            throw e;\n"
+"        } finally {\n"
+"            TIME_LOG.log(t0, \"invokeMethod "+className+"."+methodName+"\");\n"
+"        }\n"
+"    ");
    }

    @Override
    public ObjectModelClass createClass(String className, String packageName) {
        return super.createClass(className, packageName);
    }

    @Override
    public void addImport(ObjectModelClass output, Class<?> type) {
        super.addImport(output, type);
    }

    @Override
    public void addImport(ObjectModelClass output, String type) {
        super.addImport(output, type);
    }

    @Override
    public void setSuperClass(ObjectModelClass output, Class<?> superClass) {
        super.setSuperClass(output, superClass);
    }

    @Override
    public ObjectModelAnnotation addAnnotation(ObjectModelClass output, ObjectModelElement output1, Class<?> annotationType) {
        return super.addAnnotation(output, output1, annotationType);
    }

    @Override
    public void addAnnotationParameter(ObjectModelClass output, ObjectModelAnnotation annotation, String name, String value) {
        super.addAnnotationParameter(output, annotation, name, value);
    }

    @Override
    public ObjectModelOperation addConstructor(ObjectModelClass output, ObjectModelJavaModifier modifiers) {
        return super.addConstructor(output, modifiers);
    }

    @Override
    public void setOperationBody(ObjectModelOperation constructor, String body) {
        super.setOperationBody(constructor, body);
    }

    @Override
    public ObjectModelParameter addParameter(ObjectModelOperation operation, Class<?> type, String name) {
        return super.addParameter(operation, type, name);
    }

    public ObjectModelOperation addOperation(ObjectModelClassifier classifier, String name, String type, ObjectModelModifier... modifiers) {
        return super.addOperation(classifier, name, type, modifiers);
    }

    @Override
    public ObjectModelAttribute addAttribute(ObjectModelClassifier classifier, String name, String type, String value, ObjectModelModifier... modifiers) {
        return super.addAttribute(classifier, name, type, value, modifiers);
    }
}


