package fr.ird.observe.toolkit.templates.validation;

/*
 * #%L
 * ObServe Toolkit :: Templates
 * %%
 * Copyright (C) 2017 - 2021 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */




import com.google.common.collect.Multimap;
import com.google.common.collect.MultimapBuilder;
import fr.ird.observe.dto.data.WithProportion;
import fr.ird.observe.toolkit.templates.TemplateContract;
import fr.ird.observe.toolkit.templates.ToolkitTagValues;
import fr.ird.observe.validation.ValidatorDto;
import fr.ird.observe.validation.ValidatorsManager;
import io.ultreia.java4all.i18n.spi.bean.BeanPropertyI18nKeyProducer;
import io.ultreia.java4all.i18n.spi.bean.BeanPropertyI18nKeyProducerProvider;
import io.ultreia.java4all.i18n.spi.builder.I18nKeySet;
import io.ultreia.java4all.lang.Objects2;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.nuiton.eugene.EugeneCoreTagValues;
import org.nuiton.eugene.GeneratorUtil;
import org.nuiton.eugene.TemplateConfiguration;
import org.nuiton.eugene.java.BeanTransformerContext;
import org.nuiton.eugene.java.BeanTransformerTagValues;
import org.nuiton.eugene.java.EugeneJavaTagValues;
import org.nuiton.eugene.java.JavaGeneratorUtil;
import org.nuiton.eugene.java.ObjectModelTransformerToJava;
import org.nuiton.eugene.models.object.ObjectModel;
import org.nuiton.eugene.models.object.ObjectModelAttribute;
import org.nuiton.eugene.models.object.ObjectModelClass;
import org.nuiton.eugene.models.object.ObjectModelPackage;
import org.nuiton.eugene.models.tagvalue.ObjectModelTagValuesStore;
import org.nuiton.validator.NuitonValidatorScope;

import java.beans.Introspector;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.net.URLClassLoader;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.TreeMap;
import java.util.function.BiFunction;
import java.util.function.BiPredicate;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * To generate some validators from dot model.
 * <p>
 * Created by tchemit on 17/10/2018.
 *
 * @author Tony Chemit - dev@tchemit.fr
 */
public class DtoFormValidatorTransformer extends ObjectModelTransformerToJava implements ValidatorCacheRequest {

    private static final Logger log = LogManager.getLogger(DtoFormValidatorTransformer.class);
    private static final String FILE_TEMPLATE = "<!DOCTYPE validators PUBLIC\n" +
            "    \"-//Apache Struts//XWork Validator 1.0.3//EN\"\n" +
            "    \"http://struts.apache.org/dtds/xwork-validator-1.0.3.dtd\">\n" +
            "<validators>\n" +
            "\n" +
            "%1$s" +
            "\n" +
            "</validators>";
    private static final String FIELD_TEMPLATE =
            "  <field name=\"%1$s\">\n" +
                    "%2$s" +
                    "  </field>\n";

    private static final String DISABLED_ERRORS_FIELD_TEMPLATE =
            "    <!-- check if referential %1$s is disabled (only if validation is strong) -->\n" +
                    "    <field-validator type=\"%2$s\" short-circuit=\"true\">\n" +
                    "      <message/>\n" +
                    "    </field-validator>\n";
    private static final String DISABLED_WARNING_FIELD_TEMPLATE =
            "    <!-- check if referential %1$s is disabled (only if validation is not strong) -->\n" +
                    "    <field-validator type=\"%2$s\" short-circuit=\"true\">\n" +
                    "      <message/>\n" +
                    "    </field-validator>\n";
    private static final String COMMENT_NEED_FIELD_TEMPLATE =
            "    <!-- %1$s is required if one of the selected referential requires it (%2$s) -->\n" +
                    "    <field-validator type=\"commentNeeded\" short-circuit=\"true\">\n" +
                    "      <param name=\"propertyNames\">%2$s</param>\n" +
                    "      <message/>\n" +
                    "    </field-validator>\n";
    private static final String MANDATORY_FIELD_TEMPLATE =
            "    <!-- %1$s is mandatory -->\n" +
                    "    <field-validator type=\"mandatory\" short-circuit=\"true\">\n" +
                    "      <message/>\n" +
                    "    </field-validator>\n";
    private static final String MANDATORY_LATITUDE_FIELD_TEMPLATE =
            "    <!-- check latitude format -->\n" +
                    "    <field-validator type=\"coordinateLatitudeDto\" short-circuit=\"true\">\n" +
                    "      <param name=\"editorName\">%2$s</param>\n" +
                    "      <message/>\n" +
                    "    </field-validator>\n";
    private static final String MANDATORY_LONGITUDE_FIELD_TEMPLATE =
            "    <!-- check longitude format -->\n" +
                    "    <field-validator type=\"coordinateLongitudeDto\" short-circuit=\"true\">\n" +
                    "      <param name=\"editorName\">%2$s</param>\n" +
                    "      <message/>\n" +
                    "    </field-validator>\n";
    private static final String MANDATORY_QUADRANT_FIELD_TEMPLATE =
            "    <!-- check quadrant (on %1$s) -->\n" +
                    "    <field-validator type=\"quadrantDto\" short-circuit=\"true\">\n" +
                    "      <param name=\"ocean\">%2$s</param>\n" +
                    "      <message/>\n" +
                    "    </field-validator>\n";
    private static final String OPTIONAL_LATITUDE_FIELD_TEMPLATE =
            "    <!-- is %1$s required ? -->\n" +
                    "    <field-validator type=\"fieldexpression\" short-circuit=\"true\">\n" +
                    "      <param name=\"expression\"> <![CDATA[ %1$s != null || ( %2$s == null && %3$s == null )]]> </param>\n" +
                    "      <message>observe.Id.validation.required.latitude</message>\n" +
                    "    </field-validator>\n";
    private static final String OPTIONAL_COORDINATE_FIELD_TEMPLATE =
            "    <!-- is %1$s filled ? -->\n" +
                    "    <field-validator type=\"notFilled\" short-circuit=\"true\">\n" +
                    "      <param name=\"skip\"><![CDATA[ %1$s != null && %2$s != null && %3$s != null ]]></param>\n" +
                    "      <message>observe.data.Data.coordinate.validation.notFilled</message>\n" +
                    "    </field-validator>\n";
    private static final String OPTIONAL_LONGITUDE_FIELD_TEMPLATE =
            "    <!-- is %1$s required ? -->\n" +
                    "    <field-validator type=\"fieldexpression\" short-circuit=\"true\">\n" +
                    "      <param name=\"expression\"><![CDATA[ %1$s != null || ( %2$s == null && %3$s == null )]]> </param>\n" +
                    "      <message>observe.Id.validation.required.longitude</message>\n" +
                    "    </field-validator>\n";
    private static final String OPTIONAL_QUADRANT_FIELD_TEMPLATE =
            "    <!-- is %1$s required ? -->\n" +
                    "    <field-validator type=\"fieldexpression\" short-circuit=\"true\">\n" +
                    "      <param name=\"expression\"><![CDATA[ %1$s != null || ( %2$s == null && %3$s == null )]]> </param>\n" +
                    "      <message>observe.data.Data.validation.required.quadrant</message>\n" +
                    "    </field-validator>";
    private static final String MANDATORY_IF_FIELD_TEMPLATE =
            "    <!-- %1$s is mandatory except if %2$s -->\n" +
                    "    <field-validator type=\"mandatory\" short-circuit=\"true\">\n" +
                    "      <param name=\"skip\"><![CDATA[ %2$s ]]></param>\n" +
                    "      <message>%3$s.validation.required</message>\n" +
                    "    </field-validator>\n";
    private static final String PROPORTION_TOTAL_FIELD_TEMPLATE =
            "    <!-- is proportion sum equals 100 (on %1$s) ? -->\n" +
                    "    <field-validator type=\"proportionTotal\" short-circuit=\"true\">\n" +
                    "      <message/>\n" +
                    "    </field-validator>\n";
    private static final String BOUND_NUMBER_FIELD_TEMPLATE =
            "    <!-- %2$s <= %1$s <= %3$s -->\n" +
                    "    <field-validator type=\"boundNumber\" short-circuit=\"true\">\n" +
                    "      <param name=\"min\">%2$s</param>\n" +
                    "      <param name=\"max\">%3$s</param>\n" +
                    "      <message/>\n" +
                    "    </field-validator>\n";
    private static final String BOUND_NUMBER_IF_FIELD_TEMPLATE =
            "    <!-- %2$s <= %1$s <= %3$s except if %4$s -->\n" +
                    "    <field-validator type=\"boundNumber\" short-circuit=\"true\">\n" +
                    "      <param name=\"skip\"><![CDATA[ %4$s ]]></param>\n" +
                    "      <param name=\"min\">%2$s</param>\n" +
                    "      <param name=\"max\">%3$s</param>\n" +
                    "      <message/>\n" +
                    "    </field-validator>\n";
    private static final String BOUND_TEMPERATURE_FIELD_TEMPLATE =
            "    <!-- Temperature bound (unit %4$s) %2$s <= %1$s <= %3$s -->\n" +
                    "    <field-validator type=\"temperatureBound\" short-circuit=\"true\">\n" +
                    "      <param name=\"min\">%2$s</param>\n" +
                    "      <param name=\"max\">%3$s</param>\n" +
                    "      <param name=\"defaultTemperatureFormat\">%4$s</param>\n" +
                    "      <message/>\n" +
                    "    </field-validator>\n";
    private static final String MANDATORY_STRING_FIELD_TEMPLATE =
            "    <!-- %1$s is a mandatory string -->\n" +
                    "    <field-validator type=\"mandatoryString\" short-circuit=\"true\">\n" +
                    "      <message/>\n" +
                    "    </field-validator>\n";
    private static final String MANDATORY_COLLECTION_FIELD_TEMPLATE =
            "    <!-- %1$s is a mandatory collection -->\n" +
                    "    <field-validator type=\"mandatoryCollection\" short-circuit=\"true\">\n" +
                    "      <message/>\n" +
                    "    </field-validator>\n";
    private static final String NOT_FILLED_FIELD_TEMPLATE =
            "    <!-- %1$s should be filled -->\n" +
                    "    <field-validator type=\"notFilled\" short-circuit=\"true\">\n" +
                    "      <message/>\n" +
                    "    </field-validator>\n";
    private static final String NOT_FILLED_IF_FIELD_TEMPLATE =
            "    <!-- %1$s should be filled except if %2$s -->\n" +
                    "    <field-validator type=\"notFilled\" short-circuit=\"true\">\n" +
                    "      <param name=\"skip\"><![CDATA[ %2$s ]]></param>\n" +
                    "      <message/>\n" +
                    "    </field-validator>\n";
    private static final String NOT_FILLED_IF_WITH_I18N_FIELD_TEMPLATE =
            "    <!-- %1$s should be filled except if %2$s -->\n" +
                    "    <field-validator type=\"notFilled\" short-circuit=\"true\">\n" +
                    "      <param name=\"skip\"><![CDATA[ %2$s ]]></param>\n" +
                    "      <message>%3$s.validation.notFilled</message>\n" +
                    "    </field-validator>\n";
    private static final String NOT_FILLED_STRING_FIELD_TEMPLATE =
            "    <!-- %1$s string should be filled -->\n" +
                    "    <field-validator type=\"stringNotFilled\" short-circuit=\"true\">\n" +
                    "      <message/>\n" +
                    "    </field-validator>\n";
    private static final String NOT_FILLED_COLLECTION_FIELD_TEMPLATE =
            "    <!-- %1$s collection should be filled -->\n" +
                    "    <field-validator type=\"collectionNotFilled\" short-circuit=\"true\">\n" +
                    "      <message/>\n" +
                    "    </field-validator>\n";
    private static final String REFERENTIAL_UNIQUE_FIELD_TEMPLATE =
            "    <!-- %1$s should be unique -->\n" +
                    "    <field-validator type=\"referentialUniqueField\" short-circuit=\"true\">\n" +
                    "      <message/>\n" +
                    "    </field-validator>\n";
    private static final String POSITIVE_NUMBER_FIELD_TEMPLATE =
            "    <!-- %1$s is a positive number -->\n" +
                    "    <field-validator type=\"positiveNumber\" short-circuit=\"true\">\n" +
                    "      <message/>\n" +
                    "    </field-validator>\n";
    private static final String STRICTLY_POSITIVE_NUMBER_FIELD_TEMPLATE =
            "    <!-- %1$s is a strictly positive number -->\n" +
                    "    <field-validator type=\"positiveNumber\" short-circuit=\"true\">\n" +
                    "      <param name=\"strict\">true</param>\n" +
                    "      <message/>\n" +
                    "    </field-validator>\n";
    private static final String PROPORTION_FIELD_TEMPLATE =
            "    <!-- %1$s is a proportion -->\n" +
                    "    <field-validator type=\"proportionField\" short-circuit=\"true\">\n" +
                    "      <message/>\n" +
                    "    </field-validator>\n";
    private static final String STRING_MAX_LENGTH_FIELD_TEMPLATE =
            "    <!-- %1$s length <= %2$s -->\n" +
                    "    <field-validator type=\"stringMaxLength\" short-circuit=\"true\">\n" +
                    "      <param name=\"maxLength\">%2$s</param>\n" +
                    "      <message/>\n" +
                    "    </field-validator>\n";
    private static final String SPECIES_WEIGHT_FIELD_TEMPLATE =
            "    <!-- check species weight bound on %1$s if %2$s -->\n" +
                    "    <field-validator type=\"species_weightDto\" short-circuit=\"true\">\n" +
                    "      <param name=\"ratio\">1.0</param>\n" +
                    "      <param name=\"expression\"><![CDATA[ %2$s ]]></param>\n" +
                    "      <message>observe.referential.common.Species.validation.weight.bound##${min}##${max}</message>\n" +
                    "    </field-validator>\n";
    private static final String SPECIES_LENGTH_FIELD_TEMPLATE =
            "    <!-- check species length bound on %1$s if %2$s -->\n" +
                    "    <field-validator type=\"species_lengthDto\" short-circuit=\"true\">\n" +
                    "      <param name=\"ratio\">1.0</param>\n" +
                    "      <param name=\"expression\"><![CDATA[ %2$s ]]></param>\n" +
                    "      <message>observe.referential.common.Species.validation.length.bound##${min}##${max}</message>\n" +
                    "    </field-validator>\n";
    private static final String COLLECTION_UNIQUE_KEY_FIELD_TEMPLATE =
            "    <!-- check unique value on %1$s for tuple %2$s -->\n" +
                    "    <field-validator type=\"collectionUniqueKey\" short-circuit=\"true\">\n" +
                    "      <param name=\"keys\">%2$s</param>\n" +
                    "      <message>%3$s.validation.uniqueKey##${firstBadIndex}</message>\n" +
                    "    </field-validator>\n";
//    private static final String COLLECTION_UNIQUE_KEY_FIELD_TEMPLATE =
//            "    <!-- check unique value on %1$s for tuple %2$s -->\n" +
//                    "    <field-validator type=\"collectionFieldExpression\" short-circuit=\"true\">\n" +
//                    "      <param name=\"mode\">UNIQUE_KEY</param>\n" +
//                    "      <param name=\"keys\">%2$s</param>\n" +
//                    "      <message>%3$s.validation.uniqueKey</message>\n" +
//                    "    </field-validator>\n";
    private final EugeneCoreTagValues coreTagValues;
    private final EugeneJavaTagValues javaTemplatesTagValues;
    private final BeanTransformerTagValues beanTagValues;
    private final ToolkitTagValues observeTagValues;
    private final ValidationTagValues validationTagValues;
    /**
     * Store all fields for any type found in a validator (used to generate i18n keys)
     */
    private final Multimap<Class<?>, String> allFieldsByType;
    private final Collection<ValidatorInfo> allValidators;
    private Collection<ValidatorInfo> userValidators;
    private Map<String, List<String>> update_errorValidatorsByField;
    private Map<String, List<String>> create_errorValidatorsByField;
    private Map<String, List<String>> update_warningValidatorsByField;
    private Map<String, List<String>> create_warningValidatorsByField;
    private URLClassLoader loader;
    private Class<?> dtoClazz;
    private BeanPropertyI18nKeyProducer labelsBuilder;

    public DtoFormValidatorTransformer() {
        coreTagValues = new EugeneCoreTagValues();
        javaTemplatesTagValues = new EugeneJavaTagValues();
        beanTagValues = new BeanTransformerTagValues();
        observeTagValues = new ToolkitTagValues();
        validationTagValues = new ValidationTagValues();
        allFieldsByType = MultimapBuilder.hashKeys().hashSetValues().build();
        allValidators = new LinkedList<>();
    }

    @Override
    public void transformFromModel(ObjectModel model) {
        super.transformFromModel(model);
        loadUserValidators();
        BeanTransformerContext context = new BeanTransformerContext(model, coreTagValues, javaTemplatesTagValues, beanTagValues, false, false, input -> {
            ObjectModelPackage aPackage = model.getPackage(input.getPackageName());
            Collection<ObjectModelAttribute> attributes = new LinkedList<>(input.getAttributes());
            attributes.addAll(input.getAllOtherAttributes());
            boolean needComment = attributes.stream().anyMatch(e -> e.getName().equals("comment"));
            String formTagValue = observeTagValues.getFormTagValue(input, aPackage);
            return needComment || observeTagValues.notSkip(formTagValue) != null;
        }, getLog());

        context.report();
        labelsBuilder = BeanPropertyI18nKeyProducerProvider.get().getDefaultLabelsBuilder();

        for (ObjectModelClass beanClass : context.selectedClasses) {

            if (beanClass.isStatic()) {
                // do not generate validators for static dto
                continue;
            }
            update_errorValidatorsByField = new TreeMap<>();
            create_errorValidatorsByField = new TreeMap<>();
            update_warningValidatorsByField = new TreeMap<>();
            create_warningValidatorsByField = new TreeMap<>();

            prepareBean(context, beanClass);
            processBean(context, beanClass);
        }
        generateI18n(getI18nGetterFile());

        Path targetDirectory = configuration.getProperty(TemplateConfiguration.PROP_OUTPUT_DIRECTORY, File.class).toPath();

        Path descriptorFile = targetDirectory.resolve(ValidatorsManager.RESOURCE_VALIDATORS.substring(1));
        generateDescriptor(descriptorFile);
    }

    private void generateDescriptor(Path target) {

        List<ValidatorInfo> validatorList = new ArrayList<>(allValidators);
        getLog().info(String.format("Detected %d validator descriptor(s).", validatorList.size()));
        List<ValidatorDto> validators = validatorList.stream().sorted(Comparator.comparing(ValidatorInfo::getOrderKey)).map(ValidatorInfo::toValidatorDto).collect(Collectors.toList());
        try {
            if (Files.notExists(target.getParent())) {
                Files.createDirectories(target.getParent());
            }
            new ValidatorsManager(validators).store(target);
        } catch (IOException e) {
            throw new IllegalStateException("Can't store validator descriptors to file: " + target, e);
        }
    }

    protected void generateI18n(I18nKeySet i18nGetterFile) {
        BeanPropertyI18nKeyProducer labelsBuilder = BeanPropertyI18nKeyProducerProvider.get().getDefaultLabelsBuilder();
        for (Map.Entry<Class<?>, Collection<String>> entry : allFieldsByType.asMap().entrySet()) {
            dtoClazz = entry.getKey();
            Collection<String> allFields = entry.getValue();
            getLog().info(String.format("Detected %d fields for type: %s", allFields.size(), dtoClazz.getName()));
            getLog().debug(String.format("\n\t%s", String.join("\n\t", allFields)));
            for (String field : allFields) {
                String key = labelsBuilder.getI18nPropertyKey(dtoClazz, field);
                i18nGetterFile.addKey(key);
            }
        }
    }

    protected void prepareBean(BeanTransformerContext context, ObjectModelClass beanClass) {

        String dtoType = context.classesNameTranslation.get(beanClass);
        ObjectModelPackage thisPackage = getPackage(beanClass);
        boolean referential = TemplateContract.isReferentialFromPackageName(thisPackage.getName());

        dtoClazz = Objects2.forName(thisPackage.getName() + "." + dtoType);

        String formTagValue = observeTagValues.getFormTagValue(beanClass, thisPackage).trim();

        Collection<ObjectModelAttribute> attributes = new LinkedList<>(beanClass.getAttributes());
        attributes.addAll(beanClass.getAllOtherAttributes());

        ObjectModelTagValuesStore tagValuesStore = model.getTagValuesStore();

        boolean needComment = attributes.stream().anyMatch(e -> e.getName().equals("comment"));
        if (needComment || formTagValue.equals(thisPackage.getName() + "." + dtoType)) {

            String mandatoryCoordinate = getStringProperties(beanClass, c -> validationTagValues.getNotNullCoordinate(tagValuesStore, c));
            String mayNotNullCoordinate = getStringProperties(beanClass, c -> validationTagValues.getMayNotNullCoordinate(tagValuesStore, c));
            String proportionTotal = getStringProperties(beanClass, c -> validationTagValues.getProportionTotal(tagValuesStore, c));
            Map<String, String> boundNumber = getStringProperties(beanClass, attributes, (c, a) -> validationTagValues.getBoundNumber(tagValuesStore, c, a));
            Map<String, String> boundTemperature = getStringProperties(beanClass, attributes, (c, a) -> validationTagValues.getBoundTemperature(tagValuesStore, c, a));
            Map<String, String> mandatoryIf = getStringProperties(beanClass, attributes, (c, a) -> validationTagValues.getNotNullIf(tagValuesStore, c, a));
            Map<String, String> mayNotNullIf = getStringProperties(beanClass, attributes, (c, a) -> validationTagValues.getMayNotNullIf(tagValuesStore, c, a));
            Map<String, String> commentNeeded = getStringProperties(beanClass, attributes, (c, a) -> validationTagValues.getCommentNeeded(tagValuesStore, c, a));
            Map<String, String> speciesWeight = getStringProperties(beanClass, attributes, (c, a) -> validationTagValues.getSpeciesWeight(tagValuesStore, c, a));
            Map<String, String> speciesLength = getStringProperties(beanClass, attributes, (c, a) -> validationTagValues.getSpeciesLength(tagValuesStore, c, a));
            Map<String, String> collectionUniqueKey = getStringProperties(beanClass, attributes, (c, a) -> validationTagValues.getCollectionUniqueKey(tagValuesStore, c, a));
            Map<String, Integer> stringMaxLength = getIntegerProperties(beanClass, attributes, (c, a) -> validationTagValues.getStringMaxLength(tagValuesStore, c, a));
            List<String> mandatory = getProperties(beanClass, attributes, (c, a) -> validationTagValues.isNotNull(tagValuesStore, c, a) && !"String".equals(GeneratorUtil.getSimpleName(a.getType())));
            List<String> mandatoryString = getProperties(beanClass, attributes, (c, a) -> validationTagValues.isNotNull(tagValuesStore, c, a) && "String".equals(GeneratorUtil.getSimpleName(a.getType())));
            List<String> mandatoryCollection = getProperties(beanClass, attributes, (c, a) -> validationTagValues.isNotNull(tagValuesStore, c, a) && GeneratorUtil.isNMultiplicity(a));
            mandatory.removeAll(mandatoryString);
            mandatory.removeAll(mandatoryCollection);
            mandatoryString.removeAll(mandatoryCollection);

            List<String> mayNotNull = getProperties(beanClass, attributes, (c, a) -> validationTagValues.isMayNotNull(tagValuesStore, c, a) && !"String".equals(GeneratorUtil.getSimpleName(a.getType())));
            List<String> mayNotNullString = getProperties(beanClass, attributes, (c, a) -> validationTagValues.isMayNotNull(tagValuesStore, c, a) && "String".equals(GeneratorUtil.getSimpleName(a.getType())));
            List<String> mayNotNullCollection = getProperties(beanClass, attributes, (c, a) -> validationTagValues.isMayNotNull(tagValuesStore, c, a) && GeneratorUtil.isNMultiplicity(a));
            mayNotNull.removeAll(mayNotNullString);
            mayNotNull.removeAll(mayNotNullCollection);
            mayNotNullString.removeAll(mayNotNullCollection);

            List<String> positiveNumber = getProperties(beanClass, attributes, (c, a) -> isNumber(a) && validationTagValues.isPositiveNumber(tagValuesStore, c, a));
            List<String> strictlyPositiveNumber = getProperties(beanClass, attributes, (c, a) -> isNumber(a) && validationTagValues.isStrictlyPositiveNumber(tagValuesStore, c, a));
            List<String> referentialUnique = getProperties(beanClass, attributes, (c, a) -> referential && validationTagValues.isUnique(tagValuesStore, c, a));
            List<String> proportion = getProperties(beanClass, attributes, (c, a) -> !referential && a.getName().equals(WithProportion.PROPERTY_PROPORTION));

            Map<String, String> properties = getProperties(attributes);

            getLog().debug(String.format("form: %s, found %d properties.", dtoType, properties.size()));

            addErrorValidators(mandatory, MANDATORY_FIELD_TEMPLATE);
            addErrorValidators(mandatoryString, MANDATORY_STRING_FIELD_TEMPLATE);
            addErrorValidators(mandatoryCollection, MANDATORY_COLLECTION_FIELD_TEMPLATE);
            addErrorValidators(mandatoryIf, (f, v) -> {
                String value = (String) v;
                String i18n = labelsBuilder.getI18nPropertyKey(dtoClazz, f);
                return String.format(MANDATORY_IF_FIELD_TEMPLATE, f, value, i18n);
            });
            addErrorValidators(positiveNumber, POSITIVE_NUMBER_FIELD_TEMPLATE);
            addErrorValidators(strictlyPositiveNumber, STRICTLY_POSITIVE_NUMBER_FIELD_TEMPLATE);
            addErrorValidators(referentialUnique, REFERENTIAL_UNIQUE_FIELD_TEMPLATE);
            addErrorValidators(proportion, PROPORTION_FIELD_TEMPLATE);
            addErrorValidators(stringMaxLength, STRING_MAX_LENGTH_FIELD_TEMPLATE);
            addErrorValidators(commentNeeded, COMMENT_NEED_FIELD_TEMPLATE);
            addSpecies(speciesLength, SPECIES_LENGTH_FIELD_TEMPLATE);
            addSpecies(speciesWeight, SPECIES_WEIGHT_FIELD_TEMPLATE);

            for (Map.Entry<String, String> entry : boundNumber.entrySet()) {
                String value = entry.getValue();
                boolean warning = value.startsWith(":");
                if (warning) {
                    value = value.substring(1);
                }
                String field = entry.getKey();
                String[] split = value.split(":");
                String content;
                if (split.length == 2) {
                    content = String.format(BOUND_NUMBER_FIELD_TEMPLATE, field, split[0], split[1]);
                } else {
                    content = String.format(BOUND_NUMBER_IF_FIELD_TEMPLATE, field, split[0], split[1], split[2]);
                }
                if (warning) {
                    addWarningValidator(field, content);
                } else {
                    addErrorValidator(field, content);
                }
            }
            addErrorValidators(boundTemperature, (f, v) -> {
                String value = (String) v;
                String[] split = value.split(":");
                return String.format(BOUND_TEMPERATURE_FIELD_TEMPLATE, f, split[0], split[1], split[2]);
            });

            addErrorValidators(collectionUniqueKey, (f, v) -> {
                String value = (String) v;
                String i18n = labelsBuilder.getI18nPropertyKey(dtoClazz, f);
                return String.format(COLLECTION_UNIQUE_KEY_FIELD_TEMPLATE, f, value, i18n);
            });

            if (mandatoryCoordinate != null) {
                String[] split = mandatoryCoordinate.split("\\s*,\\s*");
                for (String coordinatePrefix : split) {
                    boolean before = coordinatePrefix.startsWith(":");
                    Pair<String, String> pair = getCoordinateMainAndPrefix(before, coordinatePrefix);
                    String main = pair.getLeft();
                    coordinatePrefix = pair.getRight();

                    String latitude = getCoordinateField(before, coordinatePrefix, "Latitude");
                    String longitude = getCoordinateField(before, coordinatePrefix, "Longitude");
                    String quadrant = getCoordinateField(before, coordinatePrefix, "Quadrant");

                    addLatitude(main, latitude, null);
                    addLongitude(main, longitude, null);
                    if (!referential) {
                        addDataQuadrant(quadrant);
                    }
                }
            }
            addWarningValidators(mayNotNullIf, (f, v) -> {
                String value = (String) v;
                boolean withI18n = value.startsWith(":");
                if (withI18n) {
                    String i18n = labelsBuilder.getI18nPropertyKey(dtoClazz, f);
                    return String.format(NOT_FILLED_IF_WITH_I18N_FIELD_TEMPLATE, f, value.substring(1), i18n);
                }
                return String.format(NOT_FILLED_IF_FIELD_TEMPLATE, f, value);
            });
            if (mayNotNullCoordinate != null) {
                boolean warning = mayNotNullCoordinate.startsWith("+");
                if (warning) {
                    mayNotNullCoordinate = mayNotNullCoordinate.substring(1);
                }
                String[] split = mayNotNullCoordinate.split("\\s*,\\s*");
                for (String coordinatePrefix : split) {
                    boolean before = coordinatePrefix.startsWith(":");
                    Pair<String, String> pair = getCoordinateMainAndPrefix(before, coordinatePrefix);
                    String main = pair.getLeft();
                    coordinatePrefix = pair.getRight();

                    String latitude = getCoordinateField(before, coordinatePrefix, "Latitude");
                    String longitude = getCoordinateField(before, coordinatePrefix, "Longitude");
                    String quadrant = getCoordinateField(before, coordinatePrefix, "Quadrant");

                    addLatitude(main, latitude, String.format(OPTIONAL_LATITUDE_FIELD_TEMPLATE, latitude, longitude, quadrant));
                    addLongitude(main, longitude, String.format(OPTIONAL_LONGITUDE_FIELD_TEMPLATE, longitude, latitude, quadrant));
                    addErrorValidator(quadrant, String.format(OPTIONAL_QUADRANT_FIELD_TEMPLATE, quadrant, latitude, longitude));
                    if (!referential) {
                        addDataQuadrant(quadrant);
                    }
                    if (warning) {
                        String content = String.format(OPTIONAL_COORDINATE_FIELD_TEMPLATE, quadrant, latitude, longitude);
                        addWarningValidator(quadrant, content);
                        addWarningValidator(latitude, content);
                        addWarningValidator(longitude, content);
                    }
                }
            }

            if (proportionTotal != null) {
                String[] split = proportionTotal.split("\\s*,\\s*");
                for (String field : split) {
                    field += "ProportionSum";
                    String content = String.format(PROPORTION_TOTAL_FIELD_TEMPLATE, field);
                    addErrorValidator(field, content);
                }
            }

            addWarningValidators(mayNotNull, NOT_FILLED_FIELD_TEMPLATE);
            addWarningValidators(mayNotNullString, NOT_FILLED_STRING_FIELD_TEMPLATE);
            addWarningValidators(mayNotNullCollection, NOT_FILLED_COLLECTION_FIELD_TEMPLATE);

            for (Map.Entry<String, String> entry : properties.entrySet()) {
                String field = entry.getKey();
                String checkDisabledReferentialOnErrorScope = String.format(DISABLED_ERRORS_FIELD_TEMPLATE, field, "checkDisabledReferentialOnErrorScope");
                addErrorValidator(field, checkDisabledReferentialOnErrorScope);
                String checkDisabledReferentialOnWarningScope = String.format(DISABLED_WARNING_FIELD_TEMPLATE, field, "checkDisabledReferentialOnWarningScope");
                addWarningValidator(field, checkDisabledReferentialOnWarningScope);
            }
            if (needComment && !properties.isEmpty()) {
                String content = String.format(COMMENT_NEED_FIELD_TEMPLATE, "comment", String.join(",", properties.keySet()));
                addErrorValidator("comment", content);
            }
        }

        addUserValidators(dtoClazz);
    }

    private void addSpecies(Map<String, String> fields, String template) {
        for (Map.Entry<String, String> entry : fields.entrySet()) {
            String value = entry.getValue();
            boolean warning = value.startsWith(":");
            if (warning) {
                value = value.substring(1);
            }
            String field = entry.getKey();
            String content = String.format(template, field, value);
            if (warning) {
                addWarningValidator(field, content);
            } else {
                addErrorValidator(field, content);
            }
        }
    }

    protected void addDataQuadrant(String field) {
        String packageName = dtoClazz.getPackageName();

        int i = packageName.indexOf(".data");
        String packagePart = packageName.substring(i + 6, packageName.indexOf('.', i + 7));
        String ocean = "current" + StringUtils.capitalize(packagePart) + "CommonTrip.ocean";
        String content = String.format(MANDATORY_QUADRANT_FIELD_TEMPLATE, field, ocean);
        addErrorValidator(field, content);
    }


    protected void addLatitude(String main, String field, String extra) {
        String content = (extra == null ? "" : extra) + String.format(MANDATORY_LATITUDE_FIELD_TEMPLATE, field, main);
        addErrorValidator(field, content);

    }

    protected void addLongitude(String main, String field, String extra) {
        String content = (extra == null ? "" : extra) + String.format(MANDATORY_LONGITUDE_FIELD_TEMPLATE, field, main);
        addErrorValidator(field, content);
    }

    protected String getCoordinateField(boolean before, String coordinatePrefix, String type) {
        return Introspector.decapitalize(before ? (type + coordinatePrefix) : (coordinatePrefix + type));
    }

    protected Pair<String, String> getCoordinateMainAndPrefix(boolean before, String coordinatePrefix) {
        String main;
        if (before) {
            coordinatePrefix = coordinatePrefix.substring(1);
        }
        if (coordinatePrefix.equals("default")) {
            coordinatePrefix = "";
            main = "coordinate";
        } else {
            if (before) {
                main = "coordinate" + coordinatePrefix;
            } else {
                main = coordinatePrefix;
            }
        }
        return Pair.of(main, coordinatePrefix);
    }

    protected void addUserValidators(Class<?> dtoClazz) {
        List<ValidatorInfo> userValidators = this.userValidators.stream().filter(v -> v.getType().isAssignableFrom(dtoClazz)).collect(Collectors.toList());

        for (ValidatorInfo userValidator : userValidators) {
            NuitonValidatorScope scope = userValidator.getScope();
            Map<String, List<String>> map = null;
            boolean create = Objects.equals(userValidator.getContext(), "create");
            switch (scope) {
                case FATAL:
                case ERROR:
                    map = create ? create_errorValidatorsByField : update_errorValidatorsByField;
                    break;
                case WARNING:
                    map = create ? create_warningValidatorsByField : update_warningValidatorsByField;
                    break;
                case INFO:
                    continue;
            }
            if (map == null) {
                throw new IllegalStateException("Can't deal with validator (scope or context unknown): " + userValidator);
            }
            for (String field : userValidator.getFields()) {
                getLog().debug(String.format("Add user validation on field %s-%s for context: %s/%s", dtoClazz.getName(), field, userValidator.getContext(), scope));
                String content = userValidator.getFieldFragment(field);
                if (content != null) {
                    addValidator(map, field, content);
                }
            }
        }
    }

    protected void processBean(BeanTransformerContext context, ObjectModelClass beanClass) {

        String packageName = beanClass.getPackageName();
        String dtoType = context.classesNameTranslation.get(beanClass);

        Path file = configuration.getProperty(TemplateConfiguration.PROP_OUTPUT_DIRECTORY, File.class).toPath();
        for (String dir : packageName.split("\\.")) {
            file = file.resolve(dir);
        }

        if (!create_errorValidatorsByField.isEmpty()) {
            generateGeneratedValidatorFile(file, "create", NuitonValidatorScope.ERROR, dtoType, create_errorValidatorsByField);
        }
        if (!update_errorValidatorsByField.isEmpty()) {
            generateGeneratedValidatorFile(file, "update", NuitonValidatorScope.ERROR, dtoType, update_errorValidatorsByField);
        }
        if (!create_warningValidatorsByField.isEmpty()) {
            generateGeneratedValidatorFile(file, "create", NuitonValidatorScope.WARNING, dtoType, create_warningValidatorsByField);
        }
        if (!update_warningValidatorsByField.isEmpty()) {
            generateGeneratedValidatorFile(file, "update", NuitonValidatorScope.WARNING, dtoType, update_warningValidatorsByField);
        }

    }

    private void generateGeneratedValidatorFile(Path dir, String context, NuitonValidatorScope scope, String fileName, Map<String, List<String>> fragmentEntry) {
        fileName = String.format("%s-%s-%s-validation.xml", fileName, context, scope.name().toLowerCase());
        List<String> fields = new LinkedList<>();
        Set<String> properties = new LinkedHashSet<>();
        for (Map.Entry<String, List<String>> entry : fragmentEntry.entrySet()) {
            String propertyName = entry.getKey();
            List<String> validators = entry.getValue();
            fields.add(String.format(FIELD_TEMPLATE, propertyName, String.join("\n", validators)));
            allFieldsByType.put(dtoClazz, propertyName);
            properties.add(propertyName);
        }
        String fileContent = String.format(FILE_TEMPLATE, String.join("\n", fields));
        Path target = dir.resolve(fileName);
        generateFile(target, fileContent);

        ValidatorInfo validator = new ValidatorInfo(target, dtoClazz, context, scope, properties);
        allValidators.add(validator);
    }

    private Map<String, String> getProperties(Collection<ObjectModelAttribute> attributes) {
        Map<String, String> properties = new TreeMap<>();
        for (ObjectModelAttribute attr : attributes) {
            if (!attr.isNavigable()) {
                continue;
            }
            String type = attr.getType();
            if (!type.endsWith("Reference") || !type.contains("referential")) {
                continue;
            }
            properties.put(attr.getName(), JavaGeneratorUtil.getSimpleName(type));
        }
        return properties;
    }

    private List<String> getProperties(ObjectModelClass beanClass, Collection<ObjectModelAttribute> attributes, BiPredicate<ObjectModelClass, ObjectModelAttribute> filter) {
        List<String> properties = new LinkedList<>();
        for (ObjectModelAttribute attr : attributes) {
            if (attr.isNavigable() && filter.test(beanClass, attr)) {
                properties.add(attr.getName());
            }
        }
        return properties;
    }

    private Map<String, Integer> getIntegerProperties(ObjectModelClass beanClass, Collection<ObjectModelAttribute> attributes, BiFunction<ObjectModelClass, ObjectModelAttribute, Integer> filter) {
        Map<String, Integer> properties = new LinkedHashMap<>();
        for (ObjectModelAttribute attr : attributes) {
            if (attr.isNavigable()) {
                Integer tagValue = filter.apply(beanClass, attr);
                if (tagValue != null) {
                    properties.put(attr.getName(), tagValue);
                }
            }
        }
        return properties;
    }

    private Map<String, String> getStringProperties(ObjectModelClass beanClass, Collection<ObjectModelAttribute> attributes, BiFunction<ObjectModelClass, ObjectModelAttribute, String> filter) {
        Map<String, String> properties = new LinkedHashMap<>();
        for (ObjectModelAttribute attr : attributes) {
            if (attr.isNavigable()) {
                String tagValue = filter.apply(beanClass, attr);
                if (tagValue != null) {
                    properties.put(attr.getName(), tagValue);
                }
            }
        }
        return properties;
    }

    private String getStringProperties(ObjectModelClass beanClass, Function<ObjectModelClass, String> filter) {
        return filter.apply(beanClass);
    }

    private boolean isNumber(ObjectModelAttribute attribute) {
        String type = GeneratorUtil.getSimpleName(attribute.getType());
        return "byte".equals(type) || "short".equals(type)
                || "Byte".equals(type) || "Short".equals(type)
                || "int".equals(type) || "long".equals(type)
                || "Integer".equals(type) || "Long".equals(type)
                || "float".equals(type) || "double".equals(type)
                || "Float".equals(type) || "Double".equals(type);
    }

    private void addErrorValidators(List<String> fields, String template) {
        for (String field : fields) {
            String content = String.format(template, field);
            addErrorValidator(field, content);
        }
    }

    private void addErrorValidators(Map<String, ?> fields, BiFunction<String, Object, String> toContent) {
        for (Map.Entry<String, ?> entry : fields.entrySet()) {
            String field = entry.getKey();
            Object value = entry.getValue();
            String content = toContent.apply(field, value);
            addErrorValidator(field, content);
        }
    }

    private void addWarningValidators(Map<String, ?> fields, BiFunction<String, Object, String> toContent) {
        for (Map.Entry<String, ?> entry : fields.entrySet()) {
            String field = entry.getKey();
            Object value = entry.getValue();
            String content = toContent.apply(field, value);
            addWarningValidator(field, content);
        }
    }

    private void addErrorValidators(Map<String, ?> fields, String template) {
        addErrorValidators(fields, (f, v) -> String.format(template, f, v));
    }

    private void addWarningValidators(List<String> fields, String template) {
        for (String field : fields) {
            String content = String.format(template, field);
            addWarningValidator(field, content);
        }
    }

    private void addErrorValidator(String field, String content) {
        addValidator(update_errorValidatorsByField, field, content);
        addValidator(create_errorValidatorsByField, field, content);
    }

    private void addWarningValidator(String field, String content) {
        addValidator(update_warningValidatorsByField, field, content);
        addValidator(create_warningValidatorsByField, field, content);
    }

    private void addValidator(Map<String, List<String>> errorValidatorsByField, String field, String template) {
        errorValidatorsByField.computeIfAbsent(field, e -> new LinkedList<>()).add(String.format(template, field));
    }

    private void generateFile(Path target, String content) {
        try {
            if (!Files.exists(target.getParent())) {
                Files.createDirectories(target.getParent());
            }
            log.info("Generate validation file: " + target);
            Files.write(target, content.getBytes());
        } catch (IOException e) {
            throw new IllegalStateException("Could not create file: " + target);
        }
    }

    @Override
    public Path getSourceRootPath() {
        return configuration.getProperty(TemplateConfiguration.PROP_RESOURCE_DIRECTORY, File.class).toPath().getParent().resolve("validation");
    }

    @Override
    public URLClassLoader getUrlClassLoader() {
        return loader;
    }

    private void loadUserValidators() {
        try {
            this.loader = new URLClassLoader(new URL[]{getSourceRootPath().toUri().toURL()}, configuration.getClassLoader());
            this.userValidators = ValidatorsCache.get().getValidators(this);
        } catch (IOException e) {
            throw new IllegalStateException("Can't get validators cache", e);
        }
        getLog().info(String.format("%d user validator(s) detected.", userValidators.size()));
        for (ValidatorInfo validator : userValidators) {
            getLog().debug("Detect user validator: " + validator);
        }
        for (ValidatorInfo validator : userValidators) {
            Class<?> type = validator.getType();
            Set<String> fields = validator.getFields();
            allFieldsByType.putAll(type, fields);
        }
    }
}
