package fr.ird.observe.toolkit.templates.validation;

/*-
 * #%L
 * ObServe Toolkit :: Templates
 * %%
 * Copyright (C) 2017 - 2021 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.ArrayListMultimap;
import fr.ird.observe.validation.ValidatorDto;
import fr.ird.observe.validation.ValidatorFieldDto;
import org.nuiton.validator.NuitonValidatorScope;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.StringJoiner;

/**
 * Created on 11/05/2021.
 *
 * @author Tony Chemit - dev@tchemit.fr
 * @since 5.0.24
 */
public class ValidatorInfo {

    private transient final Path file;
    private final Class<?> type;
    private final String context;
    private final NuitonValidatorScope scope;
    private final Set<String> fields;
    private transient Map<String, String> fragments;
    private ArrayListMultimap<String, String> comments;

    ValidatorInfo(Path file, Class<?> type, String context, NuitonValidatorScope scope, Set<String> fields) {
        this.file = file;
        this.type = type;
        this.context = context;
        this.scope = scope;
        this.fields = fields;
    }

    public Path getFile() {
        return file;
    }

    public String getOrderKey() {
        return type.getName() + "-" + context + "-" + scope;
    }

    public Class<?> getType() {
        return type;
    }

    public String getContext() {
        return context;
    }

    public NuitonValidatorScope getScope() {
        return scope;
    }

    public Set<String> getFields() {
        return fields;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        ValidatorInfo that = (ValidatorInfo) o;
        return Objects.equals(type, that.type) &&
                Objects.equals(context, that.context) &&
                scope == that.scope;
    }

    @Override
    public int hashCode() {
        return Objects.hash(type, context, scope);
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", ValidatorInfo.class.getSimpleName() + "[", "]")
                .add("type=" + type)
                .add("context='" + context + "'")
                .add("scope=" + scope)
                .add("fields=" + fields)
                .toString();
    }

    @SuppressWarnings({"unchecked", "rawtypes"})
    public ValidatorDto toValidatorDto() {
        List<String> fieldNames = new ArrayList<>(getFields());
        List<ValidatorFieldDto> fields = new ArrayList<>(fieldNames.size());
        fieldNames.sort(String::compareTo);
        for (String fieldName : fieldNames) {
            List<String> comments = getComments().get(fieldName);
            fields.add(new ValidatorFieldDto(fieldName, comments));
        }
        return new ValidatorDto((Class) type, scope, context, fields);
    }

    public String getFieldFragment(String field) {
        return getFragments().get(field);
    }

    protected Map<String, String> getFragments() {
        if (fragments == null) {
            fragments = new LinkedHashMap<>();
            try {
                String lastField = null;
                StringBuilder lastFieldContent = null;
                int index = 0;
                for (String line : Files.readAllLines(file, StandardCharsets.UTF_8)) {
                    index++;
                    //  <field name="well">
                    if (line.contains("<field name")) {
                        // new field
                        lastField = line.substring(line.indexOf("<field ") + 7);
                        lastField = lastField.substring(lastField.indexOf("\"") + 1);
                        lastField = lastField.substring(0, lastField.indexOf("\""));
                        lastFieldContent = new StringBuilder();
                        continue;
                    }
                    if (line.contains("</field>")) {
                        // end of field
                        if (lastFieldContent == null) {
                            throw new IllegalStateException(String.format("Found a end of field, but no begin was detected line %d on file %s", index, file));
                        }
                        fragments.put(lastField, lastFieldContent.toString());
                        lastField = null;
                        lastFieldContent = null;
                        continue;
                    }
                    if (lastFieldContent != null) {
                        lastFieldContent.append(line).append("\n");
                    }
                }
            } catch (IOException e) {
                throw new IllegalStateException("Can't read validator file: " + file, e);
            }
        }
        return fragments;
    }

    public Collection<String> getComments(String field) {
        return getComments().get(field);
    }

    protected ArrayListMultimap<String, String> getComments() {
        if (comments == null) {
            comments = ArrayListMultimap.create();
            for (Map.Entry<String, String> entry : getFragments().entrySet()) {
                String field = entry.getKey();
                String fragment = entry.getValue();
                String[] lines = fragment.split("\\s*\n\\s*");
                for (String line : lines) {
                    if (line.trim().startsWith("<!-- ")) {
                        comments.put(field, line.replace("<!--", "").replace("-->", "").trim());
                    }
                }
            }
        }
        return comments;
    }
}
