package configuration_file_parser;

import java.util.ArrayDeque;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.Set;

import org.apache.commons.configuration2.Configuration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import constants.BRunnerKeywords;

public class ParserUtils {

	static final Logger LOG = LoggerFactory.getLogger(ParserUtils.class);

	public static Set<String> getRelevantSubset(String parameter, Set<String> parameterSet) {
		if (parameter.equals(ParserConstants.EXECUTE_ONLY_FORALL_WILDCARD))
			return new HashSet<>(parameterSet);

		Set<String> returnSet = new HashSet<>();

		boolean found = false;
		for (String p : parameterSet) {
			if (p.startsWith(parameter + ParserConstants.MULTISCENARIO_SEPARATOR) || p.equals(parameter)) {
				found = true;
				returnSet.add(parameter);
			}
		}
		if (!found)
			throw new IllegalArgumentException("Unrecognized parameter value " + parameter + " among " + parameterSet
					+ " in " + ParserConstants.EXECUTE_ONLY_KW + ". This could because " + parameter
					+ " was not defined, or because the argument order \"" + ParserConstants.EXECUTE_ONLY_KW
					+ "=scenarioName configurationName executionName measureName\" wasn't respected");

		return returnSet;
	}

	public static Queue<Map<String, String>> generateExecutionPermutations(Set<String> scenarios,
			Set<String> configurations, Set<String> executionParams, Set<String> measures) {
		Queue<Map<String, String>> executions = new ArrayDeque<>();
		for (String inputData : scenarios)
			for (String tool : configurations)
				for (String platform : executionParams)
					for (String measure : measures) {
						Map<String, String> execution = new LinkedHashMap<>();
						execution.put(BRunnerKeywords.OuterLevel.INPUTDATA.kw, inputData);
						execution.put(BRunnerKeywords.OuterLevel.TOOL_PARAMETERS.kw, tool);
						execution.put(BRunnerKeywords.OuterLevel.EXECUTION_PLATFORM_PARAMETERS.kw, platform);
						execution.put(BRunnerKeywords.OuterLevel.MEASURE.kw, measure);
						executions.add(execution);
					}
		return executions;
	}

	public static Set<String> getUnused(Configuration apacheConfigurationObject) {
		Iterator<String> unusedIt = apacheConfigurationObject.getKeys();
		Set<String> unused = new HashSet<>();
		unusedIt.forEachRemaining(unused::add);
		return unused;
	}

	/**
	 * Given a key prefix, returns all the subkeys which directly follow that prefix
	 * 
	 * @param prefix
	 * @return a set containing the subkeys that follow
	 */
	public static Set<String> getTreeChildrenNames(String prefix, Configuration apacheConfigurationObject) {
		Set<String> measureConfigNames = new HashSet<>();
		Iterator<String> fullNamesIt = apacheConfigurationObject.getKeys(prefix);
		String name;
		while (fullNamesIt.hasNext()) {
			name = fullNamesIt.next().split(ParserConstants.KEY_DELIMITER_REGEX)[1].trim();
			measureConfigNames.add(name);
		}
		return measureConfigNames;
	}

	/**
	 * Takes a set of full property names and returns a map associating their suffix
	 * with the corresponding value. A line with {@code a.b.c=value} in the
	 * configuration file will correspond to a {@code ("c","value")} entry in the
	 * map
	 * 
	 * @param propertyFullNames a set of properties with their full path
	 * @param prefix
	 * @return a map associating the properties' short names with their
	 *         corresponding value
	 */
	public static Map<String, String> fillInPropertyValues(Set<String> propertyFullNames, String prefix,
			Configuration apacheConfigurationObject) {
		Map<String, String> filledInProperties = new LinkedHashMap<>();
		for (String property : propertyFullNames) {
			String d = ParserConstants.KEY_DELIMITER;
			String shortName = property.replaceFirst("([^" + d + "]+)" + d + "[^" + d + "]+", "$1");
			String propertyValue = popProperty(property, apacheConfigurationObject);
			if (propertyValue.matches(ParserConstants.UNSET_VAR_REGEX)) {
				LOG.error(ParserConstants.WARNING_PREFIX + "Detected a potentially undefined variable "
						+ "in the configuration file: " + property + "=" + propertyValue);
			}
			filledInProperties.put(shortName, propertyValue);
		}

		return filledInProperties;
	}

	/**
	 * Removes and returns the value associated with a key
	 * 
	 * @param key the key for the property to get and remove
	 * @return the value for the given key
	 */
	public static String popProperty(String key, Configuration apacheConfigurationObject) {
		String key_simple_name;
		int lastIndex = key.lastIndexOf('.');

		if (lastIndex != -1) {
			key_simple_name = key.substring(lastIndex + 1);
		} else {
			key_simple_name = key;
		}

		List<String> listValues = popPropertyList(key, apacheConfigurationObject);
		if (listValues == null || listValues.isEmpty()) {
			// if (ParserConstants.SET_TO_TRUE.stream().filter(k ->
			// k.equalsIgnoreCase(key_simple_name)).findAny().isPresent()) {
			LOG.debug("Could not find a value for " + key + " setting it to TRUE");
			return "TRUE";

			// TODO : add a dynamic mechanism to discover such keys
			// } else {
			// LOG.debug("Could not find a value for " + key + " : no value is affected to
			// this key");
			// return null;

			// }
		}
		if (listValues.size() > 1) {
			throw new IllegalArgumentException(
					"Received more than one value for " + key + " which should be sinle-valued");
		}
		return listValues.get(0);
	}

	/**
	 * Removes and returns the value associated with a key, or null if that property
	 * doens't exist
	 * 
	 * @param key the key for the property to get and remove
	 * @return the value for the given key, or null if that property doens't exist
	 */
	public static String popPropertyOrNull(String key, Configuration apacheConfigurationObject) {
		try {
			String listValues = popProperty(key, apacheConfigurationObject);
			return listValues;
		} catch (IllegalArgumentException e) {
			return null;
		}
	}

	/**
	 * Removes and returns the values associated with a key
	 * 
	 * @param key the key for the property to get and remove
	 * @return a list of values for the given key
	 */
	public static List<String> popPropertyList(String key, Configuration apacheConfigurationObject) {
		List<String> propValues = apacheConfigurationObject.getList(String.class, key);
		apacheConfigurationObject.clearProperty(key);
		return propValues;

	}

	/**
	 * Given a {@code prefix}, parses all the properties and values
	 * 
	 * @param prefix              the prefix for which sub-properties should be
	 *                            parsed
	 * @param apacheConfigurationObject a Configuration object
	 * @return a map that associates property names with a map of
	 *         {@code prefix.attributeName, attributeValue} map for injection
	 */
	public static Map<String, Map<String, String>> parsePrefix(String prefix, Configuration apacheConfigurationObject) {
		Set<String> keyNames = getTreeChildrenNames(prefix, apacheConfigurationObject);
		Map<String, Map<String, String>> parsedMap = new LinkedHashMap<>();
		for (String keyName : keyNames) {
			Set<String> fullProperties = new HashSet<>();

			Iterator<String> v = apacheConfigurationObject.getKeys(prefix + ParserConstants.KEY_DELIMITER + keyName);

			while (v.hasNext()) {
				fullProperties.add(v.next());
			}
			Map<String, String> propsAndValues = fillInPropertyValues(fullProperties, prefix,
					apacheConfigurationObject);
			parsedMap.put(keyName, propsAndValues);
		}
		LOG.debug("\nConfiguration file for " + prefix + " : " + prettyPrintToString(parsedMap));
		return parsedMap;

	}

	public static String getFileExtension(String filePath) {
		int lastIndexOfDot = filePath.lastIndexOf('.');
		int lastIndexOfSeparator = Math.max(filePath.lastIndexOf('/'), filePath.lastIndexOf('\\'));

		// Check if the last dot is after the last directory separator
		if (lastIndexOfDot > lastIndexOfSeparator) {
			return filePath.substring(lastIndexOfDot + 1);
		} else {
			return "";
		}
	}

	public static String fileNotFoundMessage(String filePath) {
		return "\n\n\n*** FILE NOT FOUND! ***\n" + filePath + "\n" + "***********************\n\n";
	}

	private static String prettyPrintToString(Map<String, Map<String, String>> map) {
		StringBuilder sb = new StringBuilder();
		map.forEach((key, value) -> {
			sb.append(key).append(": {\n");
			value.forEach((innerKey, innerValue) -> {
				sb.append("  ").append(innerKey).append(": ").append(innerValue).append("\n");
			});
			sb.append("}\n");
		});
		return sb.toString();
	}

}
