package configuration_file_parser.config;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import constants.BRunnerKeywords;

public class ConfigPreprocessor {

	private static int globalId = 1; // Start IDs from 1
	private static final Map<String, Integer> prefixCounts = new LinkedHashMap<>();
	private static final Set<String> referencedFiles = new HashSet<>();
	private static String baseDirectory = "";
	protected static Logger LOG = LoggerFactory.getLogger(ConfigPreprocessor.class);

	/**
	 * 
	 * Preprocesses the configuration file.
	 * - Resolves imports.
	 * - Handles single configurations
	 * - Adds Ids to declarations
	 * 
	 * @param inputFilePath
	 * @param outputFilePath
	 */
	public static void preprocessConfigFile(String inputFilePath, String outputFilePath) {
		String tempFilePath = inputFilePath + "_tmp";
		baseDirectory = new File(inputFilePath).getParent() + File.separator;
		// First resolve the imports and write to a temporary file
		resolveImports(inputFilePath, tempFilePath);
		// Then add the IDs based on the temporary file and write to the final output
		// file
		addIds(tempFilePath, outputFilePath);
		// Clean up the temporary file
		new File(tempFilePath).delete();
	}

	
	private static void resolveImports(String inputFilePath, String outputFilePath) {
		try (BufferedReader reader = new BufferedReader(new FileReader(inputFilePath));
				PrintWriter writer = new PrintWriter(new FileWriter(outputFilePath))) {
			String line;
			while ((line = reader.readLine()) != null) {
				if (line.trim().startsWith(BRunnerKeywords.IMPORT.kw)) {
					String importFilePath = baseDirectory + line.split("=")[1].trim();
					resolveImports(importFilePath, writer); // Recursively process the imported file
				} else {
					// Write the original line to output
					writer.println(line);
				}
			}
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	private static void resolveImports(String filePath, PrintWriter writer) throws IOException {

		if (referencedFiles.contains(filePath)) {

			LOG.error("There is a cycle in import dependencies for " + filePath);

		} else {

			referencedFiles.add(filePath);

			try (BufferedReader reader = new BufferedReader(new FileReader(filePath))) {

				String line;

				while ((line = reader.readLine()) != null) {

					if (line.trim().startsWith("import =")) {

						String importFilePath = baseDirectory + line.split("=")[1].trim();

						resolveImports(importFilePath, writer);

					} else {

						writer.println(line);
					}
				}
			}
		}
	}

	private static void addIds(String inputFilePath, String outputFilePath) {
		try (BufferedReader reader = new BufferedReader(new FileReader(inputFilePath));
				PrintWriter writer = new PrintWriter(new FileWriter(outputFilePath))) {
			String line;
			while ((line = reader.readLine()) != null) {
				if (line.trim().startsWith(BRunnerKeywords.OuterLevel.INPUTDATA.kw)
						|| line.trim().startsWith(BRunnerKeywords.OuterLevel.TOOL_PARAMETERS.kw)
						|| line.trim().startsWith(BRunnerKeywords.OuterLevel.EXECUTION_PLATFORM_PARAMETERS.kw)) {
					String prefix = line.split("\\.")[0] + "." + line.split("\\.")[1];
					if (!prefixCounts.containsKey(prefix)) {
						prefixCounts.put(prefix, globalId++);
						String added_line = prefix + "." + BRunnerKeywords.InnerLevel.PARAMETER_ID.kw + " = "
								+ prefixCounts.get(prefix);
						writer.println(added_line);
					}
				}
				writer.println(line); // Write the original line
				
			}
		} catch (IOException e) {
			e.printStackTrace();
		}
	}
}
