package configuration_file_parser.segment;

import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.configuration2.Configuration;

import configuration_file_parser.ParserConstants;
import configuration_file_parser.ParserUtils;
import constants.BRunnerKeywords;

public class PlatformParser {

	public static Map<String, Map<String, String>> parse(Configuration config) {

		Set<String> executionNames = ParserUtils
				.getTreeChildrenNames(BRunnerKeywords.OuterLevel.EXECUTION_PLATFORM_PARAMETERS.kw, config);
		Map<String, String> mapExecutionJvmArgs = new LinkedHashMap<>();
		String jvmArgs;
		for (String executionName : executionNames) {
			if ((jvmArgs = ParserUtils.popPropertyOrNull(
					BRunnerKeywords.OuterLevel.EXECUTION_PLATFORM_PARAMETERS.kw + ParserConstants.KEY_DELIMITER
							+ executionName + ParserConstants.KEY_DELIMITER + BRunnerKeywords.InnerLevel.JVM_ARGS.kw,
					config)) != null) {
				mapExecutionJvmArgs.put(executionName, jvmArgs);
			}
		}
		Map<String, Map<String, String>> executionParams = ParserUtils
				.parsePrefix(BRunnerKeywords.OuterLevel.EXECUTION_PLATFORM_PARAMETERS.kw, config);
		for (Map.Entry<String, String> entry : mapExecutionJvmArgs.entrySet()) {
			executionParams.get(entry.getKey()).put(BRunnerKeywords.InnerLevel.JVM_ARGS.kw, entry.getValue());
		}
		parseMemoryArgs(executionParams);

		return executionParams;

	}

	/**
	 * Parses the JVM memory parameters. Checks for duplicate memory settings
	 */
	private static void parseMemoryArgs(Map<String, Map<String, String>> executionParams) {
		for (Map.Entry<String, Map<String, String>> entry : executionParams.entrySet()) {
			Map<String, String> executionMap = entry.getValue();
			String memoryArgs = "";
			String memoryValue = "";
			List<String> memoryProperties = Arrays.asList(ParserConstants.MIN_MEMORY_KW, ParserConstants.MAX_MEMORY_KW);
			for (String memoryProperty : memoryProperties) {
				if ((memoryValue = executionMap.get(BRunnerKeywords.OuterLevel.EXECUTION_PLATFORM_PARAMETERS.kw
						+ ParserConstants.KEY_DELIMITER + memoryProperty)) != null) {
					String memoryArg = formatMemoryArg(memoryValue, memoryProperty.substring(0, 3));
					memoryArgs = (memoryArgs + " " + memoryArg).trim();
				}

			}
			String preExistingJvmArgs = executionMap.get("jvmArgs") ;
			
			if (!memoryArgs.isEmpty() && preExistingJvmArgs !=null
					&& preExistingJvmArgs.matches(".*-Xm[s|x][0-9]+[GgMmKk]?")) {
				throw new IllegalArgumentException(
						"Detected heap size arguments -Xmx and/or -Xms in execution." + entry.getKey()
								+ ".jvmArgs even though heap size was already defined with execution." + entry.getKey()
								+ "." + ParserConstants.MIN_MEMORY_KW + " and/or " + ParserConstants.MAX_MEMORY_KW + "."
								+ "\n\t\t  To ensure proper execution only one method must be used.");
			}
			if (preExistingJvmArgs == null)
				preExistingJvmArgs = "";
			executionMap.put("jvmArgs", (preExistingJvmArgs + " " + memoryArgs).trim());
			executionMap.remove(ParserConstants.MIN_MEMORY_KW);
			executionMap.remove(ParserConstants.MAX_MEMORY_KW);

		}
	}

	/**
	 * From a defined memory argument, returns a String formatted for JVM heap
	 * memory parameters
	 * 
	 * @param arg
	 * @param type
	 * @return a String formatted for JVM heap memory parameters
	 */
	private static String formatMemoryArg(String arg, String type) {
		String regex = "^([0-9]+)([GgMmKk]?)([Bb]?)$";
		Matcher matcher = Pattern.compile(regex).matcher(arg);
		arg = "";
		if (matcher.matches()) {
			if (type.equals("min"))
				arg = "-Xms";
			else
				arg = "-Xmx";
			arg += matcher.group(1) + matcher.group(2);
		}
		return arg;
	}

}
