package configuration_file_parser.segment;

import java.io.File;
import java.io.FileNotFoundException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.configuration2.Configuration;

import configuration_file_parser.ParserConstants;
import configuration_file_parser.ParserUtils;
import configuration_file_parser.SupportedFormatsConstants;
import constants.BRunnerKeywords;
import radicchio.FileUtils;

/**
 * Parser for the input data scenario from the configuration file
 */
public class ScenarioParser {

	/**
	 * @param config file
	 * @return Parses the scenario section of the configuration file
	 * @throws FileNotFoundException 
	 */
	public static Map<String, Map<String, String>> parse(Configuration config, String configurationFilePath)
			throws FileNotFoundException {
		Map<String, Map<String, String>> scenarios = ParserUtils.parsePrefix(BRunnerKeywords.OuterLevel.INPUTDATA.kw,
				config);

		return extractFilesFromFolders(scenarios, configurationFilePath);
	}

	protected static Map<String, Map<String, String>> extractFilesFromFolders(
			Map<String, Map<String, String>> originalMap, String configurationFilePath) throws FileNotFoundException {
		Map<String, Map<String, Set<String>>> newMap = new LinkedHashMap<>();

		for (Map.Entry<String, Map<String, String>> outerEntry : originalMap.entrySet()) {
			Map<String, Set<String>> innerMap = new LinkedHashMap<>();
			for (Map.Entry<String, String> entry : outerEntry.getValue().entrySet()) {

				if (!(entry.getKey().equalsIgnoreCase(
						BRunnerKeywords.OuterLevel.INPUTDATA.kw + "." + BRunnerKeywords.InnerLevel.DATA.kw)
						|| (entry.getKey().equalsIgnoreCase(
								BRunnerKeywords.OuterLevel.INPUTDATA.kw + "." + BRunnerKeywords.InnerLevel.RULES.kw))
						|| (entry.getKey().equalsIgnoreCase(BRunnerKeywords.OuterLevel.INPUTDATA.kw + "."
								+ BRunnerKeywords.InnerLevel.WORKLOAD.kw)))) {

					innerMap.put(entry.getKey(), Set.of(entry.getValue()));

					continue;
				}

				String resolvedFile;

				if (entry.getValue().startsWith("/")) {
					// absolute path
					resolvedFile = entry.getValue();

				} else {

					// relative path from the configuration file
					Path configFilePath = Paths.get(configurationFilePath);
					Path parentDir = configFilePath.getParent();

					resolvedFile = parentDir + "/" + entry.getValue();

				}

				// absolute path
				FileUtils.checkFile(resolvedFile);

				resolvedFile = Paths.get(resolvedFile).normalize().toAbsolutePath().toString();

				File fileOrDir = new File(resolvedFile);

				Set<String> files = new LinkedHashSet<>();

				if (fileOrDir.isFile()){
						
					if(!SupportedFormatsConstants.isSupported(ParserUtils.getFileExtension(fileOrDir.getName()))) {
					
						throw new FileNotFoundException("Unsupported file extension. Supported extensions are: " + SupportedFormatsConstants.getListOfSupportedValues());

					}
					else {
					files.add(fileOrDir.getAbsolutePath());}
				} else if (fileOrDir.isDirectory()) {
					File[] allFiles = fileOrDir.listFiles();
					Arrays.sort(allFiles, (f1, f2) -> f1.getName().compareToIgnoreCase(f2.getName()));

					if (allFiles != null) {
						for (File file : allFiles) {
							if (file.isFile() && SupportedFormatsConstants
									.isSupported(ParserUtils.getFileExtension((file.getName())))) {
								files.add(file.getAbsolutePath());
							}
						}
					}
				}

				if (files.isEmpty()) {

					throw new FileNotFoundException(ParserUtils.fileNotFoundMessage(entry.getValue()));

				}

				innerMap.put(entry.getKey(), files);
			}
			newMap.put(outerEntry.getKey(), innerMap);
		}

		return combineMap(newMap);
	}

	protected static Map<String, Map<String, String>> combineMap(Map<String, Map<String, Set<String>>> inputMap) {
		Map<String, Map<String, String>> combinedMap = new LinkedHashMap<>();

		int numberOfCombinations = 0;
		for (Map.Entry<String, Map<String, Set<String>>> outerEntry : inputMap.entrySet()) {
			Map<String, Set<String>> innerMap = outerEntry.getValue();

			// Generate all combinations
			List<Map<String, String>> combinations = new ArrayList<>();
			generateCombinations(innerMap, new LinkedHashMap<>(), combinations);

			// Process each combination
			for (Map<String, String> combination : combinations) {

				numberOfCombinations++;
				String combinedKey = outerEntry.getKey();

				if (combinations.size() > 1) {
					combinedKey += outerEntry.getKey() + ParserConstants.MULTISCENARIO_SEPARATOR + numberOfCombinations;
				}

				combinedMap.put(combinedKey, new LinkedHashMap<>(combination));
			}
		}

		return combinedMap;
	}

	protected static void generateCombinations(Map<String, Set<String>> innerMap, LinkedHashMap<String, String> current,
			List<Map<String, String>> result) {
		if (current.size() == innerMap.size()) {
			result.add(new LinkedHashMap<>(current));
			return;
		}

		String nextKey = innerMap.keySet().stream().filter(k -> !current.containsKey(k)).findFirst().orElse(null);
		if (nextKey != null) {
			for (String value : innerMap.get(nextKey)) {
				current.put(nextKey, value);
				generateCombinations(innerMap, current, result);
				current.remove(nextKey);
			}
		}
	}

}
