package configuration_file_parser.util;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

/**
 * 
 * This interface defines the behavior for a set of properties.
 * 
 *
 * @author Quentin Yeche, Federico Ulliana
 * 
 */

interface IConfigurationFilePropertySet {

	/*
	 * Returns the file path from which the configuration is built
	 */
	public String getFilePath();

	/*
	 * Returns true iff the configuration file contains a value for the given
	 * property
	 */

	public boolean containsProperty(String property);

	/*
	 * Returns the value for a given property
	 */

	public String getPropertyValue(String property);

	/*
	 * Returns the set of properties
	 */

	public default Collection<String> gePropertySet() {

		return getSubPropertySet("");

	}

	/*
	 * Returns all (property,value) couples for the set of properties starting with
	 * a given prefix
	 */

	public default boolean containsAllProperties(Collection<String> properties) {

		for (String p : properties) {

			if (!containsProperty(p)) {

				return false;

			}
		}

		return true;
	}

	/*
	 * Returns all (property,value) couples for a given set of keys.
	 */

	public default Map<String, String> getPropertyValues(Collection<String> subproperties) {

		Map<String, String> result = new HashMap<>();

		if (!containsAllProperties(subproperties)) {

			throw new IllegalArgumentException(
					"not all properties in " + subproperties + " exist in the configuration file " + getFilePath());

		}

		subproperties.forEach(x -> result.put(x, getPropertyValue(x)));

		return result;
	}

	/*
	 * Returns the set of properties starting with a given prefix
	 */

	public Collection<String> getSubPropertySet(String prefix);

	/*
	 * Returns all (property,value) couples such that property is a chain of keys
	 * that starts with a given prefix
	 */

	public default Map<String, String> getSubPropertyValues(String prefix) {

		Collection<String> subproperties = getSubPropertySet(prefix);

		if (subproperties.isEmpty()) {

			throw new IllegalArgumentException("No property found with prefix " + prefix + " in file " + getFilePath());

		}

		return getPropertyValues(subproperties);
	}

	/*
	 * Prints the list of properties
	 */

	public default String defaultStringRepresentation() {

		StringBuilder result = new StringBuilder();

		result.append("Property list :n").append("\n");

		getSubPropertyValues("").entrySet()
				.forEach(entry -> result.append(entry.getKey() + "->" + entry.getValue()).append("\n"));

		return result.toString();

	}

}
