package gg.gamerewards

import android.content.Context
import android.content.Intent
import android.util.Log
import androidx.lifecycle.ProcessLifecycleOwner
import com.google.gson.Gson
import dagger.Component
import dagger.Module
import dagger.Provides
import dagger.hilt.InstallIn
import dagger.hilt.components.SingletonComponent
import gg.gamerewards.builder.RichieExtensions
import gg.gamerewards.builder.RichieParams
import gg.gamerewards.builder.RichieUserProfile
import gg.gamerewards.data.api.ApiRepository
import gg.gamerewards.data.local.PrefManager
import gg.gamerewards.data.model.DeviceInfo
import gg.gamerewards.data.model.response.Claim
import gg.gamerewards.data.model.response.Claims
import gg.gamerewards.di.ApiModule
import gg.gamerewards.interfaces.RichieInitializationListener
import gg.gamerewards.interfaces.RichieLifecycleListener
import gg.gamerewards.interfaces.RichiePermissionRequestCallback
import gg.gamerewards.interfaces.RichieRewardCallback
import gg.gamerewards.ui.activities.offerwallActivity.OfferwallActivity
import gg.gamerewards.ui.fragments.gameDetail.GameDetailViewModel
import gg.gamerewards.ui.fragments.gameDetail.checkPointsSeeAll.CheckPointsSeeAllViewModel
import gg.gamerewards.ui.fragments.games.GamesViewModel
import gg.gamerewards.utils.PermissionManager
import javax.inject.Inject
import javax.inject.Singleton


internal object ViewModelProvider{

    private val viewModelProviderInstance: ViewModelProviderInstance by lazy {
        val viewModelComponent = DaggerViewModelComponent.factory().create()
        viewModelComponent.provider()
    }

    internal fun getInstance(): ViewModelProviderInstance = viewModelProviderInstance
}

@Component(modules = [RichieModule::class,ApiModule::class,ViewModelProviderModule::class])
@Singleton
internal interface ViewModelComponent {
    fun provider(): ViewModelProviderInstance

    @Component.Factory
    interface Factory {
        fun create(): ViewModelComponent
    }
}

@Module
@InstallIn(SingletonComponent::class)
internal class ViewModelProviderModule {

    @Singleton
    @Provides
    internal fun provideGamesViewModel(apiRepository: ApiRepository): GamesViewModel {
        return GamesViewModel(apiRepository)
    }
    @Singleton
    @Provides
    internal fun provideGameDetailViewModel(apiRepository: ApiRepository): GameDetailViewModel {
        return GameDetailViewModel(apiRepository)
    }
    @Singleton
    @Provides
    internal fun provideCheckpointSeeAllViewModel(): CheckPointsSeeAllViewModel {
        return CheckPointsSeeAllViewModel()
    }

    @Singleton
    @Provides
    internal fun provideRichieViewModel(apiRepository: ApiRepository,prefManager: PrefManager): RichieViewModel {
        return RichieViewModel(apiRepository,prefManager)
    }
}


@Singleton
internal class ViewModelProviderInstance
@Inject constructor(private val gamesViewModel: GamesViewModel,private val richieViewModel: RichieViewModel,
                    private val gamesDetailViewModel: GameDetailViewModel,private val checkPointsSeeAllViewModel: CheckPointsSeeAllViewModel) {

    internal fun provideGamesViewModel() : GamesViewModel{
        return gamesViewModel
    }
    internal fun provideGameDetailViewModel() : GameDetailViewModel{
        return gamesDetailViewModel
    }

    internal fun provideRichieViewModel() : RichieViewModel{
        return richieViewModel
    }

    internal fun provideCheckpointSeeAllViewModel() : CheckPointsSeeAllViewModel{
        return checkPointsSeeAllViewModel
    }
}

class Options(
    private var userId: String = "",
    private var applicationProcessName: String = "",
    private var params: RichieParams? = null, // Assuming RichieParams is defined elsewhere
    private var userProfile: RichieUserProfile? = null, // Assuming RichieUserProfile is defined elsewhere
    private var extensions: RichieExtensions? = null, // Assuming RichieExtensions is defined elsewhere
) {

    constructor() : this("","",null,null,null) {

    }

    fun setUserId(userId: String) = apply { this.userId = userId }
    fun setApplicationProcessName(applicationProcessName: String) = apply { this.applicationProcessName = applicationProcessName }
    fun setParams(params: RichieParams) = apply { this.params = params }
    fun setUserProfile(userProfile: RichieUserProfile) = apply { this.userProfile = userProfile }
    fun setExtensions(extensions: RichieExtensions) = apply { this.extensions = extensions }

    override fun toString(): String {
        return "Options userId : $userId, applicationProcessName : $applicationProcessName, params : ${params.toString()}, userProfile : ${userProfile.toString()}, extensions : ${extensions.toString()}"
    }
}

private val richieInstance: RichieInstance by lazy {
    val richieComponent = DaggerRichieComponent.factory().create()
    richieComponent.richie()
}

object Richie {

    private var applicationContext: Context? = null

    internal var options : Options? = null

    internal var rewardCallBack : RichieRewardCallback? = null

    internal var placementId = ""

    internal var richiePermissionRequestCallback :  RichiePermissionRequestCallback? = null

    var isInitialized : Boolean = false

    fun initialize(context: Context?,
                   apiKey : String, placementId: String, options: Options?,
                   initializationListener : RichieInitializationListener?) {
        applicationContext = context
        Log.d("Richie",options.toString())
        this.options = options
        this.placementId = placementId
        richieInstance.initialize(applicationContext,apiKey, placementId, initializationListener)
    }

    fun registerRewardCallback(rewardCallback: RichieRewardCallback){
        this.rewardCallBack = rewardCallback
    }

    fun unregisterRewardCallback(){
        this.rewardCallBack = null
    }

    fun registerCheckpointTimeBasedPermissionCallBack(richiePermissionRequestCallback : RichiePermissionRequestCallback){
        this.richiePermissionRequestCallback = richiePermissionRequestCallback
    }

    fun unregisterCheckpointTimeBasedPermissionCallBack(){
        this.richiePermissionRequestCallback = null
    }

    internal fun getContext(): Context {
        return applicationContext ?: throw IllegalStateException("Richie has not been initialized!")
    }

    fun show(context: Context) {
        richieInstance.show(context)
    }

}



@Component(modules = [RichieModule::class])
@Singleton
internal interface RichieComponent {
    fun richie(): RichieInstance

    @Component.Factory
    interface Factory {
        fun create(): RichieComponent
    }
}

@Module
@InstallIn(SingletonComponent::class)
internal class RichieModule {
    @Provides
    @Singleton
    internal fun provideInstance(prefManager: PrefManager): RichieInstance {
        return RichieInstance(prefManager)
    }

    @Provides
    internal fun provideContext(): Context {
        return Richie.getContext()
    }

    @Provides
    @Singleton
    internal fun providePrefManager() = PrefManager(provideContext())

}

@Singleton
internal class RichieInstance @Inject constructor(private val prefManager: PrefManager){
    fun show(context: Context){
        context.startActivity(Intent(context, OfferwallActivity::class.java).apply {
            flags = Intent.FLAG_ACTIVITY_SINGLE_TOP or Intent.FLAG_ACTIVITY_CLEAR_TOP or Intent.FLAG_ACTIVITY_NEW_TASK
        })
    }

    fun initialize(applicationContext: Context?, apiKey: String, placementId: String, initializationListener: RichieInitializationListener?) {
        applicationContext?.let {
            setInitializationTime()
            getPartnerInfo(it){ error ->
                error?.let { message ->
                    Richie.isInitialized = false
                    initializationListener?.initializationError(message)
                } ?: run {
                    initializeLifecycle(applicationContext, placementId)
                    Richie.isInitialized = true
                    initializationListener?.initializationSuccess()
                    cacheOfferwall(applicationContext, placementId)
                }
            }
        } ?: run {
            Richie.isInitialized = false
            throw IllegalStateException("Please initialize context properly!")
        }
    }

    private fun setInitializationTime() {
        if(prefManager.lastPlaytimeClaim == -1L)
            prefManager.lastPlaytimeClaim = System.currentTimeMillis()
    }

    private fun initializeLifecycle(context: Context, placementId: String) {
        ProcessLifecycleOwner.get().lifecycle.addObserver(
            gg.gamerewards.RichieLifecycleOwner(
                object : RichieLifecycleListener {
                    override fun onForeground() {
                        Log.d("Richie", "App is in the foreground")
                        onResume(context, placementId)
                    }

                    override fun onBackground() {
                        Log.d("Richie", "App is in the background")
                    }

                    override fun onDestroy() {
                        Log.d("Richie", "App is destroyed")
                    }
                })
        )
    }

    private fun cacheOfferwall(context: Context, placementId: String, onComplete: (() -> Unit)? = null){
        val gamesViewModel = ViewModelProvider.getInstance().provideGamesViewModel()
        gamesViewModel.getPlacementDetailsByPlacementId(context, placementId, onComplete)
    }

    private fun getPartnerInfo(context: Context, onComplete : (error : String?) -> Unit){
        val viewModel = ViewModelProvider.getInstance().provideRichieViewModel()
        Log.d("Richie",DeviceInfo.create(context).toString())
        viewModel.getPartnerInfo(prefManager, DeviceInfo.create(context)){ onComplete(it) }
    }

    fun onResume(context: Context, placementId: String){
        //TODO calculate playtime
        //TODO if reward earned and rewardCallback set, call onRewardEarned
        val viewModel = ViewModelProvider.getInstance().provideRichieViewModel()
        cacheOfferwall(context, placementId, onComplete = {
            if(PermissionManager.areRequiredPermissionsGranted(context)){
                viewModel.calculatePlaytime(context, onSuccess = {
                    getClaims()
                }, onError = {
                    getClaims()
                })
            }
            else{
                getClaims()
            }
        })
    }

    private fun getClaims(){
        val viewModel = ViewModelProvider.getInstance().provideRichieViewModel()
        Richie.rewardCallBack?.let { callback ->
            viewModel.getClaims { values ->
                if(values.isNotEmpty()){
                    callback.onRewardEarned(values)
                    val totalAmount = calculateTotalReward(values)
                    callback.onRewardEarnedAmount(totalAmount.toString())
                }
            }
        }
    }

    private fun calculateTotalReward(claimList: List<Claim>): Long{
        var sum = 0L
        claimList.forEach {
            it.coinRewardAmount?.let { rewardedAmount ->
                sum += rewardedAmount
            }
        }
        return sum
    }
}