package gg.gamerewards

import android.app.usage.UsageEvents
import android.app.usage.UsageStatsManager
import android.content.Context
import android.util.Log
import gg.gamerewards.data.api.ApiRepository
import gg.gamerewards.data.api.CommonException
import gg.gamerewards.data.local.PrefManager
import gg.gamerewards.data.model.DeviceInfo
import gg.gamerewards.data.model.Profile
import gg.gamerewards.data.model.request.PlayTimeEventsRequest
import gg.gamerewards.data.model.request.SendPlayTimeEventRequest
import gg.gamerewards.data.model.response.Claim
import gg.gamerewards.ui.base.BaseNavigator
import gg.gamerewards.ui.base.BaseViewModel
import gg.gamerewards.utils.getIOScopeForAPICall
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import java.util.Calendar


/**
 * Created by Hasan Güler on 16.05.2023.
 */



internal class RichieViewModel(private val apiRepository: ApiRepository,private val prefManager: PrefManager): BaseViewModel<BaseNavigator>() {


    private val INTERVAL_DETECTION = 5000L // Interval for detecting usage stats events (in milliseconds)
    private val HISTORY_CHECK_BUFFER = 5L // Claim history buffer timing for older claims

    fun getPartnerInfo(prefManager: PrefManager,deviceInfo : DeviceInfo,onComplete : (errorMessage : String?) -> Unit) {
        getIOScopeForAPICall(onError = { onComplete("Error") }) {
            kotlin.runCatching {
                val response = if(prefManager.hasJwtToken) apiRepository.refreshToken(deviceInfo) else apiRepository.guestLogin(deviceInfo)
                withContext(Dispatchers.Main) {
                    if (response.isSuccessful && response.body() != null && !response.body()?.loginData?.token.isNullOrEmpty()) {
                        val profileData = response.body()?.loginData?.user
                        Profile.profileData = profileData
                        Log.d("Richie",Profile.profileData.toString())
                        prefManager.jwtToken = response.body()?.loginData?.token ?: ""
                        onComplete(null)
                    } else {
                        onComplete(response.message())
                        Log.e("guestLogin", "error")
                        navigator?.handleAPIException(CommonException())
                    }
                    navigator?.hideLoading()
                }
            }.onFailure {
                onComplete(it.message)
                navigator?.handleAPIException(it)
            }
        }
    }

    fun getClaims(onSuccess: (ArrayList<Claim>) -> Unit) {
        getIOScopeForAPICall() {
            val systemTimeResponse = apiRepository.getSystemTime()
            Log.d("getClaims", "System time : ${systemTimeResponse.body()?.systemTime}")
            if (systemTimeResponse.isSuccessful) {
                if(prefManager.lastClaimHistoryCheck == -1L){
                    Log.d("getClaims", "prefManager.lastClaimHistoryCheck is -1, setting to System time : ${systemTimeResponse.body()?.systemTime}")
                    prefManager.lastClaimHistoryCheck = systemTimeResponse.body()?.systemTime ?: -1
                }
                Log.d("getClaims", "LastClaimHistoryBefore : ${prefManager.lastClaimHistoryCheck}")
                val response = apiRepository.getClaimHistory(prefManager.lastClaimHistoryCheck)
                withContext(Dispatchers.Main) {
                    if (response.isSuccessful) {
                        prefManager.lastClaimHistoryCheck = if(systemTimeResponse.body()?.systemTime != null) systemTimeResponse.body()?.systemTime!! + HISTORY_CHECK_BUFFER else -1
                        Log.d("getClaims", "LastClaimHistoryAfter : ${prefManager.lastClaimHistoryCheck}")
                        response.body()?.claims?.let {
                            Log.d("getClaims", "Claims : $it")
                            onSuccess(it)
                        }
                    } else {
                        Log.e("getClaims", "error")
                    }
                }
            } else {
                Log.e("getClaims", "systemTimeError")
            }
        }
    }

    lateinit var playTimeList : MutableList<PlaytimeGameModel>

    fun calculatePlaytime(context: Context, onSuccess: (Claim?) -> Unit, onError: () -> Unit){
        val gamesViewModel = ViewModelProvider.getInstance().provideGamesViewModel()
        playTimeList = (gamesViewModel.placementComponentList.value?.offerGroups?.flatMap { dashboardComponent ->
            dashboardComponent.offers?.map { comp ->
                PlaytimeGameModel(comp.bundleID,comp.title,0L)
            } ?: listOf()
        } ?: listOf()).toMutableList()
        val calendar = Calendar.getInstance()
        val calendarTemp = Calendar.getInstance()
        calendarTemp.add(Calendar.DAY_OF_MONTH,-7) //
        val endDate = calendar.timeInMillis // Today at midnight
        val startMillis = (prefManager.lastPlaytimeClaim) - INTERVAL_DETECTION
        val endMillis = System.currentTimeMillis() + INTERVAL_DETECTION
        //if lastPlaytimeClaim and calendar now difference is greater than 7 days, get last seven days if its not get lastPlaytimeClaim
        calendar.timeInMillis = if(calendarTemp.timeInMillis > prefManager.lastPlaytimeClaim) calendarTemp.timeInMillis else prefManager.lastPlaytimeClaim
        val startDate = calendar.timeInMillis
        val startTime: Long = startDate
        val endTime: Long = endDate
        val usageStatsManager = context.getSystemService(Context.USAGE_STATS_SERVICE) as UsageStatsManager
        val events = usageStatsManager.queryEvents(startTime, endTime)
        val usageEvents = mutableListOf<UsageEvents.Event>()

        while (events.hasNextEvent()) {
            val event = UsageEvents.Event()
            events.getNextEvent(event)
            if(playTimeList.any { it.packageName == event.packageName })
                if (event.eventType == UsageEvents.Event.ACTIVITY_RESUMED || event.eventType == UsageEvents.Event.ACTIVITY_PAUSED) {
                    usageEvents.add(event)
                }
        }

        playTimeList.forEach { element ->
            val appUsageMap = mutableMapOf<String, MutableList<Pair<Long, Long>>>()

            usageEvents.forEach { event ->
                if(event.packageName == element.packageName){
                    if (event.eventType == UsageEvents.Event.ACTIVITY_RESUMED) {
                        // Start tracking time for this package
                        appUsageMap.getOrPut(event.packageName) { mutableListOf() }.add(Pair(event.timeStamp, 0L))
                    } else if (event.eventType == UsageEvents.Event.ACTIVITY_PAUSED && appUsageMap.containsKey(event.packageName)) {
                        // Find the last foreground entry for this package and set its end time
                        appUsageMap[event.packageName]?.let { sessions ->
                            val lastSessionIndex = sessions.size - 1
                            if (lastSessionIndex >= 0 && sessions[lastSessionIndex].second == 0L) {
                                sessions[lastSessionIndex] = sessions[lastSessionIndex].copy(second = event.timeStamp)
                            }
                        }
                    }
                }
            }
            var totalPlayTime = 0L
            appUsageMap.forEach { packageItem ->
                packageItem.value.forEach {
                    if(it.first >= startMillis && it.second <= endMillis && it.second != 0L)
                        totalPlayTime += it.second - it.first
                }
            }
            playTimeList[playTimeList.indexOfFirst { it.packageName == element.packageName }] =
                playTimeList[playTimeList.indexOfFirst { it.packageName == element.packageName }].copy(playTime = totalPlayTime)
        }
        println(playTimeList)
        sendPlayTimeEvents(playTimeList,onSuccess = {
            prefManager.lastPlaytimeClaim = Calendar.getInstance().timeInMillis
            onSuccess(it)
        },onError)
    }


    private fun sendPlayTimeEvents(playTimes: List<PlaytimeGameModel>?, onSuccess: (claim: Claim?) -> Unit, onError: () -> Unit) {
        getIOScopeForAPICall(handleAPIExceptions = false, onError = onError) {
            val systemTimeResponse = apiRepository.getSystemTime()
            if(systemTimeResponse.isSuccessful && systemTimeResponse.body()?.systemTime != null){
                val playTimeRequestList = playTimes?.map { SendPlayTimeEventRequest(
                    (it.playTime ?: 0) / 1000,it.packageName,
                    gg.gamerewards.utils.RandomStringGenerator.generateWithTimestamp(systemTimeResponse.body()?.systemTime)) }?.filter { (it.playTime ?: 0) > 0 }
                val response = apiRepository.sendPlayTimeEvents(PlayTimeEventsRequest(playTimes = playTimeRequestList))
                withContext(Dispatchers.Main) {
                    if (response.isSuccessful) {
                        onSuccess(response.body()?.claim)
                    } else {
                        onError()
                        Log.e("sendPlayTimeEvent", "error")
                    }
                }
            }
            else {
                onError()
                Log.e("getSystemTime", "error")
            }
        }
    }

    fun customPlaytimeEvent(playTimes: List<PlaytimeGameModel>?, onSuccess: (claim: Claim?) -> Unit, onError: () -> Unit){
        sendPlayTimeEvents(playTimes?.take(1),onSuccess,onError)
    }

    fun consumeClaimRequest(claimId : Int?, onError: (String) -> Unit, onSuccess: () -> Unit) {
        claimId?.let { id ->
            getIOScopeForAPICall(onError = {
                onError("Api Error")
            }) {
                val response = apiRepository.consumeClaim(id)
                withContext(Dispatchers.Main) {
                    if (response.isSuccessful) {
                        onSuccess()
                    } else {
                        onError(response.body().toString())
                    }
                }
            }
        } ?: kotlin.run {
            onError("Claim Id Cannot Be Null")
            return
        }
    }
    fun consumeMultipleClaimsRequest(claimIds : List<Int?>?, onError: (String) -> Unit, onSuccess: () -> Unit) {
        claimIds?.filterNotNull()?.let { id ->
            getIOScopeForAPICall(onError = {
                onError("Api Error")
            }) {
                //TODO consume multiple claims
               /* val response = apiRepository.consumeClaim(id)
                withContext(Dispatchers.Main) {
                    if (response.isSuccessful) {
                        onSuccess()
                    } else {
                        onError(response.body().toString())
                    }
                }*/
                onSuccess()
            }
        } ?: kotlin.run {
            onError("Claim Id Cannot Be Null")
            return
        }
    }

}

data class PlaytimeGameModel(val packageName : String?,val name : String?,val playTime: Long?)
