package gg.gamerewards.ui.base

import android.os.Bundle
import android.os.Handler
import android.os.Looper
import android.view.LayoutInflater
import androidx.appcompat.app.AlertDialog
import androidx.appcompat.app.AppCompatActivity
import androidx.appcompat.widget.AppCompatButton
import androidx.cardview.widget.CardView
import androidx.lifecycle.setViewTreeLifecycleOwner
import androidx.navigation.NavController
import androidx.viewbinding.ViewBinding
import com.scottyab.rootbeer.RootBeer
import gg.gamerewards.R
import gg.gamerewards.data.api.DetectVpnException
import gg.gamerewards.data.local.PrefManager
import gg.gamerewards.utils.Constants
import gg.gamerewards.utils.isVpnDetected
import gg.gamerewards.utils.setDialogWidthPercentageBasedOnScreenSize
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import java.io.IOException

internal abstract class BaseActivity<VB : ViewBinding> : AppCompatActivity() {

    private lateinit var _binding: ViewBinding
    abstract val bindingInflater: (LayoutInflater) -> VB
    private var commonWarningDialog: AlertDialog? = null
    lateinit var prefManager: PrefManager
    private var rootDialog: AlertDialog? = null

    @Suppress("UNCHECKED_CAST")
    protected val binding: VB
        get() = _binding as VB

    abstract fun setupUi(savedInstanceState: Bundle?)

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        _binding = bindingInflater.invoke(layoutInflater)
        _binding.root.setViewTreeLifecycleOwner(this)
        prefManager = PrefManager(applicationContext)
        setContentView(_binding.root)
        setupUi(savedInstanceState)
    }

    fun handleAPIException(ex: Throwable) {
        when (ex) {
            is DetectVpnException -> {
                showVpnDetectionWarning()
            }
        }
    }

    override fun onResume() {
        super.onResume()
        //vpn check
        if (isVpnDetected(prefManager)) {
            handleAPIException(DetectVpnException())
        }

        //root check
        CoroutineScope(Dispatchers.IO).launch {
            val rootBeer = RootBeer(this@BaseActivity)
            rootBeer.setLogging(false)
            if (rootBeer.isRooted) {
                CoroutineScope(Dispatchers.Main).launch {
                    showRootDetectionDialog()
                    Handler(Looper.getMainLooper()).postDelayed({
                        finish()
                    }, 5000)
                }
            }
        }
    }

    private fun showVpnDetectionWarning() {
        if (!isFinishing && !isDestroyed) {
            Handler(Looper.getMainLooper()).post {
                commonWarningDialog?.dismiss()
                commonWarningDialog = null
                commonWarningDialog =
                    AlertDialog.Builder(this, R.style.AlertDialogTransparentBackground90Width)
                        .create()
                val view =
                    LayoutInflater.from(this).inflate(R.layout.vpn_detect_layout, null)
                view.findViewById<CardView>(R.id.cv_container)
                    .setDialogWidthPercentageBasedOnScreenSize(this)
                commonWarningDialog?.setView(view)
                val retryButton: AppCompatButton = view.findViewById(R.id.btn_retry)
                retryButton.setOnClickListener {
                    try {
                        if (!isVpnDetected(prefManager)) {
                            commonWarningDialog?.dismiss()
                            commonWarningDialog = null
                        }
                    } catch (e: IOException) {
                        e.printStackTrace()
                    }
                }
                commonWarningDialog?.setCancelable(false)
                commonWarningDialog?.show()
            }
        } else {
            commonWarningDialog = null
        }
    }

    private fun showRootDetectionDialog() {
        Handler(Looper.getMainLooper()).post {
            rootDialog?.dismiss()
            rootDialog = null
            rootDialog =
                AlertDialog.Builder(this, R.style.FullScreenAlertDialog)
                    .create()
            rootDialog?.setCancelable(false)
            val view = LayoutInflater.from(this).inflate(R.layout.dialog_rooted_device_error, null)
            view.findViewById<CardView>(R.id.cv_container)
                .setDialogWidthPercentageBasedOnScreenSize(this)

            val button = view.findViewById<AppCompatButton>(R.id.btn_error)
            button.setOnClickListener {
                finish()
            }
            rootDialog?.setView(view)
            rootDialog?.show()
        }
    }
    fun navigate(
        navController : NavController?,
        toFragmentId: Int,
        redirectFrom: String?,
        presentToUp: Boolean = false,
        extras: Bundle? = null
    ) {
        if (isFinishing) {
            return
        }
        extras?.putString(Constants.EXTRA_REDIRECT, redirectFrom)

        runCatching {
            val navOptions = androidx.navigation.navOptions {
                if (presentToUp) {
                    anim {
                        enter = R.anim.slide_to_up
                    }
                } else {
                    anim {
                        enter = R.anim.fragment_in
                        exit = R.anim.fragment_out
                        popEnter = R.anim.fragment_pop_in
                        popExit = R.anim.fragment_pop_out
                    }
                }
                launchSingleTop = true
            }
            navController?.navigate(toFragmentId, extras, navOptions)
        }
    }
}