package gg.gamerewards.ui.fragments.gameDetail

import android.annotation.SuppressLint
import android.content.Context
import android.content.res.ColorStateList
import android.util.Log
import android.view.LayoutInflater
import android.view.ViewGroup
import android.widget.ProgressBar
import android.widget.TextView
import androidx.constraintlayout.widget.ConstraintLayout
import androidx.core.content.ContextCompat
import androidx.core.view.isVisible
import androidx.lifecycle.LifecycleOwner
import androidx.recyclerview.widget.RecyclerView
import com.airbnb.lottie.LottieAnimationView
import com.bumptech.glide.Glide
import com.bumptech.glide.load.resource.bitmap.CircleCrop
import com.skydoves.balloon.ArrowOrientation
import com.skydoves.balloon.ArrowPositionRules
import com.skydoves.balloon.Balloon
import com.skydoves.balloon.BalloonAnimation
import com.skydoves.balloon.BalloonSizeSpec
import com.skydoves.balloon.OnBalloonClickListener
import dagger.hilt.android.qualifiers.ApplicationContext
import gg.gamerewards.R
import gg.gamerewards.data.model.response.CheckPoint
import gg.gamerewards.data.model.response.CheckpointViewType
import gg.gamerewards.databinding.ItemCheckpointBinding
import gg.gamerewards.databinding.ItemCheckpointTimeBasedBinding
import gg.gamerewards.databinding.ItemEmptyViewHolderBinding
import gg.gamerewards.ui.fragments.games.GamesAdapter
import gg.gamerewards.utils.CircleBorderTransformation
import gg.gamerewards.utils.CompactNumberFormatter
import gg.gamerewards.utils.LockedCheckpointDialog
import gg.gamerewards.utils.PermissionManager
import javax.inject.Inject


/**
 * Created by Hasan Güler on 26.05.2023.
 */

interface CheckpointTimeBasedPermissionListener{
    fun requestPermissions()
}


class CheckPointAdapter @Inject constructor(@ApplicationContext val context: Context,
                                            private val isPlaytimeHandledByOffer : Boolean?,
                                            private val checkpointTimeBasedPermissionListener: CheckpointTimeBasedPermissionListener?,
                                            val onPermitAndEarnClicked: () -> Unit
                                            ) :
    RecyclerView.Adapter<RecyclerView.ViewHolder>() {
    private var items: ArrayList<CheckPoint> = arrayListOf()
    val numberFormatter = CompactNumberFormatter()


    override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): RecyclerView.ViewHolder {
        return when (viewType) {
            CheckpointViewType.LEVEL.order -> {
                CheckPointViewHolder(
                    ItemCheckpointBinding.inflate(
                        LayoutInflater.from(
                            parent.context
                        ), parent, false
                    )
                )
            }
            CheckpointViewType.TIME_BASED.ordinal -> {
                TimeBasedCheckPointViewHolder(
                    ItemCheckpointTimeBasedBinding.inflate(
                        LayoutInflater.from(
                            parent.context
                        ), parent, false
                    )
                )
            }
            else -> GamesAdapter.EmptyViewHolder(
                ItemEmptyViewHolderBinding.inflate(
                    LayoutInflater.from(
                        parent.context
                    ), parent, false
                )
            )
        }
    }

    override fun onBindViewHolder(holder: RecyclerView.ViewHolder, position: Int) {
        val currentItem = getCurrentItem(position)
        when (holder) {
            is CheckPointViewHolder -> holder.bind(currentItem, context, position == itemCount - 1,position)
            is TimeBasedCheckPointViewHolder -> holder.bind(currentItem,context,position == itemCount - 1,position)
        }
    }

    private fun getCurrentItem(position: Int): CheckPoint = items[position]

    override fun getItemViewType(position: Int): Int {
        return items[position].checkpointType.order
    }

    override fun getItemCount(): Int = items.count()

    fun submitList(addedItems: ArrayList<CheckPoint>?) {
        this.items.clear()
        if (addedItems != null) {
            this.items = ArrayList(addedItems)
        }

        notifyDataSetChanged()
    }

    inner class CheckPointViewHolder(val binding: ItemCheckpointBinding) :
        RecyclerView.ViewHolder(binding.root) {

        @SuppressLint("SetTextI18n")
        fun bind(
            model: CheckPoint,
            context: Context,
            isLastItem: Boolean,
            position: Int
        ) {
            try {
                binding.tvTitle.text = model.title
                binding.tvSubTitle.text = model.subtitle
                Glide.with(context)
                    .load(model.iconUrl)
                    .transform(CircleCrop(), CircleBorderTransformation())
                    .into(binding.ivIcon)

                /** If the checkpoint is a task that needs to be completed all at once, don't show progress */
                val isShowStatusTextView = if (model.isCountBased != true) (model.targetValue != null && model.targetValue == 1.0) else false
                binding.progressBar.isVisible = !isShowStatusTextView
                binding.clStatus.isVisible = isShowStatusTextView && !model.isCompleted

                binding.apply {
                    /** set count based checkpoints */
                    setCountBasedCheckpoint(model,progressBar,clMainContainer, tvCount)
                    /** set conditional checkpoints */
                    setConditionalCheckpoint(model,lvLockAnimation, progressBar, clMainContainer)

                    setDefaultViews(model,progressBar,!isShowStatusTextView,clMainContainer)
                }

                setProgressColors(model,context)
                binding.separator.isVisible = !isLastItem
                binding.progressBar.progress = model.viewProgress

            } catch (e: Exception) {
                e.printStackTrace()
            }
        }

        private fun setProgressColors(model: CheckPoint, context: Context) {
            binding.tvCoinAmount.text = if(model.isCompleted) context.getString(R.string.game_detail_checkpoint_completed) else {
                if (model.coinRewardAmount == null || model.coinRewardAmount == 0L) {
                    context.getString(R.string.coin_up_to, numberFormatter.format(model.coinUpToAmount))
                } else {
                    numberFormatter.format(model.coinRewardAmount.toLong())
                }
            }
            binding.tvCoinAmount.setTextColor(if(model.isCompleted) context.getColor(R.color.green_primary) else context.getColor(R.color.purple_dark))
            binding.ivCoin.visibility = if (model.isCompleted) android.view.View.GONE else android.view.View.VISIBLE
            binding.progressBar.progressTintList = ColorStateList.valueOf(if(model.isCompleted) context.getColor(R.color.green_primary) else context.getColor(R.color.yellow_primary))
        }
    }
    inner class TimeBasedCheckPointViewHolder(val binding: ItemCheckpointTimeBasedBinding) :
        RecyclerView.ViewHolder(binding.root) {

        @SuppressLint("SetTextI18n")
        fun bind(
            model: CheckPoint,
            context: Context,
            isLastItem: Boolean,
            position: Int
        ) {
            try {
                binding.apply {

                    clPermission.isVisible = (!PermissionManager.areRequiredPermissionsGranted(context))
                    clPermission.setOnClickListener {

                    }
                            //&& isPlaytimeHandledByOffer == false

                    llPermission.setOnClickListener {
                        checkpointTimeBasedPermissionListener?.requestPermissions()
                        onPermitAndEarnClicked.invoke()
                    }

                    ivQuestion.setOnClickListener {

                        val balloon: Balloon = Balloon.Builder(context)
                            .setArrowSize(10)
                            .setArrowOrientation(ArrowOrientation.TOP)
                            .setArrowPositionRules(ArrowPositionRules.ALIGN_ANCHOR)
                            .setArrowPosition(0.5f)
                            .setWidth(BalloonSizeSpec.WRAP)
                            .setHeight(BalloonSizeSpec.WRAP)
                            .setTextSize(11f)
                            .setCornerRadius(4f)
                            .setPadding(8)
                            .setText(context.getString(R.string.time_based_checkpoint_tooltip_text))
                            .setTextColor(ContextCompat.getColor(context, R.color.grayscale400))
                            .setTextIsHtml(true)
                            .setBackgroundColor(
                                ContextCompat.getColor(
                                    context,
                                    R.color.grayscale200
                                )
                            )
                            .setBalloonAnimation(BalloonAnimation.FADE)
                            .setLifecycleOwner(context as? LifecycleOwner)
                            .build()
                        balloon.setOnBalloonClickListener(OnBalloonClickListener {
                            balloon.dismiss()
                        })
                        balloon.showAlignBottom(ivQuestion)
                    }

                    /** set count based checkpoints */
                    setCountBasedCheckpoint(model,progressBar,clMainContainer, tvCount)
                    /** set conditional checkpoints */
                    setConditionalCheckpoint(model,lvLockAnimation, progressBar, clMainContainer,clPermission)

                    setDefaultViews(model,progressBar,true,clMainContainer)

                    tvTitle.text = model.title
                    tvSubTitle.text = model.subtitle
                    Glide.with(context)
                        .load(model.iconUrl)
                        .transform(CircleCrop(),CircleBorderTransformation())
                        .into(ivIcon)

                    setProgressColors(model,context)
                    separator.isVisible = !isLastItem
                    progressBar.progress = model.viewProgress
                }

            } catch (e: Exception) {
                e.printStackTrace()
            }
        }

        private fun setProgressColors(model: CheckPoint, context: Context) {
            binding.apply {
                tvCoinAmount.text = if(model.isCompleted) context.getString(R.string.game_detail_checkpoint_completed) else {
                    if (model.coinRewardAmount == null || model.coinRewardAmount == 0L) {
                        context.getString(R.string.coin_up_to, numberFormatter.format(model.coinUpToAmount)
                        )
                    } else {
                        numberFormatter.format(model.coinRewardAmount.toLong())

                    }
                }
                tvCoinAmount.setTextColor(if(model.isCompleted) context.getColor(R.color.green_primary) else context.getColor(R.color.purple_dark))
                ivCoin.visibility = if (model.isCompleted) android.view.View.GONE else android.view.View.VISIBLE
                progressBar.progressTintList = ColorStateList.valueOf(if(model.isCompleted) context.getColor(R.color.green_primary) else context.getColor(R.color.yellow_primary))
            }

        }
    }
    private fun setConditionalCheckpoint(model: CheckPoint, lvLockAnimation: LottieAnimationView, progressBar: ProgressBar, clMainContainer: ConstraintLayout, permissionButtonContainer: ConstraintLayout? = null){
        if (model.isBlocked == true){
            runCatching {
                lvLockAnimation.isVisible = true
                progressBar.isVisible = false
                permissionButtonContainer?.isVisible = false
                clMainContainer.setBackgroundResource(R.drawable.bg_locked_checkpoint)
                clMainContainer.setOnClickListener {
                    val lockedCheckpointDialog = LockedCheckpointDialog(
                        context,model.iconUrl,
                        model.title,
                        model.subtitle,
                        model.blockedReasonText
                    )
                    lockedCheckpointDialog.show()
                }
            }
        } else {
            lvLockAnimation.isVisible = false
            permissionButtonContainer?.isVisible = (!PermissionManager.areRequiredPermissionsGranted(context))
                    //&& isPlaytimeHandledByOffer == false
            clMainContainer.setOnClickListener {
                onPermitAndEarnClicked.invoke()
            }
        }
    }
    private fun setCountBasedCheckpoint(model: CheckPoint, progressBar: ProgressBar, clMainContainer: ConstraintLayout, tvCount: TextView){
        if (model.isCountBased == true && model.targetCount != null) {
            runCatching {
                clMainContainer.setBackgroundResource(R.drawable.bg_count_based_checkpoint)
                val isInfiniteCount = model.targetCount == -1.0 //If value is -1, do not show progress and show infinity sign
                progressBar.isVisible = !isInfiniteCount
                tvCount.isVisible = true //show remaining number of rewards
                tvCount.text = "${model.currentCount?.toInt() ?: 0} / ${if (!isInfiniteCount) model.targetCount.toInt() else "∞"}"
            }.onFailure {
                Log.e("CheckpointAdapter", it.message, it)
            }
        } else {tvCount.isVisible = false }
    }

    private fun setDefaultViews(model: CheckPoint, progressBar: ProgressBar, isProgressbarVisible: Boolean, clMainContainer: ConstraintLayout){
        if (model.isBlocked != true && model.isCountBased != true){
            progressBar.isVisible = isProgressbarVisible
            clMainContainer.setBackgroundColor(ContextCompat.getColor(context,R.color.white))
        }
    }
}