package gg.gamerewards.utils

import android.annotation.SuppressLint
import android.app.Activity
import android.content.Context
import android.content.Intent
import android.content.pm.PackageManager
import android.content.res.Resources
import android.graphics.Color
import android.net.Uri
import android.os.Build
import android.os.Handler
import android.os.Looper
import android.provider.Settings
import androidx.core.app.ShareCompat
import gg.gamerewards.R
import kotlinx.coroutines.CoroutineExceptionHandler
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import java.util.UUID

internal object RichieUtils {
    fun isAppInstalled(packageName: String, context: Context): Boolean {
        return try {
            context.packageManager.getPackageInfo(packageName, PackageManager.GET_ACTIVITIES)
            true // Package exists
        } catch (_: Exception) {
            false // Package not found
        }
    }

    fun openLink(activity: Activity?, url: String) {
        val intent = Intent(Intent.ACTION_VIEW, Uri.parse(url))
        activity?.startActivity(intent)
    }

    fun shareText(context: Context, text: String) {
        ShareCompat.IntentBuilder(context)
            .setType("text/plain")
            .setChooserTitle("Share")
            .setText(text)
            .startChooser()
    }

    /**
     * Returns whether the device is known as an emulator.
     *
     * @return whether the device is known as an emulator.
     */
    fun isEmulator(): Boolean {
        return (Build.BRAND.startsWith("generic") && Build.DEVICE.startsWith("generic")
                || Build.FINGERPRINT.startsWith("generic")
                || Build.FINGERPRINT.startsWith("unknown")
                || Build.HARDWARE.contains("goldfish")
                || Build.HARDWARE.contains("ranchu")
                || Build.MODEL.contains("google_sdk")
                || Build.MODEL.contains("Emulator")
                || Build.MODEL.contains("Android SDK built for x86")
                || Build.MANUFACTURER.contains("Genymotion")
                || Build.PRODUCT.contains("sdk_google")
                || Build.PRODUCT.contains("google_sdk")
                || Build.PRODUCT.contains("sdk")
                || Build.PRODUCT.contains("sdk_x86")
                || Build.PRODUCT.contains("vbox86p")
                || Build.PRODUCT.contains("emulator")
                || Build.PRODUCT.contains("simulator"))

    }

    fun isTablet(context: Context): Boolean {
        return context.isLargeScreen()
    }

    fun getScreenHeight(): Int {
        return Resources.getSystem().displayMetrics.heightPixels
    }

    fun parseColor(colorHex: String?): Int {
        if (colorHex.isNullOrEmpty() || colorHex.replace(
                "#", ""
            ).length % 3 != 0
        ) return R.color.white
        return Color.parseColor(colorHex)
    }

    fun getIOScope(): CoroutineScope =
        CoroutineScope(Dispatchers.IO + SupervisorJob() + CoroutineExceptionHandler { _, throwable ->
            Handler(Looper.getMainLooper()).post {
                throwable.printStackTrace()
            }
        })

    fun formatAsClockTime(milliseconds: Long): String {
        val seconds = (milliseconds / 1000) % 60
        val minutes = (milliseconds / (1000 * 60)) % 60
        val hours = milliseconds / (1000 * 60 * 60)

        return String.format("%02d:%02d:%02d", hours, minutes, seconds)
    }

    @SuppressLint("HardwareIds")
    fun getDeviceId(context: Context): String {
        // 1. Try ANDROID_ID
        val androidId = Settings.Secure.getString(
            context.contentResolver, Settings.Secure.ANDROID_ID
        )
        if (!androidId.isNullOrEmpty() && androidId != "9774d56d682e549c") { // '9774d56d682e549c' is a known bad value from some devices
            return androidId
        }

        // 2. Try SERIAL (deprecated in API 29 and higher, but can be used for older devices)
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.Q) {
            val serial = Build.SERIAL
            if (!serial.isNullOrEmpty() && serial != "unknown") {
                return serial
            }
        }

        // 3. Fallback to app generated ID, stored in SharedPreferences for persistence
        val prefs = context.getSharedPreferences("app_prefs", Context.MODE_PRIVATE)
        var generatedId = prefs.getString("GENERATED_UNIQUE_ID", null)
        if (generatedId == null) {
            generatedId = UUID.randomUUID().toString()
            prefs.edit().putString("GENERATED_UNIQUE_ID", generatedId).apply()
        }

        return generatedId
    }

    fun wrapApplicationContext(original: Context?): Context? {
        val target = original?.applicationContext
        return if (target != null && target !== original) target else original
    }
}