package group.flyfish.rest.configuration;

import group.flyfish.rest.core.factory.DefaultHttpClientProvider;
import group.flyfish.rest.core.factory.HttpClientFactoryBean;
import group.flyfish.rest.core.factory.HttpClientProvider;
import group.flyfish.rest.mapping.RestResultMapping;
import group.flyfish.rest.registry.RestApiRegistry;
import group.flyfish.rest.registry.proxy.support.RestArgumentResolver;
import group.flyfish.rest.registry.proxy.support.RestArgumentResolverComposite;
import group.flyfish.rest.registry.proxy.support.resolvers.*;
import group.flyfish.rest.utils.DataUtils;
import org.apache.http.impl.client.CloseableHttpClient;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Bean;

import java.util.Arrays;
import java.util.List;

/**
 * rest请求相关配置
 *
 * @author wangyu
 */
public class RestClientConfiguration {

    /**
     * 实例化参数bean
     *
     * @return 结果
     */
    @ConfigurationProperties(prefix = "rest.client", ignoreUnknownFields = false)
    @Bean
    public RestClientProperties restClientProperties() {
        return new RestClientProperties();
    }

    /**
     * http client工厂bean
     *
     * @return 结果
     */
    @Bean
    @ConditionalOnMissingBean(CloseableHttpClient.class)
    public HttpClientFactoryBean httpClientFactoryBean() {
        return new HttpClientFactoryBean();
    }

    /**
     * 创建默认的提供者，默认使用factory bean创建的client
     *
     * @return 结果
     */
    @Bean
    @ConditionalOnMissingBean
    public HttpClientProvider httpClientProvider() {
        return new DefaultHttpClientProvider();
    }

    /**
     * 注册rest自动代理
     *
     * @return 结果
     */
    @Bean
    public RestApiRegistry restApiRegistry(RestArgumentResolverComposite composite, HttpClientProvider provider,
                                           List<RestResultMapping> mappings) {
        // 先注册映射们
        if (DataUtils.isNotEmpty(mappings)) {
            mappings.forEach(mapping -> RestResultMapping.MAPPINGS.put(mapping.getClass(), mapping));
        }
        // 最后实例化
        return new RestApiRegistry(composite, provider);
    }

    /**
     * 一个很重要的bean，反向解析各种参数
     *
     * @return 结果
     */
    @Bean
    public RestArgumentResolverComposite restArgumentResolverComposite() {
        List<RestArgumentResolver> resolvers = Arrays.asList(
                new RestPathParamArgumentResolver(),
                new RestPartArgumentResolver(),
                new RestPartArgumentResolver.FilenameResolver(),
                new RestBodyArgumentResolver(),
                new RestHeaderArgumentResolver(),
                new RestParamArgumentResolver()
        );
        return new RestArgumentResolverComposite(resolvers);
    }
}
