package group.flyfish.rest.configuration;

import group.flyfish.rest.configuration.configure.PropertiesConfigurable;
import group.flyfish.rest.configuration.modifier.RestPropertiesModifier;
import group.flyfish.rest.core.auth.RestAuthProvider;
import group.flyfish.rest.registry.proxy.RestInvokers;
import group.flyfish.rest.utils.DataUtils;
import lombok.AccessLevel;
import lombok.Data;
import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Autowired;

import java.time.Duration;
import java.util.HashMap;
import java.util.Map;

/**
 * 客户端的配置
 *
 * @author wangyu
 * 整合入spring boot，简化配置，提高业务复用性
 */
@Data
@Slf4j
public class RestClientProperties implements InitializingBean {

    /**
     * 超时时间，默认30s
     */
    private Duration connectionTimeout = Duration.ofSeconds(30);

    /**
     * 基本url
     */
    private String baseUrl;

    /**
     * ssl无条件新人
     */
    private Boolean alwaysTrust = true;

    /**
     * 定义的内容字典，可以支持动态取值，使用#variable
     */
    private Map<String, String> urls = new HashMap<>();

    /**
     * 默认的认证提供者
     */
    private RestAuthProvider authProvider;

    /**
     * 修改器们
     */
    @Getter(AccessLevel.NONE)
    @Setter(onMethod_ = @Autowired)
    private ObjectProvider<RestPropertiesModifier> modifiers;

    /**
     * 配置感知项
     */
    @Getter(AccessLevel.NONE)
    @Setter(onMethod_ = @Autowired)
    private ObjectProvider<PropertiesConfigurable> configures;

    /**
     * 加载感知
     */
    @Getter(AccessLevel.NONE)
    @Setter(onMethod_ = @Autowired)
    private ObjectProvider<RestLoadedAware> aware;

    /**
     * 获取字典url
     *
     * @param key 键
     * @return 结果
     */
    public String getDictUrl(String key) {
        if (DataUtils.isEmpty(urls)) {
            return null;
        }
        return urls.get(key);
    }

    @Autowired
    public void setDefaultAuthProvider(ObjectProvider<RestAuthProvider> provider) {
        this.authProvider = provider.getIfAvailable();
    }

    @Override
    public void afterPropertiesSet() {
        // 配置修改
        modifiers.forEach(modifier -> modifier.modify(this));
        // 服务初始化
        RestInvokers.configure(this);
        // 配置感知
        configures.forEach(item -> item.configure(this));
        // 加载感知
        aware.forEach(RestLoadedAware::onClientsLoaded);
    }
}
