package group.flyfish.rest.core.client;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JavaType;
import group.flyfish.rest.constants.RestConstants;
import group.flyfish.rest.core.exception.RestClientException;
import group.flyfish.rest.core.factory.HttpClientFactoryBean;
import group.flyfish.rest.core.factory.HttpClientProvider;
import group.flyfish.rest.enums.ResponseType;
import group.flyfish.rest.utils.JacksonUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.HttpEntity;
import org.apache.http.StatusLine;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.conn.ConnectTimeoutException;
import org.apache.http.util.EntityUtils;
import org.springframework.http.HttpStatus;
import org.springframework.lang.Nullable;
import org.springframework.util.StreamUtils;

import java.io.IOException;
import java.io.InputStream;
import java.net.UnknownHostException;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.function.Consumer;

import static group.flyfish.rest.constants.RestConstants.DEFAULT_EXECUTOR;

/**
 * Rest请求客户端 Apache http实现
 *
 * @author Mr.Wang
 * <p>
 * @apiNote 1. 全builder调用，用户系统内部相互通信
 * 2. 支持异步回调
 * 3. 多样性组合
 * 4. 解耦实现
 * 5. 支持上传文件、FormData，JSON、支持自定义无侵入扩展
 * 6. 新增单例httpClient模式，复用连接让应用更高效
 */
@Slf4j
final class DefaultRestClient extends RestErrorHandler implements RestClient {

    private final HttpRequestBase request;
    private boolean async = false;
    private Consumer<HttpEntity> consumer;
    private ExecutorService executorService;
    private ResponseType responseType = ResponseType.NORMAL;
    private Class<?> resultClass;
    private TypeReference<?> typeReference;
    private JavaType resultType;
    private HttpClientProvider clientProvider;

    /**
     * 内部构造方法，不对外公开
     *
     * @param request 请求信息
     */
    DefaultRestClient(HttpRequestBase request, HttpClientProvider provider) {
        this.request = request;
        this.clientProvider = provider;
    }

    /**
     * 内部构造方法，不对外公开
     *
     * @param request 请求信息
     */
    DefaultRestClient(HttpRequestBase request) {
        this.request = request;
        this.clientProvider = HttpClientFactoryBean::createSSLClient;
    }

    /**
     * 设置请求失败时的回调
     *
     * @param errorConsumer 错误回调
     * @return 结果
     */
    @Override
    public RestClient onError(Consumer<RestClientException> errorConsumer) {
        this.errorConsumer = errorConsumer;
        return this;
    }


    /**
     * 设置响应类型
     *
     * @param responseType 响应类型
     * @return 结果
     */
    @Override
    public RestClient responseType(ResponseType responseType) {
        this.responseType = responseType;
        return this;
    }

    /**
     * 设置客户端提供者
     *
     * @param provider 客户端提供者
     */
    @Override
    public void setClient(HttpClientProvider provider) {
        this.clientProvider = provider;
    }

    /**
     * 标记线程池执行
     *
     * @return 结果
     */
    @Override
    public RestClient async() {
        this.async = true;
        this.executorService = DEFAULT_EXECUTOR;
        return this;
    }

    /**
     * 标记指定线程池执行
     *
     * @param executorService 线程池
     * @return 结果
     */
    @Override
    public RestClient async(ExecutorService executorService) {
        this.async = true;
        this.executorService = executorService;
        return this;
    }

    /**
     * 异步执行，接收结果
     *
     * @param consumer 结果
     */
    @Override
    public void execute(Consumer<HttpEntity> consumer) {
        this.consumer = consumer;
        if (this.async) {
            if (this.executorService == null) {
                handleError(RestConstants.MSG_THREAD_POOL_EMPTY);
            }
            this.executorService.submit(this::executeSafety);
        } else {
            executeSafety();
        }
    }

    /**
     * 静默执行，抛弃全部异常
     */
    @Override
    public void executeSilent() {
        executeSafety();
    }

    /**
     * 执行请求，返回Map
     *
     * @return map
     * @throws IOException 异常
     */
    @Override
    public Map<String, Object> executeForMap() throws IOException {
        this.responseType = ResponseType.JSON;
        return innerExecute();
    }

    /**
     * 执行请求，返回字符串
     *
     * @return 字符串
     * @throws IOException 异常
     */
    @Override
    public String executeForString() throws IOException {
        this.responseType = ResponseType.TEXT;
        return innerExecute();
    }

    /**
     * 安全的执行
     */
    private void executeSafety() {
        try {
            execute();
        } catch (IOException e) {
            handleError(RestConstants.MSG_IO_ERROR, e);
        }
    }

    /**
     * 执行并序列化，该方法会安全的返回对象或者空
     *
     * @param clazz 类
     * @param <T>   泛型
     * @return 结果
     */
    @Nullable
    @Override
    public <T> T execute(Class<T> clazz) {
        this.responseType = resolveType(clazz);
        this.resultClass = clazz;
        try {
            return innerExecute();
        } catch (IOException e) {
            log.error("请求时发生异常！", e);
        }
        return null;
    }

    /**
     * 执行并序列化，使用复杂的自动构造的类型
     *
     * @param type jackson的强化类型
     * @param <T>  泛型
     * @return 结果
     */
    @Nullable
    @Override
    public <T> T execute(JavaType type) {
        this.responseType = resolveType(type.getRawClass());
        this.resultType = type;
        try {
            return innerExecute();
        } catch (IOException e) {
            log.error("请求时发生异常！", e);
        }
        return null;
    }

    /**
     * 执行序列化，使用类型引用
     *
     * @param typeReference jackson 类型引用
     * @param <T>           泛型
     * @return 序列化结果
     */
    @Nullable
    @Override
    public <T> T execute(TypeReference<T> typeReference) {
        this.responseType = ResponseType.OBJECT;
        this.typeReference = typeReference;
        try {
            return innerExecute();
        } catch (IOException e) {
            log.error("请求时发生异常！", e);
        }
        return null;
    }

    /**
     * 执行请求，返回响应实体，自行处理
     *
     * @return 响应实体
     * @throws IOException 异常
     */
    @Override
    public <T> T execute() throws IOException {
        return innerExecute();
    }

    /**
     * 内部执行方法，预处理结果
     *
     * @param <T> 泛型
     * @return 结果
     */
    private <T> T innerExecute() throws IOException {
        log.info("【Rest Invoke】{} {}", request.getMethod(), request.getURI());
        try (CloseableHttpResponse response = clientProvider.getClient().execute(request)) {
            StatusLine statusLine = response.getStatusLine();
            HttpEntity entity = response.getEntity();
            if (HttpStatus.valueOf(statusLine.getStatusCode()).isError()) {
                handleError(request.getURI(), statusLine.getStatusCode(), handleEntity(entity));
            } else {
                return handleEntity(entity);
            }
        } catch (UnknownHostException e) {
            handleError(RestConstants.MSG_UNKNOWN_HOST, e);
        } catch (ConnectTimeoutException e) {
            handleError(RestConstants.MSG_TIME_OUT, e);
        } finally {
            request.releaseConnection();
        }
        return null;
    }

    /**
     * 解析目标类型
     *
     * @param clazz 简单类型
     * @return 响应类型
     */
    private ResponseType resolveType(Class<?> clazz) {
        return RestConstants.RESPONSE_TYPE_MAP.getOrDefault(clazz.getName(), ResponseType.OBJECT);
    }

    /**
     * 处理响应体
     *
     * @param entity 响应体
     * @return 结果
     * @throws IOException 异常
     */
    private <T> T handleEntity(HttpEntity entity) throws IOException {
        if (null == entity) {
            return null;
        }
        if (consumer != null) {
            consumer.accept(entity);
            return null;
        }
        return resolveResponse(entity);
    }

    /**
     * 解析结果
     *
     * @param entity 响应体
     * @param <T>    泛型
     * @return 结果
     */
    @SuppressWarnings("unchecked")
    private <T> T resolveResponse(HttpEntity entity) throws IOException {
        switch (responseType) {
            case TEXT:
                return (T) EntityUtils.toString(entity);
            case JSON:
                return (T) JacksonUtil.json2Map(EntityUtils.toString(entity));
            case BINARY:
                try (InputStream in = entity.getContent()) {
                    return (T) StreamUtils.copyToByteArray(in);
                }
            case OBJECT:
                if (null != this.resultClass) {
                    return (T) JacksonUtil.fromJson(EntityUtils.toString(entity), this.resultClass);
                }
                if (null != this.typeReference) {
                    return (T) JacksonUtil.fromJson(EntityUtils.toString(entity), this.typeReference);
                }
                if (null != this.resultType) {
                    return (T) JacksonUtil.fromJson(EntityUtils.toString(entity), this.resultType);
                }
            default:
                return (T) entity;
        }
    }

}
