package group.flyfish.rest.core.client;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JavaType;
import group.flyfish.rest.core.exception.RestClientException;
import group.flyfish.rest.core.factory.HttpClientProvider;
import group.flyfish.rest.enums.ResponseType;
import org.apache.http.HttpEntity;
import org.springframework.lang.Nullable;

import java.io.IOException;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.function.Consumer;

/**
 * Rest请求客户端
 *
 * @author Mr.Wang
 * <p>
 * @apiNote 1. 全builder调用，用户系统内部相互通信
 * 2. 支持异步回调
 * 3. 多样性组合
 * 4. 解耦实现
 * 5. 支持上传文件、FormData，JSON、支持自定义无侵入扩展
 * 6. 新增单例httpClient模式，复用连接让应用更高效
 */
public interface RestClient {

    /**
     * 新增一个构建器
     *
     * @return 结果
     */
    static RestClientBuilder create() {
        return new RestClientBuilder();
    }

    /**
     * 设置客户端提供者
     *
     * @param provider 客户端提供者
     */
    void setClient(HttpClientProvider provider);

    /**
     * 标记线程池执行
     *
     * @return 结果
     */
    RestClient async();

    /**
     * 标记指定线程池执行
     *
     * @param executorService 线程池
     * @return 结果
     */
    RestClient async(ExecutorService executorService);

    /**
     * 设置响应类型
     *
     * @param responseType 响应类型
     * @return 结果
     */
    RestClient responseType(ResponseType responseType);

    /**
     * 异步执行，接收结果
     *
     * @param consumer 结果
     */
    void execute(Consumer<HttpEntity> consumer);

    /**
     * 静默执行，抛弃全部异常
     */
    void executeSilent();

    /**
     * 执行请求，返回Map
     *
     * @return map
     * @throws IOException 异常
     */
    Map<String, Object> executeForMap() throws IOException;

    /**
     * 执行请求，返回字符串
     *
     * @return 字符串
     * @throws IOException 异常
     */
    String executeForString() throws IOException;

    /**
     * 执行并序列化，该方法会安全的返回对象或者空
     *
     * @param clazz 类
     * @param <T>   泛型
     * @return 结果
     */
    @Nullable
    <T> T execute(Class<T> clazz);

    /**
     * 执行并序列化，使用复杂的自动构造的类型
     *
     * @param type jackson的强化类型
     * @param <T>  泛型
     * @return 结果
     */
    @Nullable
    <T> T execute(JavaType type);

    /**
     * 执行序列化，使用类型引用
     *
     * @param typeReference jackson 类型引用
     * @param <T>           泛型
     * @return 序列化结果
     */
    @Nullable
    <T> T execute(TypeReference<T> typeReference);

    /**
     * 执行请求，返回响应实体，自行处理
     *
     * @return 响应实体
     * @throws IOException 异常
     */
    <T> T execute() throws IOException;

    /**
     * 设置请求失败时的回调
     *
     * @param errorConsumer 错误回调
     * @return 结果
     */
    RestClient onError(Consumer<RestClientException> errorConsumer);
}
