package group.flyfish.rest.core.client;

import group.flyfish.rest.entity.Multipart;
import group.flyfish.rest.enums.HttpMethod;
import group.flyfish.rest.utils.DataUtils;
import group.flyfish.rest.utils.JacksonUtil;
import group.flyfish.rest.utils.RequestContext;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpRequestBase;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static group.flyfish.rest.constants.RestConstants.REQUEST_CONFIG;
import static group.flyfish.rest.constants.RestConstants.RESOLVER_MAP;

/**
 * 主要的builder，核心构建
 *
 * @author wangyu
 */
public class RestClientBuilder {

    private String url;

    private HttpMethod method = HttpMethod.GET;

    private Map<String, Object> params;

    private String body;

    private Map<String, String> headers;

    private List<Multipart> multipartList;

    private boolean multipart;

    private boolean credential;

    private String charset;

    private RequestConfig config;

    public String getUrl() {
        return url;
    }

    public RestClientBuilder url(String url) {
        this.url = url;
        return this;
    }

    public HttpMethod getMethod() {
        return method;
    }

    public RestClientBuilder method(HttpMethod method) {
        this.method = method;
        return this;
    }

    public RestClientBuilder get() {
        this.method = HttpMethod.GET;
        return this;
    }

    public RestClientBuilder post() {
        this.method = HttpMethod.POST;
        return this;
    }

    public RestClientBuilder multipart() {
        this.multipart = true;
        return this;
    }

    public boolean isMultipart() {
        return multipart;
    }

    public Map<String, Object> getParams() {
        if (null == params) {
            params = new HashMap<>();
        }
        return params;
    }

    public RestClientBuilder queryParams(Map<String, Object> params) {
        this.params = params;
        return this;
    }

    public RestClientBuilder addParam(String key, Object value) {
        if (null == this.params) {
            this.params = new HashMap<>();
        }
        this.params.put(key, value);
        return this;
    }

    public RestClientBuilder charset(String charset) {
        this.charset = charset;
        return this;
    }

    public RestClientBuilder withCredential() {
        this.credential = true;
        return this;
    }

    public RestClientBuilder config(RequestConfig config) {
        this.config = config;
        return this;
    }

    public Charset getCharset() {
        return DataUtils.isBlank(charset) ? Charset.defaultCharset() : Charset.forName(charset);
    }

    public RestClientBuilder addMultipartBody(String name, String filename, Object data) {
        return addMultipartBody(new Multipart(name, filename, data));
    }

    public RestClientBuilder addMultipartBody(Multipart part) {
        if (null == this.multipartList) {
            this.multipartList = new ArrayList<>();
        }
        this.multipartList.add(part);
        return this;
    }

    public List<Multipart> getMultipartList() {
        if (null == multipartList) {
            multipartList = new ArrayList<>();
        }
        return multipartList;
    }

    public String getBody() {
        return body;
    }

    public RestClientBuilder body(String body) {
        this.body = body;
        return this;
    }

    public RestClientBuilder body(Object body) {
        this.body = JacksonUtil.toJson(body).orElse(null);
        return this;
    }

    public Map<String, String> getHeaders() {
        if (null == headers) {
            headers = new HashMap<>();
        }
        return headers;
    }

    public RestClientBuilder headers(Map<String, String> headers) {
        this.headers = headers;
        return this;
    }

    public RestClientBuilder addHeader(String key, String value) {
        if (null == this.headers) {
            this.headers = new HashMap<>();
        }
        this.headers.put(key, value);
        return this;
    }

    /**
     * 匹配解析器
     *
     * @return 结果
     */
    private HttpRequestBase buildRequest() {
        HttpRequestBase request = RESOLVER_MAP.getOrDefault(this.method, RESOLVER_MAP.get(HttpMethod.GET))
                .resolve(this);
        // 添加token凭证
        if (credential) {
            RequestContext.getCredential().ifPresent(value -> this.addHeader(
                    RequestContext.AUTHORIZATION_KEY, value)
            );
        }
        // 添加头
        getHeaders().forEach(request::addHeader);
        // 设置公共设置
        request.setConfig(null == config ? REQUEST_CONFIG : config);
        // 返回
        return request;
    }

    /**
     * 构建client
     *
     * @return 结果
     */
    public RestClient build() {
        // 创建请求
        HttpRequestBase request = buildRequest();
        return new DefaultRestClient(request);
    }

}
