package group.flyfish.rest.core.client;

import group.flyfish.rest.constants.RestConstants;
import group.flyfish.rest.core.exception.RestClientException;
import lombok.extern.slf4j.Slf4j;

import java.net.URI;
import java.text.MessageFormat;
import java.util.function.Consumer;

/**
 * rest请求错误处理器
 *
 * @author wangyu
 */
@Slf4j
public abstract class RestErrorHandler {

    protected Consumer<RestClientException> errorConsumer;

    /**
     * 错误处理
     *
     * @param e 异常
     */
    private void handleError(RestClientException e) {
        log.error(e.getMessage());
        if (null != errorConsumer) {
            errorConsumer.accept(e);
        } else {
            throw e;
        }
    }

    /**
     * 处理常规文本异常
     *
     * @param message 消息
     */
    protected void handleError(String message) {
        handleError(new RestClientException(message));
    }

    /**
     * 处理具体发生的异常
     *
     * @param message 信息
     * @param cause   造成的异常
     */
    protected void handleError(String message, Exception cause) {
        handleError(new RestClientException(message + cause.getMessage(), cause));
    }

    /**
     * 处理请求异常
     *
     * @param status 状态码
     * @param data   响应数据
     */
    protected void handleError(URI uri, int status, Object data) {
        String message = MessageFormat.format(RestConstants.MSG_REQUEST_ERROR, uri, status);
        handleError(new RestClientException(message, status, data));
    }
}
