package group.flyfish.rest.core.factory;

import group.flyfish.rest.configuration.RestClientProperties;
import group.flyfish.rest.configuration.configure.PropertiesConfigurable;
import group.flyfish.rest.utils.DataUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.ssl.SSLContextBuilder;
import org.springframework.beans.factory.FactoryBean;

import javax.net.ssl.SSLContext;
import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.util.concurrent.locks.ReentrantLock;

/**
 * 生产httpClient
 *
 * @author wangyu
 */
@Slf4j
public final class HttpClientFactoryBean implements FactoryBean<CloseableHttpClient>, PropertiesConfigurable {

    // 使用非公平锁
    private final ReentrantLock lock = new ReentrantLock();
    // 客户端实例，单例
    private volatile CloseableHttpClient client;
    // 配置，配置没进来就不初始化
    private RestClientProperties properties;

    /**
     * 构建单例的httpClient
     *
     * @return 结果
     */
    private CloseableHttpClient getClient() {
        return DataUtils.isTrue(properties.getAlwaysTrust()) ? createSSLClient() : HttpClients.createDefault();
    }

    /**
     * 不信任的证书请求客户端（默认客户端）
     *
     * @return 结果
     */
    public static CloseableHttpClient createSSLClient() {
        //信任所有
        try {
            SSLContext context = SSLContextBuilder.create().loadTrustMaterial(null, (arg0, arg1) -> true).build();
            SSLConnectionSocketFactory factory = new SSLConnectionSocketFactory(context);
            return HttpClients.custom().setSSLSocketFactory(factory).build();
        } catch (NoSuchAlgorithmException | KeyManagementException | KeyStoreException e) {
            e.printStackTrace();
        }
        return null;
    }

    @Override
    public CloseableHttpClient getObject() throws Exception {
        if (client == null) {
            // 非公平锁，二次判定，定位volatile
            lock.lock();
            try {
                if (client == null) {
                    client = getClient();
                }
            } finally {
                lock.unlock();
            }
        }
        return client;
    }

    @Override
    public Class<?> getObjectType() {
        return CloseableHttpClient.class;
    }

    /**
     * 配置属性，完成初始化
     *
     * @param properties 属性
     */
    @Override
    public void configure(RestClientProperties properties) {
        this.properties = properties;
    }
}
