package group.flyfish.rest.core.resolver.support;

import group.flyfish.rest.core.client.RestClientBuilder;
import group.flyfish.rest.utils.DataUtils;
import org.apache.http.HttpEntity;
import org.apache.http.NameValuePair;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.entity.mime.HttpMultipartMode;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.message.BasicNameValuePair;

import java.io.File;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static group.flyfish.rest.constants.RestConstants.MIME_MAP;

/**
 * 抽象的请求体解析
 *
 * @author wangyu
 */
public abstract class AbstractBodyResolver extends AbstractParamResolver {

    /**
     * 构建entity
     * 对于纯urlencoded form，我们将所有参数处理为formdata
     * 对于上传和body请求，我们照旧处理query
     *
     * @param builder 构建器
     * @return 结果
     */
    protected HttpEntity buildEntity(RestClientBuilder builder) {
        return builder.isMultipart() ? buildMultipart(resolveParams(builder)) :
                DataUtils.isNotBlank(builder.getBody()) ?
                        buildJson(resolveParams(builder)) :
                        buildFormData(builder);
    }

    /**
     * 构建上传数据
     *
     * @param clientBuilder builder
     * @return 结果
     */
    private HttpEntity buildMultipart(RestClientBuilder clientBuilder) {
        MultipartEntityBuilder builder = MultipartEntityBuilder.create();
        builder.setMode(HttpMultipartMode.BROWSER_COMPATIBLE);
        builder.setCharset(clientBuilder.getCharset());
        clientBuilder.getMultipartList().forEach(multipart -> {
            Object data = multipart.getData();
            String name = multipart.getName();
            String filename = multipart.getFilename();
            if (data instanceof byte[]) {
                builder.addBinaryBody(name, (byte[]) data, resolveType(filename), filename);
            } else if (data instanceof File) {
                builder.addBinaryBody(name, (File) data, resolveType(filename), filename);
            } else if (data instanceof InputStream) {
                builder.addBinaryBody(name, (InputStream) data, resolveType(filename), filename);
            } else {
                // 对于无法识别的内容，统一处理为string
                builder.addTextBody(name, String.valueOf(data));
            }
        });
        // 处理query参数
        return builder.build();
    }

    /**
     * 构建JSON方式的POST
     *
     * @param clientBuilder builder
     * @return 结果
     */
    private HttpEntity buildJson(RestClientBuilder clientBuilder) {
        clientBuilder.addHeader("Content-Type", "application/json;charset=UTF-8");
        Charset charset = clientBuilder.getCharset();
        StringEntity entity = new StringEntity(clientBuilder.getBody(), charset);
        entity.setContentEncoding(charset.toString());
        entity.setContentType("application/json");
        return entity;
    }

    /**
     * 构建formdata
     *
     * @param clientBuilder builder
     * @return 结果
     */
    private HttpEntity buildFormData(RestClientBuilder clientBuilder) {
        // 设置参数
        Map<String, Object> params = clientBuilder.getParams();
        List<NameValuePair> list = params.keySet()
                .stream()
                .filter(key -> null != params.get(key))
                .map(key -> new BasicNameValuePair(key, String.valueOf(params.get(key))))
                .collect(Collectors.toList());
        if (DataUtils.isNotEmpty(list)) {
            return new UrlEncodedFormEntity(list, clientBuilder.getCharset());
        }
        return null;
    }

    /**
     * 解析内容类型
     *
     * @param filename 文件名
     * @return 结果
     */
    private ContentType resolveType(String filename) {
        return ContentType.create(MIME_MAP.getOrDefault(DataUtils.getExtension(filename),
                ContentType.APPLICATION_OCTET_STREAM.getMimeType()));
    }
}
