package group.flyfish.rest.registry;

import group.flyfish.rest.annotation.EnableRestApiProxy;
import group.flyfish.rest.annotation.RestService;
import group.flyfish.rest.core.factory.HttpClientProvider;
import group.flyfish.rest.registry.proxy.RestProxyInvoker;
import group.flyfish.rest.registry.proxy.support.RestArgumentResolverComposite;
import group.flyfish.rest.utils.DataUtils;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.BeanFactoryAware;
import org.springframework.beans.factory.annotation.AnnotatedBeanDefinition;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.beans.factory.support.*;
import org.springframework.context.annotation.ClassPathScanningCandidateComponentProvider;
import org.springframework.core.type.classreading.MetadataReader;
import org.springframework.core.type.filter.AnnotationTypeFilter;
import org.springframework.lang.NonNull;
import org.springframework.util.Assert;
import org.springframework.util.ClassUtils;
import org.springframework.util.CollectionUtils;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Set;

/**
 * rest接口注册机
 *
 * @author wangyu
 */
@RequiredArgsConstructor
@Slf4j
public class RestApiRegistry implements BeanDefinitionRegistryPostProcessor, BeanFactoryAware {

    @Getter
    private final RestArgumentResolverComposite composite;

    @Getter
    private final HttpClientProvider provider;

    // bean工厂
    private ConfigurableListableBeanFactory beanFactory;

    /**
     * 动态注册bean
     *
     * @param registry 注册机
     * @throws BeansException 异常
     */
    @Override
    public void postProcessBeanDefinitionRegistry(BeanDefinitionRegistry registry) throws BeansException {
        // 包名
        List<String> packageNames = new ArrayList<>();
        // 找基本包，找不到立马报错
        beanFactory.getBeansWithAnnotation(EnableRestApiProxy.class)
                .forEach((key, value) -> {
                    EnableRestApiProxy proxy = value.getClass().getAnnotation(EnableRestApiProxy.class);
                    for (String basePackage : proxy.basePackages()) {
                        if (DataUtils.isNotBlank(basePackage)) {
                            packageNames.add(basePackage);
                        }
                    }
                });
        //  不为空时查找
        if (DataUtils.isNotEmpty(packageNames)) {
            // 初始化反射
            try {
                RestServiceComponentProvider scanner = new RestServiceComponentProvider();
                scanner.resetFilters(false);
                scanner.addIncludeFilter(new AnnotationTypeFilter(RestService.class));
                // 获取扫描器的ClassLoader，保证同源
                ClassLoader cl = scanner.getClass().getClassLoader();
                for (String packageName : packageNames) {
                    Set<BeanDefinition> bfs = scanner.findCandidateComponents(packageName);
                    // 不存在，不要浪费性能
                    if (CollectionUtils.isEmpty(bfs)) return;
                    // 代理并生成子类，并注册到ioc容器
                    bfs.stream()
                            .map(bf -> resolveType(bf, cl))
                            .filter(Objects::nonNull)
                            .forEach(clazz -> registry.registerBeanDefinition(clazz.getName(), generate(clazz)));
                }
            } catch (IllegalStateException e) {
                log.error("初始化Rest映射时出错", e);
            }
            return;
        }
        throw new BeanDefinitionValidationException("【RestApi】EnableRestApiProxy注解必须指定有效的basePackage!");
    }

    /**
     * 生成bean定义
     *
     * @param clazz 目标类型
     * @return bean定义
     */
    private BeanDefinition generate(Class<?> clazz) {
        return BeanDefinitionBuilder.genericBeanDefinition(clazz, () -> RestProxyInvoker.produce(clazz, this))
                .setAutowireMode(AbstractBeanDefinition.AUTOWIRE_BY_TYPE).getRawBeanDefinition();
    }

    @SneakyThrows
    @Override
    public void postProcessBeanFactory(ConfigurableListableBeanFactory beanFactory) throws BeansException {
        // do nothing
    }

    @Override
    public void setBeanFactory(BeanFactory beanFactory) throws BeansException {
        Assert.isTrue(beanFactory instanceof ConfigurableListableBeanFactory, "当前bean factory不被支持！");
        this.beanFactory = (ConfigurableListableBeanFactory) beanFactory;
    }

    private Class<?> resolveType(BeanDefinition bf, ClassLoader cl) {
        if (null != bf.getBeanClassName()) {
            try {
                return ClassUtils.forName(bf.getBeanClassName(), cl);
            } catch (ClassNotFoundException e) {
                return null;
            }
        }
        return null;
    }

    /**
     * 内部的包扫描器，提供特定注解扫描
     *
     * @author wangyu
     */
    private static class RestServiceComponentProvider extends ClassPathScanningCandidateComponentProvider {

        private final AnnotationTypeFilter filter = new AnnotationTypeFilter(RestService.class);

        private RestServiceComponentProvider() {
            super(false);
            resetFilters(false);
            addIncludeFilter(filter);
        }

        @Override
        protected boolean isCandidateComponent(@NonNull MetadataReader metadataReader) throws IOException {
            return filter.match(metadataReader, getMetadataReaderFactory());
        }

        @Override
        protected boolean isCandidateComponent(@NonNull AnnotatedBeanDefinition beanDefinition) {
            return true;
        }
    }
}
