package group.flyfish.rest.registry.proxy.entity;

import group.flyfish.rest.annotation.RestApi;
import group.flyfish.rest.enums.HttpMethod;
import group.flyfish.rest.registry.proxy.RestProxyInvoker;
import group.flyfish.rest.utils.DataUtils;
import lombok.Getter;
import org.springframework.core.annotation.AnnotatedElementUtils;
import org.springframework.core.annotation.AnnotationUtils;

import java.beans.Transient;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.lang.reflect.Type;
import java.util.stream.Stream;

/**
 * 表示单个请求的最小单元
 *
 * @author wangyu
 */
public class RestMethod {

    // 方法参数缓存，避免clone
    @Getter
    private Parameter[] parameters;

    // 解析@Transient注解的结果
    @Getter
    private boolean bare;

    // 解析后的路径
    @Getter
    private String url;

    // http方法
    @Getter
    private HttpMethod method;

    // 是否携带cookie
    @Getter
    private boolean credentials;

    // 多个参数时使用合并的body
    @Getter
    private boolean mergeBody;

    // 带泛型的返回类型
    @Getter
    private Type genericReturnType;

    // 不带泛型的返回类型
    @Getter
    private Class<?> returnType;

    // 是否不可用状态
    @Getter
    private boolean invalid;

    private RestMethod(Method method, RestProxyInvoker invoker) {
        // 执行初始化
        init(method, invoker);
    }

    /**
     * 解析代理方法
     *
     * @param method 方法
     * @return 结果
     */
    public static RestMethod resolve(Method method, RestProxyInvoker invoker) {
        return new RestMethod(method, invoker);
    }

    /**
     * 初始化方法
     */
    private void init(Method method, RestProxyInvoker invoker) {
        RestApi restApi = AnnotatedElementUtils.findMergedAnnotation(method, RestApi.class);
        // 无视proxy，因为啥也没
        if (null == restApi) {
            this.invalid = true;
            return;
        }
        this.url = determineUrl(restApi, invoker);
        this.method = restApi.method();
        this.credentials = restApi.credentials();
        this.mergeBody = restApi.mergedBody();
        this.parameters = method.getParameters();
        this.bare = null != AnnotationUtils.findAnnotation(method, Transient.class);
        this.genericReturnType = method.getGenericReturnType();
        this.returnType = method.getReturnType();
    }

    /**
     * 决定基本url，优先级： 方法注解url > 方法注解baseUrl + uri > 全局配置 + uri
     *
     * @return 结果
     */
    private String determineUrl(RestApi restApi, RestProxyInvoker invoker) {
        String url;
        // 解析url以支持PathVariable
        if (DataUtils.isNotBlank(restApi.url())) {
            return restApi.url();
        } else {
            // 构建基础url，优先级从小到大依次找。同时尝试取字典值
            return Stream.of(restApi.baseUrl(), invoker.getBaseUrl())
                    .filter(DataUtils::isNotBlank)
                    .findFirst()
                    // 判定和赋值
                    .map(found -> found.startsWith("#") ?
                            invoker.getProperties().getDictUrl(found.substring(1)) : found)
                    .map(base -> base + restApi.uri())
                    .orElseThrow(() -> new IllegalArgumentException("【Rest调用】未指定url或baseurl，无法调用远端服务器！"));
        }
    }

}
