package group.flyfish.rest.registry.proxy.support;

import group.flyfish.rest.entity.Multipart;
import group.flyfish.rest.registry.proxy.entity.RestMethod;
import group.flyfish.rest.utils.DataUtils;
import lombok.Builder;
import lombok.Data;

import java.util.HashMap;
import java.util.Map;

/**
 * 参数解析上下文
 *
 * @author wangyu
 */
@Data
@Builder
public class ArgumentResolveContext {

    // 解析的方法
    private RestMethod method;

    // 参数
    private Map<String, Object> param;

    // 路径参数
    private Map<String, Object> pathParams;

    // 请求头
    private Map<String, String> headers;

    // 请求体
    private Object body;

    // 文件列表
    private Map<String, Multipart> files;

    // 文件名列表
    private Map<String, String> filenames;

    // 设置参数
    public void setParam(String key, Object value) {
        if (DataUtils.isEmpty(param)) {
            param = new HashMap<>();
        }
        param.put(key, value);
    }

    // 设置请求体（分拣模式）
    public void setBody(String key, Object value) {
        if (null == body || !(body instanceof Map)) {
            body = new HashMap<>();
        }
        Map<String, Object> map = DataUtils.cast(body);
        map.put(key, value);
    }

    /**
     * 设置头部
     *
     * @param headers 头
     */
    public void setHeaders(Map<String, String> headers) {
        if (DataUtils.isEmpty(this.headers)) {
            this.headers = new HashMap<>();
        }
        this.headers.putAll(headers);
    }

    /**
     * 设置单个头
     *
     * @param name  名称
     * @param value 值
     */
    public void setHeader(String name, String value) {
        if (DataUtils.isEmpty(this.headers)) {
            this.headers = new HashMap<>();
        }
        this.headers.put(name, value);
    }

    /**
     * 设置路径参数
     *
     * @param key   名称
     * @param value 值
     */
    public void setPathParam(String key, Object value) {
        if (DataUtils.isEmpty(pathParams)) {
            pathParams = new HashMap<>();
        }
        pathParams.put(key, value);
    }

    /**
     * 设置文件
     *
     * @param name     文件key
     * @param filename 文件名
     * @param file     文件数据
     */
    public void setMultipart(String name, String filename, Object file) {
        setMultipart(new Multipart(name, filename, file));
    }

    /**
     * 添加文件名
     *
     * @param part     文件部分key
     * @param filename 文件名
     */
    public void addFilename(String part, String filename) {
        if (DataUtils.isEmpty(filenames)) {
            filenames = new HashMap<>();
        }
        filenames.put(part, filename);
    }

    /**
     * 设置文件
     *
     * @param multipart 文件
     */
    public void setMultipart(Multipart multipart) {
        if (DataUtils.isEmpty(files)) {
            files = new HashMap<>();
        }
        if (null != multipart && null != multipart.getData()) {
            files.put(multipart.getName(), multipart);
        }
    }

    public boolean hasPathParams() {
        return DataUtils.isNotEmpty(pathParams);
    }

    public boolean hasBody() {
        return null != body;
    }

    public boolean hasHeaders() {
        return DataUtils.isNotEmpty(headers);
    }

    public boolean hasParams() {
        return DataUtils.isNotEmpty(param);
    }

    public boolean hasMultipart() {
        return DataUtils.isNotEmpty(files);
    }

    public String getFilename(String part, String initial) {
        if (null == filenames || !filenames.containsKey(part)) {
            return initial;
        }
        return filenames.get(part);
    }
}
