package group.flyfish.rest.registry.proxy.support.resolvers;

import group.flyfish.rest.annotation.RestParam;
import group.flyfish.rest.annotation.RestParams;
import group.flyfish.rest.registry.proxy.support.ArgumentResolveContext;
import group.flyfish.rest.registry.proxy.support.RestArgumentResolver;
import group.flyfish.rest.utils.DataUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.util.ClassUtils;

import java.beans.PropertyDescriptor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Parameter;
import java.util.Map;
import java.util.Optional;

/**
 * 参数解析
 *
 * @author wangyu
 */
@Slf4j
public class RestParamArgumentResolver implements RestArgumentResolver {

    /**
     * 是否支持
     *
     * @param parameter 参数
     * @return 结果
     */
    @Override
    public boolean support(Parameter parameter) {
        return true;
    }

    /**
     * 解析
     *
     * @param context   上下文，赋值
     * @param parameter 参数
     * @param value     值
     */
    @Override
    public void resolve(ArgumentResolveContext context, Parameter parameter, Object value) {
        // 当参数包含@RestParams注解，使用BeanDescriptor处理
        if (null != parameter.getAnnotation(RestParams.class) || ClassUtils.isAssignable(Map.class, parameter.getType())) {
            resolveParams(context, parameter, value);
        } else {
            // 取得合法的名称
            String name = Optional.ofNullable(parameter.getAnnotation(RestParam.class))
                    .map(RestParam::value)
                    .filter(DataUtils::isNotBlank)
                    .orElse(parameter.getName());
            // 启用合并请求体，合并入
            if (context.getMethod().isMergeBody()) {
                context.setBody(name, value);
            } else {
                context.setParam(name, parseValue(value));
            }
        }
    }

    /**
     * 解析多个参数
     *
     * @param parameter 参数
     * @param value     值
     */
    private void resolveParams(ArgumentResolveContext context, Parameter parameter, Object value) {
        // 参数注解存在，报出错误
        if (null != parameter.getAnnotation(RestParam.class)) {
            throw new IllegalArgumentException("无法将对象作为一个普通的参数！");
        }
        // 非空才处理
        if (null != value) {
            // 是map，直接解包赋值
            if (ClassUtils.isAssignable(Map.class, parameter.getType())) {
                Map<String, Object> values = DataUtils.cast(value);
                values.forEach((k, v) -> {
                    if (null != v) {
                        context.setParam(k, parseValue(v));
                    }
                });
            } else {
                // 对象，解析后混入
                for (PropertyDescriptor propertyDescriptor : BeanUtils.getPropertyDescriptors(value.getClass())) {
                    if ("class".equalsIgnoreCase(propertyDescriptor.getName())) {
                        continue;
                    }
                    Object v = null;
                    try {
                        v = propertyDescriptor.getReadMethod().invoke(value);
                    } catch (IllegalAccessException | InvocationTargetException e) {
                        log.error("【Rest客户端】尝试解析参数时发生异常!获取bean的属性表失败!{}", e.getMessage(), e);
                    }
                    if (null != v) {
                        context.setParam(propertyDescriptor.getName(), parseValue(v));
                    }
                }
            }
        }
    }

    /**
     * 解析值
     *
     * @param value 值
     * @return 结果
     */
    private String parseValue(Object value) {
        if (value instanceof String) {
            return (String) value;
        }
        if (value instanceof Iterable) {
            Iterable<? extends CharSequence> mapped = DataUtils.cast(value);
            return String.join(",", mapped);
        }
        if (null != value) {
            return String.valueOf(value);
        }
        return "";
    }
}
