package group.flyfish.rest.registry.wrapper;

import group.flyfish.rest.core.exception.RestClientException;
import group.flyfish.rest.mapping.RestResultMapping;
import group.flyfish.rest.utils.TypeResolveUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.client.utils.DateUtils;

import java.lang.reflect.Type;
import java.util.Date;

/**
 * 默认缺省的结果映射
 *
 * @author wangyu
 */
@Slf4j
public class DefaultRestResultMapping implements RestResultMapping {

    /**
     * 获取内部类单例
     *
     * @return 结果
     */
    public static DefaultRestResultMapping getInstance() {
        return SingletonHolder.INSTANCE;
    }

    /**
     * 模糊的结果映射
     *
     * @param body 结果
     * @return 映射后的结果
     */
    @Override
    @SuppressWarnings("unchecked")
    public <T> T map(Object body) throws RestClientException {
        if (body instanceof RestResult) {
            RestResult<?> result = (RestResult<?>) body;
            if (result.isSuccess()) {
                return (T) result.getResult();
            }
            log.error("【RestProxy】请求发生异常！状态码：{}，时间：{}，信息：{}", result.getCode(),
                    DateUtils.formatDate(new Date(result.getTimestamp()), "yyyy-MM-dd HH:mm:ss"), result.getMessage());
            throw new RestClientException(result.getMessage());
        }
        return (T) body;
    }

    /**
     * 解析返回类型
     *
     * @param resultType 返回类型
     * @return 结果
     */
    @Override
    public Type resolve(Type resultType) {
        return TypeResolveUtils.wrap(resultType, RestResult.class);
    }

    /**
     * 静态初始化器，由JVM来保证线程安全
     */
    private interface SingletonHolder {
        DefaultRestResultMapping INSTANCE = new DefaultRestResultMapping();
    }
}
