package group.flyfish.rest.utils;

import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.Collection;
import java.util.Map;

/**
 * 数据工具类，仅限于rest模块使用
 *
 * @author wangyu
 */
public final class DataUtils {

    private static final int INDEX_NOT_FOUND = -1;

    private static final String EXTENSION_SEPARATOR = ".";

    /**
     * The Unix separator character.
     */
    private static final char UNIX_SEPARATOR = '/';

    /**
     * The Windows separator character.
     */
    private static final char WINDOWS_SEPARATOR = '\\';

    /**
     * 判断字符串是否不为空
     *
     * @param target 目标字符串
     * @return 结果
     */
    public static boolean isNotBlank(String target) {
        return !StringUtils.isEmpty(target);
    }

    /**
     * 强制转换和平滑过渡
     *
     * @param source 原对象
     * @param <T>    原类型
     * @param <R>    目标类型
     * @return 结果
     */
    @SuppressWarnings("unchecked")
    public static <T, R> T cast(R source) {
        return (T) source;
    }

    /**
     * 判断集合是否不为空
     *
     * @param collection 集合
     * @return 结果
     */
    public static boolean isNotEmpty(Collection<?> collection) {
        return !CollectionUtils.isEmpty(collection);
    }

    /**
     * 判断map是否不为空
     *
     * @param map 一个字典
     * @return 结果
     */
    public static boolean isNotEmpty(Map<?, ?> map) {
        return !CollectionUtils.isEmpty(map);
    }

    /**
     * 判断map是否为囧
     *
     * @param map 一个字典
     * @return 结果
     */
    public static boolean isEmpty(Map<?, ?> map) {
        return CollectionUtils.isEmpty(map);
    }

    /**
     * 判断集合是否为空
     *
     * @param collection 集合
     * @return 结果
     */
    public static boolean isEmpty(Collection<?> collection) {
        return CollectionUtils.isEmpty(collection);
    }

    /**
     * 是否为空
     *
     * @param target 目标
     * @return 结果
     */
    public static boolean isBlank(String target) {
        return StringUtils.isEmpty(target);
    }

    /**
     * 查找两个字符串之间的内容
     *
     * @param str   字符串
     * @param open  开始
     * @param close 结尾
     * @return 结果
     */
    public static String substringBetween(final String str, final String open, final String close) {
        if (str == null || open == null || close == null) {
            return null;
        }
        final int start = str.indexOf(open);
        if (start != INDEX_NOT_FOUND) {
            final int end = str.indexOf(close, start + open.length());
            if (end != INDEX_NOT_FOUND) {
                return str.substring(start + open.length(), end);
            }
        }
        return null;
    }

    /**
     * 判断是不是true
     *
     * @param value 值
     * @return 结果
     */
    public static boolean isTrue(Boolean value) {
        return Boolean.TRUE.equals(value);
    }

    public static String getExtension(final String filename) {
        if (filename == null) {
            return null;
        }
        final int index = indexOfExtension(filename);
        if (index == INDEX_NOT_FOUND) {
            return "";
        } else {
            return filename.substring(index + 1);
        }
    }

    public static int indexOfExtension(final String filename) {
        if (filename == null) {
            return INDEX_NOT_FOUND;
        }
        final int extensionPos = filename.lastIndexOf(EXTENSION_SEPARATOR);
        final int lastSeparator = indexOfLastSeparator(filename);
        return lastSeparator > extensionPos ? INDEX_NOT_FOUND : extensionPos;
    }

    public static int indexOfLastSeparator(final String filename) {
        if (filename == null) {
            return INDEX_NOT_FOUND;
        }
        final int lastUnixPos = filename.lastIndexOf(UNIX_SEPARATOR);
        final int lastWindowsPos = filename.lastIndexOf(WINDOWS_SEPARATOR);
        return Math.max(lastUnixPos, lastWindowsPos);
    }
}
